package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.BannerAdView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for banner ad creation.
 *
 * @param width Optional width of banner ad view in pixels
 * @param height Optional height of banner ad view in pixels
 * @param attrs Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class BannerAdSettings(
    val width: Int? = null,
    val height: Int? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val attrs: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null,
)

class BannerAdViewManager(private val config: Config) {

    /**
     * Shows a banner ad view with optional parameters.
     *
     * @param context Android context for view creation
     * @param ad Map containing the parsed ad data
     * @param adViewSettings BannerAdSettings with optional configuration
     * @return View containing the banner ad, or null if creation fails
     *
     * Example:
     * ```
     * val bannerView = bannerManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = BannerAdSettings(
     *         width = 320,
     *         height = 250
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: BannerAdSettings
    ): View? {
        return try {
            val bannerAdView = BannerAdView(
                context = context,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                errorCallback = adViewSettings.errorCallback,
                config = config,
                attrs = adViewSettings.attrs
            )
            bannerAdView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "BANNER_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}
