package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.CarouselAdView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for carousel ad creation.
 *
 * @param width Optional width of carousel ad view in pixels
 * @param height Optional height of carousel ad view in pixels
 * @param attrs Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class CarouselAdSettings(
    val width: Int? = null,
    val height: Int? = null,
    val elementWidth: Int? = null,
    val elementHeight: Int? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val attrs: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class CarouselAdViewManager(private val config: Config) {

    /**
     * Shows a carousel ad view with optional parameters.
     *
     * @param context Android context for view creation
     * @param ad Map containing the parsed ad data
     * @param adViewSettings CarouselAdSettings with optional configuration
     * @return View containing the carousel ad, or null if creation fails
     *
     * Example:
     * ```
     * val carouselView = carouselManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = CarouselAdSettings(
     *         width = 320,
     *         height = 250
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: CarouselAdSettings
    ): View? {
        return try {
            val carouselAdView = CarouselAdView(
                context = context,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                elementWidth = adViewSettings.elementWidth,
                elementHeight = adViewSettings.elementHeight,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                attrs = adViewSettings.attrs,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            carouselAdView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "CAROUSEL_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}