package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.MultiAdCarouselView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for multi-ad carousel creation.
 *
 * @param width Optional width of multi-ad carousel view in pixels
 * @param height Optional height of multi-ad carousel view in pixels
 * @param attrs Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class MultiAdCarouselSettings(
    val width: Int? = null,
    val height: Int? = null,
    val elementWidth: Int? = null,
    val elementHeight: Int? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val attrs: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class MultiAdCarouselViewManager(private val config: Config) {

    /**
     * Shows a multi-ad carousel view with optional parameters.
     *
     * @param context Android context for view creation
     * @param ad Map containing the parsed ad data
     * @param adViewSettings MultiAdCarouselSettings with optional configuration
     * @return View containing the multi-ad carousel, or null if creation fails
     *
     * Example:
     * ```
     * val multiAdCarouselView = multiCarouselManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = MultiAdCarouselSettings(
     *         width = 320,
     *         height = 250
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: MultiAdCarouselSettings
    ): View? {
        return try {
            val multiAdCarouselView = MultiAdCarouselView(
                context = context,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                elementWidth = adViewSettings.elementWidth,
                elementHeight = adViewSettings.elementHeight,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                attrs = adViewSettings.attrs,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            multiAdCarouselView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "MULTI_AD_CAROUSEL_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}