package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.NativeAdCarouselView
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Configuration parameters for native ad carousel view creation.
 *
 * @param adType Type of native ad (SMALL, MEDIUM, LARGE)
 * @param width Width of native ad carousel view in pixels
 * @param height Optional height of native ad carousel view in pixels
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customStyle Optional custom styling for the native ad
 * @param customCtaView Optional custom CTA button view
 * @param customBadgeView Optional custom badge view
 * @param attributeSet Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class NativeAdCarouselSettings(
    val adType: NativeAdStyle,
    val width: Int,
    val height: Int? = null,
    val elementWidth: Int? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val customStyle: NativeAdCustomStyle? = null,
    val customCtaView: View? = null,
    val customBadgeView: View? = null,
    val attributeSet: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class NativeAdCarouselViewManager(private val config: Config) {

    /**
     * Shows a native ad carousel view with the specified configuration.
     *
     * @param context Application context required for view creation
     * @param ad Map containing the parsed native ad data (should contain a list of native ads)
     * @param adViewSettings NativeAdCarouselSettings with configuration
     * @return View containing the native ad carousel, or null if creation fails
     *
     * Example:
     * ```
     * val nativeAdCarouselView = nativeAdCarouselManager.showAd(
     *     context = context,
     *     ad = adData, // Map containing list of native ads
     *     adViewSettings = NativeAdCarouselSettings(
     *         adType = NativeAdLayoutType.MEDIUM,
     *         width = 400,
     *         height = 300,
     *         adLabelText = "Sponsored",
     *         adLabelAlignment = Gravity.TOP or Gravity.START
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: NativeAdCarouselSettings
    ): View? {
        return try {
            val nativeAdCarouselView = NativeAdCarouselView(
                context = context,
                adData = ad,
                adType = adViewSettings.adType,
                width = adViewSettings.width,
                height = adViewSettings.height,
                elementWidth = adViewSettings.elementWidth,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                customStyle = adViewSettings.customStyle,
                customCtaView = adViewSettings.customCtaView,
                customBadgeView = adViewSettings.customBadgeView,
                attrs = adViewSettings.attributeSet,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            nativeAdCarouselView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "NATIVE_AD_CAROUSEL_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}