package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.NativeAdHorizontalView
import com.ai.osmos.ads.views.NativeAdVerticalView
import com.ai.osmos.ads.views.NativeAdView
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Unified configuration parameters for all native ad view types.
 * View type is automatically determined based on parameters:
 * - Custom view: If contentView is provided
 * - Vertical layout: If adType is NativeAdStyle.Vertical
 * - Horizontal layout: If adType is NativeAdStyle.Horizontal
 *
 * @param adType Type of native ad size/density (Vertical or Horizontal - required for layout-based ads)
 * @param width Width of native ad view in pixels (required for vertical and horizontal layouts)
 * @param height Optional height of native ad view in pixels
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customStyle Optional custom styling for the native ad
 * @param customCtaView Optional custom CTA button view
 * @param customBadgeView Optional custom badge view
 * @param contentView Optional custom content view (if provided, creates CUSTOM type)
 * @param attributeSet Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class NativeAdSettings(
    val adType: NativeAdStyle? = null, // Required for VERTICAL and HORIZONTAL, ignored if contentView provided
    val width: Int? = null, // Required for VERTICAL and HORIZONTAL, ignored if contentView provided
    val height: Int? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val customStyle: NativeAdCustomStyle? = null,
    val customCtaView: View? = null,
    val customBadgeView: View? = null,
    val contentView: View? = null, // If provided, creates CUSTOM type ad
    val attributeSet: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class NativeAdViewManager(private val config: Config) {

    /**
     * Shows a native ad view with the specified configuration.
     * Automatically determines the view type based on parameters:
     * - If contentView is provided → Custom view with user layout
     * - If adType is NativeAdStyle.Vertical → Vertical layout
     * - If adType is NativeAdStyle.Horizontal → Horizontal layout
     *
     * @param context Application context required for view creation
     * @param ad Map containing the parsed native ad data
     * @param adViewSettings NativeAdSettings with configuration
     * @return View containing the native ad, or null if creation fails
     *
     * Example for Vertical Native Ad:
     * ```
     * val nativeAdView = nativeAdManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = NativeAdSettings(
     *         adType = NativeAdStyle.Vertical,
     *         width = 300,
     *         height = 200,
     *         adLabelText = "Sponsored",
     *         adLabelAlignment = Gravity.TOP or Gravity.START
     *     )
     * )
     * ```
     *
     * Example for Horizontal Native Ad:
     * ```
     * val nativeAdView = nativeAdManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = NativeAdSettings(
     *         adType = NativeAdStyle.Horizontal,
     *         width = 400,
     *         height = 150,
     *         adLabelText = "Sponsored"
     *     )
     * )
     * ```
     *
     * Example for Custom Native Ad:
     * ```
     * val nativeAdView = nativeAdManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = NativeAdSettings(
     *         contentView = customContentView,
     *         errorCallback = errorCallback
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: NativeAdSettings
    ): View? {
        return try {
            // Automatically determine view type and create appropriate view
            when {
                adViewSettings.contentView != null -> createCustomNativeAd(
                    context,
                    ad,
                    adViewSettings
                )

                adViewSettings.adType == NativeAdStyle.Vertical -> createVerticalNativeAd(
                    context,
                    ad,
                    adViewSettings
                )

                adViewSettings.adType == NativeAdStyle.Horizontal -> createHorizontalNativeAd(
                    context,
                    ad,
                    adViewSettings
                )

                else -> {
                    adViewSettings.errorCallback?.onError(
                        "INVALID_CONFIGURATION",
                        "Either contentView must be provided for custom layout, or adType must be specified for standard layouts",
                        null
                    )
                    null
                }
            }
        } catch (e: Exception) {
            val errorCode = when {
                adViewSettings.contentView != null -> "NATIVE_AD_CUSTOM_CREATION_FAILED"
                adViewSettings.adType == NativeAdStyle.Vertical -> "NATIVE_AD_VERTICAL_CREATION_FAILED"
                adViewSettings.adType == NativeAdStyle.Horizontal -> "NATIVE_AD_HORIZONTAL_CREATION_FAILED"
                else -> "NATIVE_AD_CREATION_FAILED"
            }
            adViewSettings.errorCallback?.onError(
                errorCode,
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }

    /**
     * Creates a vertical native ad view.
     */
    private fun createVerticalNativeAd(
        context: Context,
        ad: Map<String, Any>,
        settings: NativeAdSettings
    ): View? {

        val width = settings.width ?: run {
            settings.errorCallback?.onError(
                "MISSING_WIDTH",
                "width is required for VERTICAL view type",
                null
            )
            return null
        }

        val nativeAdVerticalView = NativeAdVerticalView(
            context = context,
            adData = ad,
            width = width,
            height = settings.height,
            adLabelText = settings.adLabelText,
            adLabelAlignment = settings.adLabelAlignment,
            customStyle = settings.customStyle,
            customCtaView = settings.customCtaView,
            customBadgeView = settings.customBadgeView,
            attrs = settings.attributeSet,
            errorCallback = settings.errorCallback,
            config = config
        )
        return nativeAdVerticalView.getView()
    }

    /**
     * Creates a horizontal native ad view.
     */
    private fun createHorizontalNativeAd(
        context: Context,
        ad: Map<String, Any>,
        settings: NativeAdSettings
    ): View? {

        val width = settings.width ?: run {
            settings.errorCallback?.onError(
                "MISSING_WIDTH",
                "width is required for HORIZONTAL view type",
                null
            )
            return null
        }

        val nativeAdHorizontalView = NativeAdHorizontalView(
            context = context,
            adData = ad,
            width = width,
            height = settings.height,
            adLabelText = settings.adLabelText,
            adLabelAlignment = settings.adLabelAlignment,
            customStyle = settings.customStyle,
            customCtaView = settings.customCtaView,
            customBadgeView = settings.customBadgeView,
            attrs = settings.attributeSet,
            errorCallback = settings.errorCallback,
            config = config
        )
        return nativeAdHorizontalView.getView()
    }

    /**
     * Creates a custom native ad view with user-provided content view.
     */
    private fun createCustomNativeAd(
        context: Context,
        ad: Map<String, Any>,
        settings: NativeAdSettings
    ): View? {
        val nativeAdView = settings.contentView?.let {
            NativeAdView(
                context = context,
                adData = ad,
                contentView = it,
                attrs = settings.attributeSet,
                errorCallback = settings.errorCallback,
                config = config
            )
        }
        return nativeAdView?.getView()
    }
}