package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.View
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.views.PDAAdHorizontalView
import com.ai.osmos.ads.views.PDAAdVerticalView
import com.ai.osmos.ads.views.PDAProductStyle
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.ads.views.style.PDAAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Configuration parameters for PDA ad view creation.
 * Supports both vertical and horizontal layouts through the adType parameter.
 *
 * @param adType Type of PDA ad (VERTICAL, HORIZONTAL)
 * @param width Width of PDA ad view in pixels
 * @param height Optional height of PDA ad view in pixels
 * @param pdaAdCustomStyle Optional custom styling for the PDA ad
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customBadgeView Optional custom badge view
 * @param attributeSet Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class PDAAdSettings(
    val adType: NativeAdStyle,
    val width: Int,
    val height: Int? = null,
    val pdaAdCustomStyle: PDAAdCustomStyle? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val customBadgeView: View? = null,
    val attributeSet: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

/**
 * Unified manager class for PDA (Product Display Ad) advertisements.
 * Handles the creation and configuration of both vertical and horizontal PDA ad views.
 *
 * This manager consolidates the functionality of both PDAAdVerticalViewManager and
 * PDAAdHorizontalViewManager into a single, more maintainable class.
 */
class PDAAdViewManager(private val config: Config) {

    /**
     * Shows a PDA ad view with the specified configuration.
     * Creates either a vertical or horizontal PDA view based on the adType parameter.
     *
     * @param context Application context required for view creation
     * @param ad Map containing the parsed PDA ad data
     * @param adViewSettings PDAAdSettings with configuration
     * @return View containing the PDA ad, or null if creation fails
     *
     * Example for Vertical PDA:
     * ```
     * val pdaAdView = pdaAdManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = PDAAdSettings(
     *         adType = NativeAdStyle.VERTICAL,
     *         width = 300,
     *         height = 400,
     *         adLabelText = "Sponsored",
     *         adLabelAlignment = Gravity.TOP or Gravity.START,
     *         pdaAdCustomStyle = PDAAdCustomStyle(
     *             bannerWidth = 250,
     *             bannerHeight = 200,
     *             productScrollDirection = RecyclerView.VERTICAL,
     *             pdaProductStyle = PDAProductStyle.VERTICAL
     *         )
     *     )
     * )
     * ```
     *
     * Example for Horizontal PDA:
     * ```
     * val pdaAdView = pdaAdManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = PDAAdSettings(
     *         adType = NativeAdStyle.HORIZONTAL,
     *         width = 400,
     *         height = 200,
     *         adLabelText = "Sponsored",
     *         adLabelAlignment = Gravity.TOP or Gravity.START,
     *         pdaAdCustomStyle = PDAAdCustomStyle(
     *             bannerWidth = 350,
     *             bannerHeight = 150,
     *             productScrollDirection = RecyclerView.HORIZONTAL,
     *             pdaProductStyle = PDAProductStyle.HORIZONTAL
     *         )
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: PDAAdSettings
    ): View? {
        return try {
            when (adViewSettings.adType) {
                NativeAdStyle.Vertical -> createVerticalView(context, ad, adViewSettings)
                NativeAdStyle.Horizontal -> createHorizontalView(context, ad, adViewSettings)
            }
        } catch (e: Exception) {
            val errorCode = when (adViewSettings.adType) {
                NativeAdStyle.Vertical -> "PDA_AD_VERTICAL_CREATION_FAILED"
                NativeAdStyle.Horizontal -> "PDA_AD_HORIZONTAL_CREATION_FAILED"
            }
            adViewSettings.errorCallback?.onError(
                errorCode,
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }

    /**
     * Creates a vertical PDA ad view with the specified configuration.
     *
     * @param context Application context required for view creation
     * @param ad Map containing the parsed PDA ad data
     * @param adViewSettings PDAAdSettings with configuration
     * @return View containing the vertical PDA ad
     */
    private fun createVerticalView(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: PDAAdSettings
    ): View {
        val pdaAdVerticalView = PDAAdVerticalView(
            context = context,
            adData = ad,
            adType = adViewSettings.adType,
            width = adViewSettings.width,
            height = adViewSettings.height,
            pdaAdCustomStyle = adViewSettings.pdaAdCustomStyle ?: getDefaultPDAAdCustomStyle(
                NativeAdStyle.Vertical
            ),
            adLabelText = adViewSettings.adLabelText,
            adLabelAlignment = adViewSettings.adLabelAlignment,
            customBadgeView = adViewSettings.customBadgeView,
            attrs = adViewSettings.attributeSet,
            errorCallback = adViewSettings.errorCallback,
            config = config
        )
        return pdaAdVerticalView.getView()
    }

    /**
     * Creates a horizontal PDA ad view with the specified configuration.
     *
     * @param context Application context required for view creation
     * @param ad Map containing the parsed PDA ad data
     * @param adViewSettings PDAAdSettings with configuration
     * @return View containing the horizontal PDA ad
     */
    private fun createHorizontalView(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: PDAAdSettings
    ): View {
        val pdaAdHorizontalView = PDAAdHorizontalView(
            context = context,
            adData = ad,
            adType = adViewSettings.adType,
            width = adViewSettings.width,
            height = adViewSettings.height,
            pdaAdCustomStyle = adViewSettings.pdaAdCustomStyle ?: getDefaultPDAAdCustomStyle(
                NativeAdStyle.Horizontal
            ),
            adLabelText = adViewSettings.adLabelText,
            adLabelAlignment = adViewSettings.adLabelAlignment,
            customBadgeView = adViewSettings.customBadgeView,
            attrs = adViewSettings.attributeSet,
            errorCallback = adViewSettings.errorCallback,
            config = config
        )
        return pdaAdHorizontalView.getView()
    }

    /**
     * Provides default PDAAdCustomStyle configuration based on the ad type.
     *
     * @param adType The type of PDA ad (VERTICAL or HORIZONTAL)
     * @return PDAAdCustomStyle with appropriate defaults for the specified type
     */
    private fun getDefaultPDAAdCustomStyle(adType: NativeAdStyle): PDAAdCustomStyle {
        return when (adType) {
            NativeAdStyle.Vertical -> PDAAdCustomStyle(
                bannerWidth = 250,
                bannerHeight = 250,
                productScrollDirection = RecyclerView.VERTICAL,
                pdaProductStyle = PDAProductStyle.Vertical,
                nativeAdWidth = 150,
                nativeAdHeight = 150,
                nativeAdLayoutType = NativeAdLayoutType.Medium,
                nativeStyle = NativeAdCustomStyle(),
                ctaButton = null,
                nativeAdViewLoad = null,
                nativeAdClicked = null
            )

            NativeAdStyle.Horizontal -> PDAAdCustomStyle(
                bannerWidth = 350,
                bannerHeight = 150,
                productScrollDirection = RecyclerView.HORIZONTAL,
                pdaProductStyle = PDAProductStyle.Horizontal,
                nativeAdWidth = 120,
                nativeAdHeight = 160,
                nativeAdLayoutType = NativeAdLayoutType.Small,
                nativeStyle = NativeAdCustomStyle(),
                ctaButton = null,
                nativeAdViewLoad = null,
                nativeAdClicked = null
            )
        }
    }
}