package com.ai.osmos.ads.views.managers

import android.app.Activity
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.Coordinates
import com.ai.osmos.ads.views.PIPAdView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for picture-in-picture ad creation.
 *
 * @param width Optional width of PIP ad view in pixels
 * @param height Optional height of PIP ad view in pixels
 * @param coordinates Optional coordinates for absolute positioning
 * @param attributeSet Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class PIPAdSettings(
    val width: Int? = null,
    val height: Int? = null,
    val coordinates: Coordinates? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val attributeSet: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class PIPAdViewManager(private val config: Config) {

    /**
     * Shows a PIP ad view with optional parameters.
     *
     * @param activity Activity context required for PIP functionality
     * @param ad Map containing the parsed ad data
     * @param adViewSettings PIPAdSettings with optional configuration
     * @return View containing the PIP ad, or null if creation fails
     *
     * Example:
     * ```
     * val pipView = pipManager.showAd(
     *     activity = activity,
     *     ad = adData,
     *     adViewSettings = PIPAdSettings(
     *         width = 320,
     *         height = 250,
     *         coordinates = Coordinates(x = 50, y = 40)
     *     )
     * )
     * ```
     */
    fun showAd(
        activity: Activity,
        ad: Map<String, Any>,
        adViewSettings: PIPAdSettings
    ): View? {
        return try {
            val pipAdView = PIPAdView(
                activity = activity,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                coordinates = adViewSettings.coordinates,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                attrs = adViewSettings.attributeSet,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            pipAdView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "PIP_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}