package com.ai.osmos.ads.views.managers

import android.app.Activity
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.Coordinates
import com.ai.osmos.ads.views.SliderAdView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for slider ad creation.
 *
 * @param width Optional width of slider ad view in pixels
 * @param height Optional height of slider ad view in pixels
 * @param coordinates Optional coordinates for absolute positioning
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param attributeSet Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class SliderAdSettings(
    val width: Int? = null,
    val height: Int? = null,
    val coordinates: Coordinates? = null,
    val adLabelText: String? = null,
    val adLabelAlignment: Int? = null,
    val attributeSet: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class SliderAdViewManager(private val config: Config) {

    /**
     * Shows a slider ad view with optional parameters.
     *
     * @param activity Activity context required for slider functionality
     * @param ad Map containing the parsed ad data
     * @param adViewSettings SliderAdSettings with optional configuration
     * @return View containing the slider ad, or null if creation fails
     *
     * Example:
     * ```
     * val sliderView = sliderManager.showAd(
     *     activity = activity,
     *     ad = adData,
     *     adViewSettings = SliderAdSettings(
     *         width = 320,
     *         height = 250,
     *         coordinates = Coordinates(x = 50, y = 40),
     *         adLabelText = "Sponsored",
     *         adLabelAlignment = Gravity.TOP or Gravity.START
     *     )
     * )
     * ```
     */
    fun showAd(
        activity: Activity,
        ad: Map<String, Any>,
        adViewSettings: SliderAdSettings
    ): View? {
        return try {
            val sliderAdView = SliderAdView(
                activity = activity,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                coordinates = adViewSettings.coordinates,
                adLabelText = adViewSettings.adLabelText,
                adLabelAlignment = adViewSettings.adLabelAlignment,
                attrs = adViewSettings.attributeSet,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            sliderAdView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "SLIDER_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}