package com.ai.osmos.models.ads

/**
 * Enumeration of supported ad types in the OSMOS SDK.
 * Used to identify and handle different ad formats.
 */
enum class AdType {
    IMAGE, VIDEO, CAROUSEL, MULTI_AD, NATIVE_AD, PDA_AD, UNKNOWN;

    companion object {
        // Converts string representation to AdType enum
        fun fromString(type: String?): AdType {
            return when (type?.uppercase()) {
                "IMAGE" -> IMAGE
                "VIDEO" -> VIDEO
                "CAROUSEL" -> CAROUSEL
                "MULTI_AD" -> MULTI_AD
                "NATIVE_AD" -> NATIVE_AD
                "PDA_AD" -> PDA_AD
                else -> UNKNOWN
            }
        }
    }
}

// Product data for ad landing pages
data class LandingProductList(
    val skuId: String,
    val uclid: String?,
    val storeId: String?,
    val sellerId: String?,
    val itemGroupId: String?,
    val name: String,
    val brand: String?,
    val description: String?,
    val customLabel0: String?,
    val customLabel1: String?,
    val customLabel2: String?,
    val customLabel3: String?,
    val customLabel4: String?,
    val mrp: Float,
    val salePrice: Float,

    val imageUrl: String?,
    val videoUrl: String?,
    val productUrl: String,

    val categoryL1: String?,
    val categoryL2: String?,
    val categoryL3: String?,
)

// Core ad element containing media content and metadata
data class AdElement(
    val value: String, // Image URL or video URL
    val height: Int, // Ad height in pixels
    val width: Int, // Ad width in pixels  
    val type: String, // Media type (image/video)
    val destinationUrl: String? = null, // Click destination URL
    val landingProductList: List<LandingProductList>? = emptyList(), // Related products
    val videoProgressSec: Int? = 3, // Video progress tracking interval
    val closeButtonSec: Int? = 2 // When to show close button
)

// Image Ad Model
data class ImageAd(
    val clientId: Long,
    val au: String,
    val rank: Int,
    val clickTrackingUrl: String,
    val impressionTrackingUrl: String,
    val uclid: String,
    override val crt: String,
    val elements: AdElement,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = ""
) : BaseAd() {
    override val elementsType: AdType
        get() = AdType.fromString(elements.type)
}

// Video Ad Model
data class VideoAd(
    val clientId: Long,
    val au: String,
    val rank: Int,
    val clickTrackingUrl: String,
    val impressionTrackingUrl: String,
    val uclid: String,
    override val crt: String,
    val elements: AdElement,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = ""
) : BaseAd() {
    override val elementsType: AdType
        get() = AdType.fromString(elements.type)
}

// Media Element for Carousel Ads
data class MediaElement(
    val value: String,
    val height: Int,
    val width: Int,
    val type: String
)

// Carousel Ad Element
data class CarouselAdElement(
    val media: MediaElement,
    val destinationUrl: String
)

data class CarouselAdElementsWrapper(
    val carouselCards: List<CarouselAdElement>,
)

// Carousel Ad Model
data class CarouselAd(
    val clientId: Long,
    val au: String,
    val rank: Int,
    val clickTrackingUrl: String,
    val impressionTrackingUrl: String,
    val uclid: String,
    override val crt: String,
    val elements: CarouselAdElementsWrapper,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = ""
) : BaseAd() {
    override val elementsType: AdType = AdType.CAROUSEL
}


// MultiAd Model
data class MultiAd(
    val clientId: Long,
    val au: String,
    val rank: Int,
    val clickTrackingUrl: String,
    val impressionTrackingUrl: String,
    val uclid: String,
    override val crt: String,
    val elements: AdElement,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = ""
) : BaseAd() {
    override val elementsType: AdType = AdType.MULTI_AD
}

// Native Ad Model
data class NativeAd(
    val clientId: Long?,
    val skuId: String,
    val name: String,
    val mrp: Float,
    val productUrl: String,
    val storeId: String?,
    val imageUrl: String?,
    val videoUrl: String?,
    val salePrice: Float?,
    val sellerId: String?,
    val itemGroupId: String?,
    val brand: String?,
    val description: String?,
    val customLabel0: String?,  //Rating Star
    val customLabel1: String?,  //Rating count
    val customLabel2: String?,  //Delivery details
    val customLabel3: String?,  //Membership price
    val customLabel4: String?,
    val uclid: String?,
    val categoryL1: String?,
    val categoryL2: String?,
    val categoryL3: String?,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = "",
    override val crt: String = ""
) : BaseAd() {
    override val elementsType: AdType = AdType.NATIVE_AD
}

// PDA Ad Model
data class ProductItem(
    val skuId: String,
    val name: String,
    val mrp: Float,
    val storeId: String,
    val productUrl: String,
    val imageUrl: String?,
    val videoUrl: String?,
    val salePrice: Float?,
    val sellerId: String?,
    val itemGroupId: String?,
    val brand: String?,
    val description: String?,
    val customLabel0: String?,  //Rating Star
    val customLabel1: String?,  //Rating count
    val customLabel2: String?,  //Delivery details
    val customLabel3: String?,  //Membership price
    val customLabel4: String?,
    val categoryL1: String?,
    val categoryL2: String?,
    val categoryL3: String?,
    val sellerClientId: Long,
    val uclid: String,
    val rank: Int,
    val cliUbid: String?,
)

data class PDAAdElement(
    val type: String?,
    val value: String?,
    val carouselCards: List<CarouselAdElement>?,
    val width: Int?,
    val height: Int?,
    val productList: List<ProductItem>
) {
    fun getAdType(): AdType = AdType.fromString(type)
}

data class PDAAd(
    val clientId: Long,
    val au: String,
    val rank: Int,
    val clickTrackingUrl: String,
    val impressionTrackingUrl: String,
    val uclid: String,
    val elements: PDAAdElement,
    override val crt: String,
    override val adMetadata: Map<String, Any>? = null,
    override val cliUbid: String? = ""
) : BaseAd() {
    override val elementsType: AdType = AdType.PDA_AD
}

// AdData Wrapper to Handle Multiple Ad Types
data class AdData(
    val ads: Map<String, List<BaseAd>>,
    val imageAdList: List<ImageAd> = emptyList(),
    val videoAdList: List<VideoAd> = emptyList(),
    val carouselAdList: List<CarouselAd> = emptyList(),
    val PDAAdList: List<PDAAd> = emptyList()
)


