package com.ai.osmos.models.ads

/**
 * Immutable Product Listing Ads (PLA) filtering parameters with clean, Kotlin-idiomatic API
 *
 * This class is thread-safe and immutable, preventing the issues from the old mutable builder pattern.
 */
class ProductAdsFilter private constructor(
    private val parameters: Map<String, Any>
) {

    /**
     * Get the filter parameters as an immutable map
     */
    fun toMap(): Map<String, Any> = parameters

    companion object {

        // MARK: - Category Factory Methods

        /**
         * Create filter with category ID
         */
        fun categoryId(categoryId: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("category_id" to categoryId))
        }

        /**
         * Create filter with categories
         */
        fun categories(categories: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("categories" to categories.toList()))
        }

        /**
         * Create filter with merchant category IDs
         */
        fun mcategoryIds(mcategoryIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("mcategory_ids" to mcategoryIds.toList()))
        }

        /**
         * Create filter with SKU IDs
         */
        fun skuIds(skuIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("sku_ids" to skuIds.toList()))
        }

        // MARK: - Price Factory Methods

        /**
         * Create filter with price range
         */
        fun priceRange(minPrice: String, maxPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("min_price" to minPrice, "max_price" to maxPrice))
        }

        /**
         * Create filter with minimum price
         */
        fun minPrice(minPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("min_price" to minPrice))
        }

        /**
         * Create filter with maximum price
         */
        fun maxPrice(maxPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("max_price" to maxPrice))
        }

        // MARK: - Attribute Factory Methods

        /**
         * Create filter with colors
         */
        fun colors(colors: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("colors" to colors.toList()))
        }

        /**
         * Create filter with sizes
         */
        fun sizes(sizes: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("sizes" to sizes.toList()))
        }

        // MARK: - Search Factory Methods

        /**
         * Create filter with keywords
         */
        fun keywords(keywords: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("keywords" to keywords.toList()))
        }

        /**
         * Create filter with single keyword
         */
        fun keyword(keyword: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("keywords" to listOf(keyword)))
        }

        // MARK: - Brand Factory Methods

        /**
         * Create filter with brands
         */
        fun brands(brands: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("brands" to brands.toList()))
        }

        /**
         * Create filter with brand IDs
         */
        fun brandIds(brandIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("brand_ids" to brandIds.toList()))
        }

        // MARK: - Location Factory Methods

        /**
         * Create filter with store IDs
         */
        fun storeIds(storeIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("store_id" to storeIds.toList()))
        }

        /**
         * Create filter with networks
         */
        fun networks(networks: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("network" to networks.toList()))
        }

        /**
         * Create filter with UDPs
         */
        fun udps(udps: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("udp" to udps.toList()))
        }

        /**
         * Create filter with device
         */
        fun device(device: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("device" to device))
        }

        // MARK: - Custom Label Factory Methods

        /**
         * Create filter with custom labels
         */
        fun customLabels(
            label1: List<String>? = null,
            label2: List<String>? = null,
            label3: List<String>? = null,
            label4: List<String>? = null,
            label5: List<String>? = null,
            label6: List<String>? = null,
            label7: List<String>? = null,
            label8: List<String>? = null,
            rtlLabel0: List<String>? = null
        ): ProductAdsFilter {
            val params = mutableMapOf<String, Any>()
            label1?.let { params["custom_label_1"] = it.toList() }
            label2?.let { params["custom_label_2"] = it.toList() }
            label3?.let { params["custom_label_3"] = it.toList() }
            label4?.let { params["custom_label_4"] = it.toList() }
            label5?.let { params["custom_label_5"] = it.toList() }
            label6?.let { params["custom_label_6"] = it.toList() }
            label7?.let { params["custom_label_7"] = it.toList() }
            label8?.let { params["custom_label_8"] = it.toList() }
            rtlLabel0?.let { params["rtl_custom_label_0"] = it.toList() }
            return ProductAdsFilter(params.toMap())
        }

        // MARK: - Combined Factory Methods

        /**
         * Create comprehensive filter with multiple criteria
         */
        fun create(
            categoryId: String? = null,
            categories: List<String>? = null,
            mcategoryIds: List<String>? = null,
            skuIds: List<String>? = null,
            brands: List<String>? = null,
            minPrice: String? = null,
            maxPrice: String? = null,
            colors: List<String>? = null,
            sizes: List<String>? = null,
            keywords: List<String>? = null,
            storeIds: List<String>? = null,
            networks: List<String>? = null,
            device: String? = null
        ): ProductAdsFilter {
            val params = mutableMapOf<String, Any>()
            categoryId?.let { params["category_id"] = it }
            categories?.let { params["categories"] = it.toList() }
            mcategoryIds?.let { params["mcategory_ids"] = it.toList() }
            skuIds?.let { params["sku_ids"] = it.toList() }
            brands?.let { params["brands"] = it.toList() }
            minPrice?.let { params["min_price"] = it }
            maxPrice?.let { params["max_price"] = it }
            colors?.let { params["colors"] = it.toList() }
            sizes?.let { params["sizes"] = it.toList() }
            keywords?.let { params["keywords"] = it.toList() }
            storeIds?.let { params["store_id"] = it.toList() }
            networks?.let { params["network"] = it.toList() }
            device?.let { params["device"] = it }
            return ProductAdsFilter(params.toMap())
        }
    }

    internal fun mergeFilters(other: ProductAdsFilter): ProductAdsFilter {
        val mergedParams = this.parameters.toMutableMap()
        for ((key, value) in other.toMap()) {
            mergedParams[key] = value // Overwrites if key exists
        }
        return ProductAdsFilter(mergedParams)
    }
}