package com.ai.osmos.network

import com.ai.osmos.utils.common.Constants

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: QueryParameterBuilder
 *
 * Utility class to build query parameters for API requests.
 * Consolidates common query parameter building logic to reduce code duplication.
 */
object QueryParameterBuilder {

    /**
     * Builds query parameters for display ad requests.
     *
     * @param clientId Client identifier for authentication
     * @param isDisplayAuAds Flag to determine if ads are fetched using Ad Units
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param adUnits List of ad unit identifiers (optional)
     * @param targetingParams Optional targeting parameters for ad filtering
     * @param functionName Optional function name parameter
     * @return Mutable map containing query parameters
     */
    fun buildDisplayAdParams(
        clientId: String,
        isDisplayAuAds: Boolean,
        cliUbid: String,
        pageType: String,
        productCount: Int,
        adUnits: List<String>? = null,
        targetingParams: Map<String, Any>? = emptyMap(),
        functionName: String? = null
    ): MutableMap<String, Any> {

        val queryParams: MutableMap<String, Any> = mutableMapOf(
            "pt" to pageType,
            "pcnt_au" to productCount,
            "cli_ubid" to cliUbid,
            "client_id" to clientId,
            "os_name" to Constants.OS_NAME,
            "os_version" to Constants.SDK_VERSION
        )

        functionName?.let { queryParams["f_name"] = it }

        if (!adUnits.isNullOrEmpty() && isDisplayAuAds) {
            queryParams.addQueryParam("au", adUnits)
        }

        targetingParams?.forEach { (key, value) ->
            queryParams[key] = value
        }
        return queryParams
    }

    /**
     * Builds query parameters for PLA ad requests.
     *
     * @param clientId Client identifier for authentication
     * @param cliUbid Unique identifier for the client session
     * @param pageType The type of the page requesting the ads
     * @param productCount The number of products to display ads for
     * @param pageName Optional name of the page
     * @param filters Optional map containing filter criteria
     * @param functionName Optional function name parameter
     * @return A mutable map of query parameters for the API request
     */
    fun buildPlaAdParams(
        clientId: String,
        cliUbid: String,
        pageType: String,
        productCount: Int,
        pageName: String? = null,
        filters: Map<String, Any>? = emptyMap(),
        functionName: String? = null
    ): MutableMap<String, Any> {

        return mutableMapOf<String, Any>().apply {
            put("cli_ubid", cliUbid)
            put("page_type", pageType)
            put("pcnt", productCount.toString())
            put("client_id", clientId)
            put("os_name", Constants.OS_NAME)
            put("os_version", Constants.SDK_VERSION)

            if (!functionName.isNullOrEmpty()) put("f_name", functionName)
            if (!pageName.isNullOrEmpty()) put("page_name", pageName)

            filters?.forEach { (key, value) -> addQueryParam(key, value) }
        }
    }


    /**
     * Helper function to add a query parameter to the map.
     * Handles list values by appending "[]" to the key.
     *
     * @receiver The mutable map of query parameters
     * @param key The key for the query parameter
     * @param value The value associated with the key
     */
    private fun MutableMap<String, Any>.addQueryParam(key: String, value: Any) {
        when (value) {
            is List<*> -> put("$key[]", value)
            else -> put(key, value)
        }
    }

    /**
     * Builds TPA ad request data map.
     *
     * @param clientId Client identifier for authentication
     * @param cliUbid Unique identifier for the client session
     * @param productCount Number of products to display ads for
     * @param skuIds List of SKU identifiers
     * @param pageName Optional name of the page
     * @param filters Optional map containing filter criteria
     * @return MutableMap containing the data for TPA request
     */
    fun buildTpaAdData(
        clientId: String,
        cliUbid: String,
        productCount: Int,
        skuIds: List<Any>,
        pageName: String? = null,
        filters: Map<String, Any>? = emptyMap(),
    ): MutableMap<String, Any> {
        return mutableMapOf<String, Any>(
            "cli_ubid" to cliUbid,
            "pcnt" to productCount,
            "page_type" to "TPA",
            "client_id" to clientId,
            "sku_ids" to skuIds
        ).apply {
            pageName?.let { put("page_name", it) }
            filters?.let { putAll(it) }
        }
    }

    /**
     * Merges extra parameters into base parameters.
     * Public utility method that external apps might use.
     *
     * @param baseParams The base parameters for the event
     * @param newParams Additional parameters to be added
     * @return A combined map of parameters
     */
    fun mergeParams(
        baseParams: Map<String, Any>?,
        newParams: Map<String, Any>
    ): Map<String, Any> {
        return (baseParams ?: emptyMap()).toMutableMap().apply { putAll(newParams) }
    }
}
