package com.ai.osmos.tracking.data

import androidx.work.Data

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: ImpressionData
 */

/**
 * Data class for storing impression tracking information that can be
 * manually serialized for WorkManager without external dependencies.
 *
 * This class handles the conversion between Kotlin data types and WorkManager's
 * Data format, ensuring reliable persistence of impression tracking data.
 */
data class ImpressionData(
    val uclid: String,
    val cliUbid: String,
    val position: Int = 1,
    val timestamp: Long = System.currentTimeMillis(),
    val attemptCount: Int = 0,
    val maxRetries: Int = 3
) {

    companion object {
        // Data keys for WorkManager serialization
        private const val KEY_UCLID = "uclid"
        private const val KEY_CLI_UBID = "cli_ubid"
        private const val KEY_POSITION = "position"
        private const val KEY_TIMESTAMP = "timestamp"
        private const val KEY_ATTEMPT_COUNT = "attempt_count"
        private const val KEY_MAX_RETRIES = "max_retries"

        const val WORK_TAG_IMPRESSION = "impression_tracking"

        /**
         * Creates ImpressionData from WorkManager Data input.
         * Returns null if required fields are missing or data is malformed.
         */
        fun fromWorkData(data: Data): ImpressionData? {
            val uclid = data.getString(KEY_UCLID) ?: return null
            val cliUbid = data.getString(KEY_CLI_UBID) ?: return null

            return ImpressionData(
                uclid = uclid,
                cliUbid = cliUbid,
                position = data.getInt(KEY_POSITION, 1),
                timestamp = data.getLong(KEY_TIMESTAMP, System.currentTimeMillis()),
                attemptCount = data.getInt(KEY_ATTEMPT_COUNT, 0),
                maxRetries = data.getInt(KEY_MAX_RETRIES, 3)
            )
        }

    }

    /**
     * Converts this ImpressionData to WorkManager Data format.
     * Uses WorkManager's built-in Data.Builder for type-safe serialization.
     */
    fun toWorkData(): Data {

        return Data.Builder()
            .putString(KEY_UCLID, uclid)
            .putString(KEY_CLI_UBID, cliUbid)
            .putInt(KEY_POSITION, position)
            .putLong(KEY_TIMESTAMP, timestamp)
            .putInt(KEY_ATTEMPT_COUNT, attemptCount)
            .putInt(KEY_MAX_RETRIES, maxRetries)
            .build()
    }


    /**
     * Creates a new ImpressionData with incremented attempt count.
     * Used for retry logic in WorkManager.
     */
    fun incrementAttempt(): ImpressionData = copy(attemptCount = attemptCount + 1)

    /**
     * Checks if this impression should be retried based on attempt count.
     */
    fun shouldRetry(): Boolean = attemptCount < maxRetries

}