package com.ai.osmos.tracking.data

import androidx.work.Data
import com.ai.osmos.models.events.VideoActionType

/**
 * Data class for storing video action tracking information that can be
 * manually serialized for WorkManager without external dependencies.
 *
 * This class handles the conversion between Kotlin data types and WorkManager's
 * Data format, ensuring reliable persistence of video action tracking data.
 */
data class VideoActionData(
    val uclid: String,
    val cliUbid: String,
    val actionType: VideoActionType,
    val videoViewSec: Float,
    val position: Int = 1,
    val timestamp: Long = System.currentTimeMillis(),
    val attemptCount: Int = 0,
    val maxRetries: Int = 3
) {

    companion object {
        // Data keys for WorkManager serialization
        private const val KEY_UCLID = "uclid"
        private const val KEY_CLI_UBID = "cli_ubid"
        private const val KEY_ACTION_TYPE = "action_type"
        private const val KEY_VIDEO_VIEW_SEC = "video_view_sec"
        private const val KEY_POSITION = "position"
        private const val KEY_TIMESTAMP = "timestamp"
        private const val KEY_ATTEMPT_COUNT = "attempt_count"
        private const val KEY_MAX_RETRIES = "max_retries"

        const val WORK_TAG_VIDEO_ACTION = "video_action_tracking"

        /**
         * Creates VideoActionData from WorkManager Data input.
         * Returns null if required fields are missing or data is malformed.
         */
        fun fromWorkData(data: Data): VideoActionData? {
            val uclid = data.getString(KEY_UCLID) ?: return null
            val cliUbid = data.getString(KEY_CLI_UBID) ?: return null
            val actionTypeString = data.getString(KEY_ACTION_TYPE) ?: return null
            val actionType = VideoActionType.fromString(actionTypeString) ?: return null

            return VideoActionData(
                uclid = uclid,
                cliUbid = cliUbid,
                actionType = actionType,
                videoViewSec = data.getFloat(KEY_VIDEO_VIEW_SEC, 0f),
                position = data.getInt(KEY_POSITION, 1),
                timestamp = data.getLong(KEY_TIMESTAMP, System.currentTimeMillis()),
                attemptCount = data.getInt(KEY_ATTEMPT_COUNT, 0),
                maxRetries = data.getInt(KEY_MAX_RETRIES, 3)
            )
        }

    }

    /**
     * Converts this VideoActionData to WorkManager Data format.
     */
    fun toWorkData(): Data {

        return Data.Builder()
            .putString(KEY_UCLID, uclid)
            .putString(KEY_CLI_UBID, cliUbid)
            .putString(KEY_ACTION_TYPE, actionType.actionType)
            .putFloat(KEY_VIDEO_VIEW_SEC, videoViewSec)
            .putInt(KEY_POSITION, position)
            .putLong(KEY_TIMESTAMP, timestamp)
            .putInt(KEY_ATTEMPT_COUNT, attemptCount)
            .putInt(KEY_MAX_RETRIES, maxRetries)
            .build()
    }

    /**
     * Creates a new VideoActionData with incremented attempt count.
     */
    fun incrementAttempt(): VideoActionData = copy(attemptCount = attemptCount + 1)

    /**
     * Checks if this video action should be retried based on attempt count.
     */
    fun shouldRetry(): Boolean = attemptCount < maxRetries
}