package com.ai.osmos.tracking.tracker

import com.ai.osmos.core.Config
import com.ai.osmos.models.events.VideoActionType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdTracker
 */

/**
 * Immediate ad tracking implementation using CoroutineScope.
 * Now extends BaseAdTracker to eliminate code duplication.
 *
 * @param coroutineScope The coroutine scope used to launch background tracking jobs.
 */
class AdTracker(
    private val coroutineScope: CoroutineScope,
    config: Config
) : BaseAdTracker(config) {

    /**
     * Tracks when an ad impression occurs with immediate execution.
     */
    override fun trackImpression(uclid: String, cliUbid: String, position: Int) {
        coroutineScope.launch(Dispatchers.IO) {
            executeImpression(uclid, cliUbid, position)
        }
    }

    /**
     * Tracks when the user clicks on an ad with immediate execution.
     */
    override fun trackAdClick(uclid: String, cliUbid: String) {
        coroutineScope.launch(Dispatchers.IO) {
            executeAdClick(uclid, cliUbid)
        }
    }

    /**
     * Tracks a specific video action with immediate execution.
     */
    override fun videoActionClick(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        coroutineScope.launch(Dispatchers.IO) {
            executeVideoAction(uclid, cliUbid, actionType, videoViewSec)
        }
    }

    /**
     * Tracks periodic progress updates during video playback with immediate execution.
     */
    override fun videoProgressEvent(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        coroutineScope.launch(Dispatchers.IO) {
            executeVideoProgress(uclid, cliUbid, videoViewSec, videoDurationSec)
        }
    }

}
