package com.ai.osmos.tracking.tracker.workers

import android.content.Context
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.ai.osmos.tracking.data.AdClickData
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.tracking.tracker.BaseAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.logging.DebugLogger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.math.min
import kotlin.math.pow

/**
 * WorkManager worker for persistent ad click tracking.
 *
 * This worker handles ad click tracking requests that need to survive app restarts
 * and network failures. It implements exponential backoff retry logic and ensures
 * ad clicks are properly tracked for revenue analytics.
 */
class AdClickTrackingWorker(
    context: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(context, workerParams) {

    companion object {
        private const val TAG = "AdClickTrackingWorker"
    }


    override suspend fun doWork(): Result = withContext(Dispatchers.IO) {

        val config = BaseAdTracker.getConfigFromWorkData(inputData)
        try {
            // Deserialize ad click data from input
            val adClickData = AdClickData.fromWorkData(inputData)
                ?: return@withContext Result.failure(
                    workDataOf("error" to "Invalid ad click data")
                )

            if (config.debug == true) {
                DebugLogger.log("Processing ad click: ${adClickData.uclid}, attempt: ${adClickData.attemptCount}")
            }

            // Create RegisterEvent instance for API calls with instance-specific config
            val registerEvent = RegisterEvent(config)

            try {
                // Attempt to track the ad click
                val response = registerEvent.registerAdClickEvent(
                    cliUbid = adClickData.cliUbid,
                    uclid = adClickData.uclid,
                )

                if (response != null) {
                    DebugLogger.log("Ad click tracked successfully: ${adClickData.uclid}")
                    Result.success(
                        workDataOf(
                            "uclid" to adClickData.uclid,
                            "result" to "success",
                            "attempt_count" to adClickData.attemptCount
                        )
                    )
                } else {
                    handleRetry(adClickData)
                }

            } catch (e: Exception) {
                DebugLogger.log(TAG, "Ad click tracking failed: ${e.message}")

                // Handle retry with proper attempt count management
                handleRetry(adClickData)
            }

        } catch (e: Exception) {
            DebugLogger.log(TAG, "Worker execution failed: ${e.message}")

            Result.failure(
                workDataOf(
                    "error" to "Worker execution failed: ${e.message}"
                )
            )
        }
    }

    /**
     * Handles retry logic with exponential backoff.
     *
     * @param adClickData The ad click data to retry
     * @return Result indicating whether to retry or give up
     */
    private fun handleRetry(adClickData: AdClickData): Result {
        val updatedData = adClickData.incrementAttempt()

        return if (updatedData.shouldRetry()) {

            DebugLogger.log("Scheduling retry ${updatedData.attemptCount}/${updatedData.maxRetries} for ad click: ${adClickData.uclid}")

            // Schedule retry with exponential backoff
            val retryDelay = calculateBackoffDelay(updatedData.attemptCount)
            scheduleRetryWork(updatedData.toWorkData(), retryDelay)

            // Return failure to indicate this attempt failed but retry is scheduled
            Result.failure()
        } else {
            DebugLogger.log("Max retries exceeded for ad click: ${adClickData.uclid}. Giving up.")

            Result.failure(
                workDataOf(
                    "error" to "Max retries exceeded for ad click tracking",
                    "uclid" to adClickData.uclid,
                    "attempt_count" to adClickData.attemptCount
                )
            )
        }
    }

    /**
     * Calculates exponential backoff delay for retries.
     *
     * Uses 2^attempt formula with a maximum cap to prevent excessive delays.
     * This helps distribute retry load and respects server capacity.
     *
     * @param attemptCount Current attempt number (1-based)
     * @return Delay in seconds before next retry
     */
    private fun calculateBackoffDelay(attemptCount: Int): Long {
        // Exponential backoff: 2^attempt seconds, capped at 5 minutes (300s)
        val delaySeconds = min(
            2.0.pow(attemptCount.toDouble()).toLong(),
            300L
        )

        DebugLogger.log(
            Constants.LOG_TAG,
            "Calculated backoff delay: ${delaySeconds}s for attempt $attemptCount"
        )

        return delaySeconds
    }

    /**
     * Schedules a retry work request with appropriate constraints and backoff.
     *
     * @param data Serialized ad click data for the retry
     * @param delaySeconds Delay before executing the retry
     */
    private fun scheduleRetryWork(data: androidx.work.Data, delaySeconds: Long) {
        try {
            val retryRequest = androidx.work.OneTimeWorkRequestBuilder<AdClickTrackingWorker>()
                .setInputData(data)
                .setInitialDelay(delaySeconds, java.util.concurrent.TimeUnit.SECONDS)
                .setConstraints(
                    androidx.work.Constraints.Builder()
                        .setRequiredNetworkType(androidx.work.NetworkType.CONNECTED)
                        .build()
                )
                .addTag(AdClickData.WORK_TAG_AD_CLICK)
                .build()

            androidx.work.WorkManager.getInstance(applicationContext).enqueue(retryRequest)

            DebugLogger.log("Retry work scheduled with ${delaySeconds}s delay")

        } catch (e: Exception) {
            DebugLogger.log("Failed to schedule retry work: ${e.message}")
        }
    }
}
