package com.ai.osmos.tracking.tracker.workers

import android.content.Context
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.WorkerParameters
import com.ai.osmos.tracking.data.ImpressionData
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.tracking.tracker.BaseAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.logging.DebugLogger
import java.util.concurrent.TimeUnit
import kotlin.math.min
import kotlin.math.pow

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: ImpressionTrackingWorker
 */

/**
 * WorkManager Worker for handling persistent impression tracking.
 *
 * This worker ensures that ad impressions are reliably delivered even during:
 * - Network connectivity issues
 * - App backgrounding or termination
 * - Device restarts
 *
 * The worker implements exponential backoff retry strategy and respects
 * Android's battery optimization features.
 */
class ImpressionTrackingWorker(
    context: Context,
    params: WorkerParameters
) : CoroutineWorker(context, params) {


    override suspend fun doWork(): Result {
        return try {
            val impressionData = ImpressionData.fromWorkData(inputData)
                ?: return Result.failure()

            DebugLogger.log("Processing impression: ${impressionData.uclid} (attempt ${impressionData.attemptCount + 1}/${impressionData.maxRetries})")


            // Attempt to register the impression using instance-specific config
            val config = BaseAdTracker.getConfigFromWorkData(inputData)
            val registerEvent = RegisterEvent(config)
            val response = registerEvent.registerAdImpressionEvent(
                cliUbid = impressionData.cliUbid,
                uclid = impressionData.uclid,
                position = impressionData.position,
            )

            if (response != null) {
                DebugLogger.log("Impression tracked successfully: ${impressionData.uclid}")
                Result.success()
            } else {
                handleRetry(impressionData)
            }
        } catch (e: Exception) {
            val impressionData = ImpressionData.fromWorkData(inputData)

            if (impressionData != null) {
                handleRetry(impressionData)
            } else {
                Result.failure()
            }
        }
    }

    /**
     * Handles retry logic with exponential backoff.
     *
     * @param impressionData The impression data to retry
     * @return Result indicating whether to retry or give up
     */
    private fun handleRetry(impressionData: ImpressionData): Result {
        val updatedData = impressionData.incrementAttempt()

        return if (updatedData.shouldRetry()) {
            DebugLogger.log("Scheduling retry ${updatedData.attemptCount}/${updatedData.maxRetries} for impression: ${impressionData.uclid}")

            // Schedule retry with exponential backoff
            val retryDelay = calculateBackoffDelay(updatedData.attemptCount)
            scheduleRetryWork(updatedData.toWorkData(), retryDelay)

            // Return failure to indicate this attempt failed but retry is scheduled
            Result.failure()
        } else {
            DebugLogger.log("Max retries exceeded for impression: ${impressionData.uclid}. Giving up.")

            Result.failure()
        }
    }

    /**
     * Calculates exponential backoff delay for retries.
     *
     * Uses 2^attempt formula with a maximum cap to prevent excessive delays.
     * This helps distribute retry load and respects server capacity.
     *
     * @param attemptCount Current attempt number (1-based)
     * @return Delay in seconds before next retry
     */
    private fun calculateBackoffDelay(attemptCount: Int): Long {
        // Exponential backoff: 2^attempt seconds, capped at 5 minutes (300s)
        val delaySeconds = min(
            2.0.pow(attemptCount.toDouble()).toLong(),
            300L
        )

        DebugLogger.log(
            Constants.LOG_TAG,
            "Calculated backoff delay: ${delaySeconds}s for attempt $attemptCount"
        )

        return delaySeconds
    }

    /**
     * Schedules a retry work request with appropriate constraints and backoff.
     *
     * @param data Serialized impression data for the retry
     * @param delaySeconds Delay before executing the retry
     */
    private fun scheduleRetryWork(data: Data, delaySeconds: Long) {
        try {
            val retryRequest = OneTimeWorkRequestBuilder<ImpressionTrackingWorker>()
                .setInputData(data)
                .setInitialDelay(delaySeconds, TimeUnit.SECONDS)
                .setConstraints(
                    Constraints.Builder()
                        .setRequiredNetworkType(NetworkType.CONNECTED)
                        .build()
                )
                .addTag(ImpressionData.WORK_TAG_IMPRESSION)
                .build()

            WorkManager.getInstance(applicationContext).enqueue(retryRequest)

            DebugLogger.log("Retry work scheduled with ${delaySeconds}s delay")

        } catch (e: Exception) {
            DebugLogger.log("Failed to schedule retry work: ${e.message}")
        }
    }
}