package com.ai.osmos.utils.common

import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.AdType
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.CarouselAd
import com.ai.osmos.models.ads.CarouselAdElement
import com.ai.osmos.models.ads.CarouselAdElementsWrapper
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.LandingProductList
import com.ai.osmos.models.ads.MediaElement
import com.ai.osmos.models.ads.MultiAd
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.ads.PDAAd
import com.ai.osmos.models.ads.PDAAdElement
import com.ai.osmos.models.ads.ProductItem
import com.ai.osmos.models.ads.VideoAd
import org.json.JSONObject


internal class JSONParsing {
    companion object {

        fun parseAdData(response: String, cliUbid: String): Map<String, List<BaseAd>> {
            val result = mutableMapOf<String, List<BaseAd>>()
            try {
                val dataObject = JSONObject(response)

                val adsObject = dataObject.optJSONObject("ads") ?: return result

                val keys = adsObject.keys()
                while (keys.hasNext()) {
                    val key = keys.next()
                    val adArray = adsObject.getJSONArray(key)
                    val adList = mutableListOf<BaseAd>()

                    for (i in 0 until adArray.length()) {
                        val adObj = adArray.getJSONObject(i)
                        val elements = adObj.optJSONObject("elements") ?: continue

                        var adType = AdType.fromString(elements.optString("type", "IMAGE"))

                        if (elements.has("carousel_cards")) {
                            adType = AdType.fromString("CAROUSEL")
                        }
                        if (adArray.length() > 1) {
                            adType = AdType.fromString("MULTI_AD")
                        }
                        if (elements.has("product_list")) {
                            adType = AdType.fromString("PDA_AD")
                        }

                        val baseAd: BaseAd? = when (adType) {
                            AdType.IMAGE -> parseImageAd(adObj, cliUbid)
                            AdType.VIDEO -> parseVideoAd(adObj, cliUbid)
                            AdType.CAROUSEL -> parseCarouselAd(adObj, cliUbid)
                            AdType.MULTI_AD -> parseMultiAds(adObj, cliUbid)
                            AdType.NATIVE_AD -> null
                            AdType.PDA_AD -> parsePDAAds(adObj, cliUbid)
                            AdType.UNKNOWN -> null
                        }
                        baseAd?.let { adList.add(it) }
                    }
                    result[key] = adList
                }
                return result
            } catch (e: Exception) {
                return result
            }
        }

        private fun JSONObject.safeString(key: String, default: String = ""): String =
            if (has(key)) optString(key, default) else default

        private fun JSONObject.safeInt(key: String, default: Int = 0): Int =
            if (has(key)) optInt(key, default) else default

        private fun parseLandingProductList(elementsObject: JSONObject): List<LandingProductList> {
            val landingProducts = mutableListOf<LandingProductList>()

            if (elementsObject.has("landing_product_list")) {
                val landingArray = elementsObject.getJSONArray("landing_product_list")
                for (j in 0 until landingArray.length()) {
                    val productObj = landingArray.getJSONObject(j)

                    val landingProduct = LandingProductList(
                        skuId = productObj.optString("skuId"),
                        uclid = productObj.optString("uclid").takeIf { it != "NULL" },
                        storeId = productObj.optString("store_id").takeIf { it != "NULL" },
                        sellerId = productObj.optString("sellerId").takeIf { it != "NULL" },
                        itemGroupId = productObj.optString("itemGroupId").takeIf { it != "NULL" },

                        name = productObj.optString("name"),
                        brand = productObj.optString("brand").takeIf { it != "NULL" },
                        description = productObj.optString("description").takeIf { it != "NULL" },
                        customLabel0 = productObj.optString("customLabel0").takeIf { it != "NULL" },
                        customLabel1 = productObj.optString("customLabel1").takeIf { it != "NULL" },
                        customLabel2 = productObj.optString("customLabel2").takeIf { it != "NULL" },
                        customLabel3 = productObj.optString("customLabel3").takeIf { it != "NULL" },
                        customLabel4 = productObj.optString("customLabel4").takeIf { it != "NULL" },

                        mrp = productObj.optDouble("mrp").toFloat(),
                        salePrice = productObj.optDouble("salePrice").toFloat(),

                        imageUrl = productObj.optString("imageUrl").takeIf { it != "NULL" },
                        videoUrl = productObj.optString("videoUrl").takeIf { it != "NULL" },
                        productUrl = productObj.optString("productUrl"),

                        categoryL1 = productObj.optString("categoryL1").takeIf { it != "NULL" },
                        categoryL2 = productObj.optString("categoryL2").takeIf { it != "NULL" },
                        categoryL3 = productObj.optString("categoryL3").takeIf { it != "NULL" },
                    )

                    landingProducts.add(landingProduct)
                }
            }
            return landingProducts
        }

        private fun parseImageAd(adObject: JSONObject, cliUbid: String): ImageAd? {
            val elementsObject = adObject.optJSONObject("elements") ?: return null

            val landingProducts = parseLandingProductList(elementsObject)

            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                landingProductList = landingProducts,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            return ImageAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
        }

        private fun parseVideoAd(adObject: JSONObject, cliUbid: String): VideoAd? {
            val elementsObject = adObject.optJSONObject("elements") ?: return null

            val landingProducts = parseLandingProductList(elementsObject)

            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                landingProductList = landingProducts,
                videoProgressSec = if (elementsObject.has("video_progress_sec")) {
                    elementsObject.getInt("video_progress_sec")
                } else null,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            return VideoAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
        }

        private fun parseCarouselAd(adObject: JSONObject, cliUbid: String): CarouselAd? {

            val elementsObject = adObject.optJSONObject("elements") ?: return null
            val carouselArray = elementsObject.optJSONArray("carousel_cards") ?: return null

            val carouselCards = mutableListOf<CarouselAdElement>()
            for (i in 0 until carouselArray.length()) {
                val cardObj = carouselArray.optJSONObject(i) ?: continue
                val mediaObj = cardObj.optJSONObject("media") ?: continue

                val media = MediaElement(
                    value = mediaObj.optString("value"),
                    height = mediaObj.optInt("height"),
                    width = mediaObj.optInt("width"),
                    type = mediaObj.optString("type")
                )

                val element = CarouselAdElement(
                    media = media,
                    destinationUrl = cardObj.optString("destination_url")
                )

                carouselCards.add(element)
            }

            return CarouselAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = CarouselAdElementsWrapper(carouselCards = carouselCards), // wrapped
                cliUbid = cliUbid
            )
        }

        private fun parseMultiAds(adObject: JSONObject, cliUbid: String): MultiAd? {

            val elementsObject = adObject.optJSONObject("elements") ?: return null

            val landingProducts = parseLandingProductList(elementsObject)

            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                landingProductList = landingProducts,
                videoProgressSec = if (elementsObject.has("video_progress_sec")) {
                    elementsObject.getInt("video_progress_sec")
                } else null,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            val multiAd = MultiAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
            return multiAd
        }

        fun parseNativeAdData(response: String, cliUbid: String): Map<String, List<BaseAd>> {
            val result = mutableMapOf<String, List<BaseAd>>()
            try {
                val dataObject = JSONObject(response)

                val productsArray = dataObject.optJSONArray("products") ?: return result

                val nativeAdList = mutableListOf<BaseAd>()

                for (i in 0 until productsArray.length()) {
                    val productObj = productsArray.getJSONObject(i)

                    val nativeAd = NativeAd(
                        clientId = productObj.optLong("client_id", 0),
                        name = productObj.safeString("name"),
                        mrp = productObj.optDouble("mrp", 0.0).toFloat(),
                        skuId = productObj.safeString("skuId"),
                        productUrl = productObj.safeString("productUrl"),
                        storeId = productObj.optString("store_id", null),
                        imageUrl = productObj.optString("imageUrl", null),
                        videoUrl = productObj.optString("videoUrl", null),
                        salePrice = productObj.optDouble("salePrice", Double.NaN)
                            .takeIf { !it.isNaN() }?.toFloat(),
                        sellerId = productObj.optString("sellerId", null),
                        itemGroupId = productObj.optString("itemGroupId", null),
                        brand = productObj.optString("brand", null),
                        description = productObj.optString("description", null),
                        customLabel0 = productObj.optString("customLabel0", null),
                        customLabel1 = productObj.optString("customLabel1", null),
                        customLabel2 = productObj.optString("customLabel2", null),
                        customLabel3 = productObj.optString("customLabel3", null),
                        customLabel4 = productObj.optString("customLabel4", null),
                        uclid = productObj.optString("uclid", null),
                        categoryL1 = productObj.optString("categoryL1", null),
                        categoryL2 = productObj.optString("categoryL2", null),
                        categoryL3 = productObj.optString("categoryL3", null),
                        adMetadata = null,
                        cliUbid = cliUbid,
                        crt = productObj.safeString("crt")
                    )

                    nativeAdList.add(nativeAd)
                }

                result["native_ads"] = nativeAdList
                return result
            } catch (e: Exception) {
                return result
            }
        }

        private fun parsePDAAds(adObj: JSONObject, cliUbid: String): PDAAd {
            val elementsJson = adObj.getJSONObject("elements")
            val productListJson = elementsJson.getJSONArray("product_list")

            val productList = mutableListOf<ProductItem>()
            for (j in 0 until productListJson.length()) {
                val productJson = productListJson.getJSONObject(j)
                val product = ProductItem(
                    skuId = productJson.optString("skuId"),
                    name = productJson.optString("name"),
                    mrp = productJson.optDouble("mrp").toFloat(),
                    storeId = productJson.optString("store_id"),
                    productUrl = productJson.optString("productUrl"),
                    imageUrl = productJson.optString("imageUrl", null),
                    videoUrl = productJson.optString("videoUrl", null),
                    salePrice = if (productJson.has("salePrice")) productJson.optDouble("salePrice")
                        .toFloat() else null,
                    sellerId = productJson.optString("sellerId", null),
                    itemGroupId = productJson.optString("itemGroupId", null),
                    brand = productJson.optString("brand", null),
                    description = productJson.optString("description", null),
                    customLabel0 = productJson.optString("customLabel0", null),
                    customLabel1 = productJson.optString("customLabel1", null),
                    customLabel2 = productJson.optString("customLabel2", null),
                    customLabel3 = productJson.optString("customLabel3", null),
                    customLabel4 = productJson.optString("customLabel4", null),
                    categoryL1 = productJson.optString("categoryL1", null),
                    categoryL2 = productJson.optString("categoryL2", null),
                    categoryL3 = productJson.optString("categoryL3", null),
                    sellerClientId = productJson.optLong("sellerClientId"),
                    uclid = productJson.optString("uclid"),
                    rank = productJson.optInt("rank"),
                    cliUbid = cliUbid
                )
                productList.add(product)
            }

            val carouselCards = mutableListOf<CarouselAdElement>()
            if (elementsJson.has("carousel_cards")) {
                val carouselArray = elementsJson.getJSONArray("carousel_cards")
                for (i in 0 until carouselArray.length()) {
                    val cardObj = carouselArray.getJSONObject(i)
                    val mediaObj = cardObj.getJSONObject("media")

                    val media = MediaElement(
                        value = mediaObj.optString("value"),
                        height = mediaObj.optInt("height"),
                        width = mediaObj.optInt("width"),
                        type = mediaObj.optString("type")
                    )

                    val element = CarouselAdElement(
                        media = media,
                        destinationUrl = cardObj.optString("destination_url")
                    )

                    carouselCards.add(element)
                }
            }

            val elements = PDAAdElement(
                type = elementsJson.optString("type", ""),
                value = elementsJson.optString("value", null),
                carouselCards = carouselCards,
                width = elementsJson.safeInt("width"),
                height = elementsJson.safeInt("height"),
                productList = productList
            )

            return PDAAd(
                clientId = adObj.optLong("client_id"),
                au = adObj.optString("au"),
                rank = adObj.optInt("rank"),
                clickTrackingUrl = adObj.optString("click_tracking_url"),
                impressionTrackingUrl = adObj.optString("impression_tracking_url"),
                uclid = adObj.optString("uclid"),
                elements = elements,
                crt = adObj.optString("crt"),
                cliUbid = cliUbid,
                adMetadata = null
            )
        }
    }
}
