package com.ai.osmos.utils.ui

import android.app.Activity
import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.Rect
import android.graphics.drawable.GradientDrawable
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.renderer.loaders.VideoLoader
import com.ai.osmos.ads.views.BasePLAAdView
import com.ai.osmos.models.ads.PDAAd
import kotlin.math.abs


/**
 * Project Name: OSMOS-Android-SDK
 * File Name: ViewUtils
 */

/**
 * Utility object for common view-related helpers used across the SDK.
 */
object ViewUtils {
    private var currentIndex = -1

    /**
     * Returns a placeholder empty [View] to be shown when no ad is available or fails to load.
     *
     * This view is:
     * - A hidden [TextView] with empty text.
     * - Centered inside a [FrameLayout].
     * - Marked as [View.GONE] by default.
     *
     * @return A [View] instance representing an invisible empty view.
     */
    internal fun getEmptyView(context: Context): View {
        return TextView(context).apply {
            text = ""
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.CENTER
            }
            visibility = View.GONE
        }
    }

    internal fun createStyledButton(
        context: Context,
        iconRes: Int,
        size: Int,
        gravity: Int,
        marginStart: Int = 0,
        marginTop: Int = 0,
        marginEnd: Int = 0,
        marginBottom: Int = 0,
        onClick: (() -> Unit)? = null
    ): ImageButton {

        return ImageButton(context).apply {
            layoutParams = FrameLayout.LayoutParams(size, size).apply {
                this.gravity = gravity
                setMargins(marginStart, marginTop, marginEnd, marginBottom)
            }

            setImageResource(iconRes)
            setColorFilter(Color.BLACK, PorterDuff.Mode.SRC_IN)
            scaleType = ImageView.ScaleType.FIT_CENTER
            background = GradientDrawable().apply {
                shape = GradientDrawable.OVAL
                setStroke(3, Color.BLACK)
                setColor(Color.WHITE)
            }

            setOnClickListener { onClick?.invoke() }
        }
    }

    internal fun dpToPx(context: Context, dp: Int): Int {
        return TypedValue.applyDimension(
            TypedValue.COMPLEX_UNIT_DIP,
            dp.toFloat(),
            context.resources.displayMetrics
        ).toInt()
    }


    internal fun createAdIndicatorContainer(
        context: Context,
        label: String,
        alignment: Int?,
        width: Int,
        height: Int,
        imageView: ImageView?,
        videoView: FrameLayout?,
        isInterstitialAdView: Boolean? = false
    ): FrameLayout {

        val container = FrameLayout(context).apply {
            layoutParams = FrameLayout.LayoutParams(
                if (isInterstitialAdView == true) width else ViewGroup.LayoutParams.MATCH_PARENT,
                height
            )
        }
        if (imageView != null) {
            container.addView(imageView)
        }

        if (videoView != null) container.addView(videoView)

        // Add label as per alignment, default top-left
        val textView = getAdLabelTextview(context, label, alignment, 5, 5)

        container.addView(textView)
        return container
    }

    fun getAdLabelTextview(
        context: Context,
        label: String,
        alignment: Int?,
        topMargin: Int,
        startMargin: Int
    ): TextView {
        return TextView(context).apply {
            text = label
            setTextColor(Color.WHITE)
            textSize = 14f
            setBackgroundColor(Color.parseColor("#66808080")) //66808080
            setPadding(5, 5, 5, 5)
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT,
                if (alignment == null || alignment == Gravity.CENTER) {
                    Gravity.TOP or Gravity.START
                } else {
                    alignment
                }
            ).apply {
                setMargins(startMargin, topMargin, 5, 5)
            }
        }
    }

    internal fun getScreenHeightWidth(
        widthPercentage: Float,
        heightPercentage: Float
    ): Pair<Int, Int> {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels

        val maxAdWidth = (screenWidth * widthPercentage).toInt()
        val maxAdHeight = (screenHeight * heightPercentage).toInt()

        return Pair(maxAdWidth, maxAdHeight)
    }


    internal fun setItemDecoration(context: Context, recyclerView: RecyclerView) {
        val density = context.resources.displayMetrics.density

        val margin = (10 * density).toInt()
        recyclerView.addItemDecoration(object : RecyclerView.ItemDecoration() {
            override fun getItemOffsets(
                outRect: Rect,
                view: View,
                parent: RecyclerView,
                state: RecyclerView.State
            ) {
                val position = parent.getChildAdapterPosition(view)
                outRect.left = if (position == 0) margin else margin / 2
                outRect.right = if (position == state.itemCount - 1) margin else margin / 2
            }
        })
    }

    internal fun addDotIndicator(
        context: Context,
        ad: PDAAd,
    ): LinearLayout {
        val dotsContainer = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            gravity = Gravity.CENTER
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL
                bottomMargin = 16
            }
        }

        ad.elements.carouselCards?.forEach { _ ->
            dotsContainer.addView(TextView(context).apply {
                text = "•"
                textSize = 20f
                setTextColor(Color.GRAY)
            })
        }

        return dotsContainer
    }

    internal fun handleVisibleItem(
        recyclerView: RecyclerView,
        videoLoader: VideoLoader,
        dotsContainer: LinearLayout?
    ) {
        val center = recyclerView.width / 2
        var bestIndex = -1
        var minDistance = Int.MAX_VALUE

        // Find the closest item to the center
        for (i in 0 until recyclerView.childCount) {
            if (recyclerView.childCount == 0) return
            val view = recyclerView.getChildAt(i)
            val distance = abs(view.left + view.width / 2 - center)
            if (distance < minDistance) {
                bestIndex = recyclerView.getChildAdapterPosition(view)
                minDistance = distance
            }
        }

        // If the bestIndex has changed, update the visible item
        if (bestIndex != currentIndex) {
            // Pause the currently playing video (if any)

            pauseCurrentPlayingVideo(recyclerView, currentIndex)

            // Update the currentIndex
            currentIndex = bestIndex
            updateDots(dotsContainer)

            // Play the new video
            playCurrentVideo(recyclerView, currentIndex, videoLoader)
        }
    }

    private fun playCurrentVideo(
        recyclerView: RecyclerView,
        currentIndex: Int,
        videoLoader: VideoLoader
    ) {
        val newHolder =
            recyclerView.findViewHolderForAdapterPosition(currentIndex) as? BasePLAAdView.BannerCarouselAdapter.VideoViewHolder
        newHolder?.frame?.post {
            newHolder.playerView.let {
                videoLoader.playExclusive(it)
            }
        }
    }

    private fun pauseCurrentPlayingVideo(recyclerView: RecyclerView, currentIndex: Int) {
        val currentHolder =
            recyclerView.findViewHolderForAdapterPosition(currentIndex) as? BasePLAAdView.BannerCarouselAdapter.VideoViewHolder
        currentHolder?.let {
            val currentPlayer = it.playerView.player
            currentPlayer?.playWhenReady = false  // Pause the current video
        }
    }

    private fun updateDots(dotsContainer: LinearLayout?) {
        dotsContainer.let {
            for (i in 0 until it!!.childCount) {
                (it.getChildAt(i) as? TextView)?.setTextColor(
                    if (i == currentIndex) Color.WHITE else Color.GRAY
                )
            }
        }
    }
}