package ai.passio.passiosdk.core.network

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.token.TokenUsageTracker
import android.util.Log
import java.io.BufferedReader
import java.io.InputStream
import java.io.InputStreamReader
import javax.net.ssl.HttpsURLConnection

private const val TAG = "Network-"

internal abstract class NetworkTask<T> {

    var enableLog: Boolean = false
    protected var trackTokens: Boolean = false
    private var apiName: String? = null
    private var errorMessage: String? = null

    abstract fun executeTask(): T

    fun executeTaskTrackTokens(apiName: String): T {
        this.apiName = apiName
        trackTokens = true
        return executeTask()
    }

    fun getBody(inStream: InputStream?): String {
        if (inStream == null) return ""

        val streamReader = InputStreamReader(inStream)
        val reader = BufferedReader(streamReader)
        val stringBuilder = StringBuilder()

        var inputLine: String? = null
        while ({ inputLine = reader.readLine(); inputLine }() != null) {
            stringBuilder.append(inputLine)
        }

        return stringBuilder.toString()
    }

    fun logRequest(conn: HttpsURLConnection) {
        PassioLog.netI(getTag(), "Request -------------------------")
        PassioLog.netI(getTag(), "${conn.requestMethod} ${conn.url}")
        conn.requestProperties.forEach { (key, value) ->
            PassioLog.netI(TAG, "<Header> $key: ${value.concatenate()}")
        }
    }

    fun logBody(json: String) {
        PassioLog.netI(getTag(), "Body -------------------------")
        PassioLog.netI(getTag(), json)
    }

    fun logResponse(conn: HttpsURLConnection) {
        PassioLog.netI(getTag(), "Response -------------------------")
//        conn.headerFields.forEach { (key, value) ->
//            PassioLog.netI(TAG, "<Header> $key: ${value.concatenate()}")
//        }
        if (conn.responseCode !in (200..299)) {
            errorMessage = getBody(conn.errorStream)
            PassioLog.netE(getTag(), "${conn.responseCode} $errorMessage")
        } else {
            PassioLog.netI(getTag(), "${conn.responseCode} ${conn.responseMessage}")
        }


    }

    private fun List<String>.concatenate(): String {
        return this.map { it }.reduce { acc, s -> acc + s }
    }

    fun checkResponse(conn: HttpsURLConnection): Boolean {
        if (conn.responseCode !in (200..299)) {
            val message = errorMessage ?: getBody(conn.errorStream)
            conn.errorStream.close()
            if (message.isEmpty()) {
                throw NetworkException(conn.responseCode, "can't obtain body")
            } else {
                throw NetworkException(conn.responseCode, message)
            }
        }
        return true
    }

    private fun getTag(): String = TAG + Thread.currentThread().name.split("-").last()

    fun trackTokens(conn: HttpsURLConnection) {
        val tokenUsage = conn.headerFields["X-Request-Usage"]?.first()?.toInt()
        val overallUsage = conn.headerFields["X-Period-Usage"]?.first()?.toLong()
        val budget = conn.headerFields["X-Budget-Cap"]?.first()?.toLong()
        if (tokenUsage != null && overallUsage != null && budget != null) {
            TokenUsageTracker.track(apiName ?: "unknown", tokenUsage, overallUsage, budget)
        }
    }
}