package ai.passio.passiosdk.passiofood

import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.passiofood.data.model.PassioTokenBudget
import ai.passio.passiosdk.passiofood.nutritionfacts.PassioNutritionFacts
import android.graphics.Bitmap
import android.graphics.RectF
import android.net.Uri


typealias PassioID = String

/**
 * String that represents the code of barcode types FORMAT_EAN_13,
 * FORMAT_EAN_8, FORMAT_UPC_E and FORMAT_UPC_A.
 */
typealias Barcode = String

/**
 * String that represents the barcode of a packaged food that is
 * recognized using the image of the packaging.
 */
typealias PackagedFoodCode = String

internal typealias VotedCandidate = ObjectDetectionCandidate

/**
 * Data class that represents the results of the SDK's detection process.
 */
data class FoodCandidates(
    /**
     * Result of the food detection process.
     */
    val detectedCandidates: List<DetectedCandidate>? = null,
    /**
     * Result of the barcode detection process.
     */
    val barcodeCandidates: List<BarcodeCandidate>? = null,
    /**
     * Results of the ocr detection process.
     */
    val packagedFoodCandidates: List<PackagedFoodCandidate>? = null
)

/**
 * Class that represents the results of the food detection process.
 */
open class DetectedCandidate(
    /**
     * The id used to query the nutritional database.
     */
    val passioID: PassioID,

    val foodName: String,
    /**
     * Percentage of the neural network confidence level. Ranges from 0f - 1f or 0% to 100%.
     */
    val confidence: Float,
    /**
     * Relative coordinates of the object detected on an image. Using
     * [PassioSDK.boundingBoxToViewTransform] will determine the correct coordinates in coordinate
     * system of current view hierarchy.
     */
    val boundingBox: RectF,
    /**
     * Represents a part of the original image that the classification process has been ran on.
     */
    val croppedImage: Bitmap?,

    val alternatives: List<DetectedCandidate>
)

internal class DebugCandidate(
    passioID: PassioID,
    name: String,
    val votedCandidate: VotedCandidate,
    val odCandidate: ObjectDetectionCandidate,
    val hnnCandidate: ClassificationCandidate,
    val knnCandidate: ClassificationCandidate,
    croppedImage: Bitmap?,
    val processTime: Long = 0L,
    alternatives: List<DetectedCandidate> = listOf()
) : DetectedCandidate(
    passioID,
    name,
    votedCandidate.confidence,
    odCandidate.boundingBox,
    croppedImage,
    alternatives
) {
    override fun toString(): String = "vote: ${votedCandidate.passioID}, " +
            "od: ${odCandidate.passioID}, " +
            "hnn: ${hnnCandidate.passioID}, " +
            "knn: ${knnCandidate.passioID}"
}

/**
 * Represents the result of the classification process.
 */
open class ClassificationCandidate(var passioID: PassioID, val confidence: Float)

/**
 * Represents the result of the packaged food detection
 * process.
 */
data class PackagedFoodCandidate(
    val packagedFoodCode: PackagedFoodCode,
    val confidence: Float
)

/**
 * Represents the result of the object detection process.
 */
class ObjectDetectionCandidate(
    passioID: PassioID,
    confidence: Float,
    val boundingBox: RectF
) : ClassificationCandidate(passioID, confidence) {

    override fun toString(): String {
        return "passioID: $passioID, " +
                "confidence: $confidence, " +
                "boundingBox: ${boundingBox.toShortString()}"
    }
}

/**
 * Data class that represents the results of the barcode detection process.
 */
data class BarcodeCandidate(
    /**
     * Barcode string used to fetch nutritional information using the
     * [PassioSDK.fetchPassioIDAttributesForBarcode] method.
     */
    val barcode: Barcode,
    /**
     * The bounding box of the detected barcode, in absolute coordinates.
     */
    val boundingBox: RectF
)

/**
 * Interface that serves as a callback for the [PassioSDK.startFoodDetection] method.
 */
interface FoodRecognitionListener {

    /**
     * Callback method for the food detection process.
     *
     * @param candidates top level object that represents all of the detection results based on the
     *        configuration of the detection session.
     * @param image that represents the camera frame that the recognition was ran on.
     * @param nutritionFacts recognized nutrition facts for the camera frame.
     */
    fun onRecognitionResults(
        candidates: FoodCandidates?,
        image: Bitmap?
    )
}

interface NutritionFactsRecognitionListener {
    fun onRecognitionResult(nutritionFacts: PassioNutritionFacts?, text: String)
}

/**
 * Registering this callback gives the ability to
 * receive the results of the object detection
 * process.
 */
internal interface ObjectDetectionListener {
    fun onObjectDetectionResult(candidates: List<ObjectDetectionCandidate>)
}

/**
 * Registering this callback gives the ability to
 * receive the results of the classification
 * process.
 */
internal interface ClassificationListener {
    fun onClassificationResult(candidates: List<ClassificationCandidate>)
}

/**
 * Callback interface used to receive information about
 * the SDK's configuration process.
 */
interface PassioStatusListener {
    /**
     * Every time the SDK's configuration process changes
     * a state, a new event will be emitted with the
     * current [PassioStatus].
     */
    fun onPassioStatusChanged(status: PassioStatus)

    /**
     * Will be called once all of the files are downloaded.
     * This doesn't mean that the SDK will immediately run
     * the newly downloaded files.
     */
    fun onCompletedDownloadingAllFiles(fileUris: List<Uri>)

    /**
     * Signals the completion of the download process for a
     * single files. This method also informs how many files
     * are still left in the download queue.
     */
    fun onCompletedDownloadingFile(fileUri: Uri, filesLeft: Int)

    /**
     * If a certain file cannot be downloaded, [onDownloadError]
     * will be invoked with the download error message attached.
     */
    fun onDownloadError(message: String)
}

data class InflammatoryEffectData(
    val nutrient: String,
    val amount: Double,
    val unit: String,
    val inflammatoryEffectScore: Double,
)

data class PassioFoodDataInfo(
    val foodName: String,
    val brandName: String,
    val iconID: PassioID,
    val score: Double,
    val scoredName: String,
    val labelId: String,
    val type: String,
    val resultId: String,
    val isShortName: Boolean,
    val nutritionPreview: PassioSearchNutritionPreview,
    val tags: List<String>?
)

data class PassioSearchNutritionPreview(
    val calories: Int,
    val carbs: Double,
    val protein: Double,
    val fat: Double,
    val fiber: Double,
    val servingUnit: String,
    val servingQuantity: Double,
    val weightUnit: String,
    val weightQuantity: Double
)

enum class PassioMealTime(val mealName: String) {
    BREAKFAST("breakfast"),
    LUNCH("lunch"),
    DINNER("dinner"),
    SNACK("snack")
}

enum class PassioImageResolution {
    RES_512,
    RES_1080,
    FULL
}

interface PassioAccountListener {
    fun onTokenBudgetUpdate(tokenBudget: PassioTokenBudget)
}


