package ai.passio.passiosdk.passiofood.advisor

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.PostNetworkTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.ImageUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioFoodDataInfo
import ai.passio.passiosdk.passiofood.PassioImageResolution
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorFoodInfo
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorResponse
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioFoodResultType
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.data.network.ResponseAdvisorItem
import ai.passio.passiosdk.passiofood.toDataModel
import ai.passio.passiosdk.passiofood.upc.ResponseIngredient
import ai.passio.passiosdk.passiofood.utils.MarkdownHelper
import android.graphics.Bitmap
import android.util.Base64
import android.util.Log
import org.json.JSONArray
import org.json.JSONObject
import java.io.ByteArrayOutputStream

private const val IMAGE_URL = "vision/extractIngredients"
private const val PACKAGED_PRODUCT_URL = "vision/extractPackagedProduct"
private const val HIDDEN_INGREDIENTS_URL = "invisibleIngredientSuggestions"
private const val ALTERNATIVES_URL = "visualAlternativeSuggestions"
private const val POSSIBLE_INGREDIENTS_URL = "possibleIngredientSuggestions"
private const val PROFILE_URL = "generateIntelligenceProfile"

internal class AdvisorService(private val dev: Boolean) {

    private var currentThread: String? = null
    private var userProfile: UserProfile? = null

    fun initConversation(callback: (result: PassioResult<Any>) -> Unit) {
        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val url = NativeUtils.instance.nativeGetAdvisorBaseUrl(dev)
            val task = PostNetworkTask(
                url + "threads",
                listOf(
                    "Content-Type" to "application/json",
                    "Authorization" to token,
                ),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    val errorMessage = "Could not fetch thread: $message"
                    PassioLog.e(AdvisorService::class.java.simpleName, errorMessage)
                    callback(PassioResult.Error(errorMessage))
                }

                override fun onTokenExpired() {
                    initConversation(callback)
                }

                override fun onSuccess(result: String) {
                    val responseJson = JSONObject(result)
                    currentThread = responseJson.optString("threadId")
                    callback(PassioResult.Success(Any()))
                }
            }, "initConversation")
        }
    }

    fun sendMessage(
        message: String,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        if (currentThread == null) {
            callback(PassioResult.Error("Conversation not initialized. Must call the initConversation() first"))
            return
        }

        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val body = JSONObject()
            body.put("inputSensors", null)
            body.put("message", message)
            userProfile?.let {
                body.put(it.headerKey, it.profileValue)
            }

            val url = NativeUtils.instance.nativeGetAdvisorBaseUrl(dev)
            val task = PostNetworkTask(
                "${url}threads/$currentThread/messages",
                listOf(
                    "Content-Type" to "application/json",
                    "Authorization" to token,
                ),
                body.toString(),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    val errorMessage = "Could not send message: $message"
                    PassioLog.e(AdvisorService::class.java.simpleName, errorMessage)
                    callback(PassioResult.Error(errorMessage))
                }

                override fun onTokenExpired() {
                    sendMessage(message, callback)
                }

                override fun onSuccess(result: String) {
                    val response = ResponseTextAdvisor(result)
                    val model = PassioAdvisorResponse(
                        response.threadId,
                        response.messageId,
                        response.content,
                        MarkdownHelper.removeMarkdownTags(response.content),
                        response.contentToolHints,
                        null
                    )
                    callback(PassioResult.Success(model))
                }
            }, "sendMessage")
        }
    }

    fun sendImage(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        languageCode: String?,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        if (currentThread == null) {
            callback(PassioResult.Error("Conversation not initialized. Must call the initConversation() first"))
            return
        }

        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val encoded = imageToBase64(bitmap, resolution)
            val body = "data:image/jpeg;base64,$encoded"

            val jsonBody = JSONObject().apply {
                put("image", body)
            }

            val headers = mutableListOf(
                "Content-Type" to "application/json",
                "Authorization" to token,
            )
            if (languageCode != null) {
                headers.add("Localization-ISO" to languageCode)
            }
            val url = NativeUtils.instance.nativeGetAdvisorBaseUrl(dev)
            val task = PostNetworkTask(
                "${url}threads/${currentThread}/messages/tools/vision/VisualFoodExtraction",
                headers,
                jsonBody.toString(),
                readTimeout = 20000,
            )


            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    val errorMessage = "Could not send image: $message"
                    PassioLog.e(AdvisorService::class.java.simpleName, errorMessage)
                    callback(PassioResult.Error(errorMessage))
                }

                override fun onTokenExpired() {
                    sendImage(bitmap, resolution, languageCode, callback)
                }

                override fun onSuccess(result: String) {
                    val response = parseSearchIngredientResponse("sendImage", result)
                    callback(response)
                }
            }, "sendImage")
        }
    }

    fun fetchIngredients(
        response: PassioAdvisorResponse,
        languageCode: String?,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        if (currentThread == null) {
            callback(PassioResult.Error("Conversation not initialized. Must call the initConversation() first"))
            return
        }

        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val body = JSONObject()
            body.put("messageId", response.messageId)
            val headers = mutableListOf(
                "Content-Type" to "application/json",
                "Authorization" to token,
            )
            if (languageCode != null) {
                headers.add("Localization-ISO" to languageCode)
            }

            val url = NativeUtils.instance.nativeGetAdvisorBaseUrl(dev)
            val task = PostNetworkTask(
                "${url}threads/$currentThread/messages/tools/target/SearchIngredientMatches",
                headers,
                body.toString(),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    val errorMessage = "Could not fetchIngredients: $message"
                    PassioLog.e(AdvisorService::class.java.simpleName, errorMessage)
                    callback(PassioResult.Error(errorMessage))
                }

                override fun onTokenExpired() {
                    fetchIngredients(response, languageCode, callback)
                }

                override fun onSuccess(result: String) {
                    val ingredientResponse =
                        parseSearchIngredientResponse("fetchIngredients", result)
                    callback(ingredientResponse)
                }
            }, "fetchIngredients")
        }
    }

    private fun toolsAPI(
        url: String,
        content: String,
        retries: Int,
        apiName: String,
        languageCode: String?,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {

        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val body = JSONObject()
            body.put("content", content)

            val headers = mutableListOf(
                "Content-Type" to "application/json",
                "Authorization" to token,
            )
            if (languageCode != null) {
                headers.add("Localization-ISO" to languageCode)
            }

            val baseUrl = NativeUtils.instance.nativeGetAdvisorToolsUrl(dev)
            val task = PostNetworkTask(
                baseUrl + url,
                headers,
                body.toString(),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(
                task,
                object : AdvisorNetworkCallback<String>() {
                    override fun onSuccess(result: String) {
                        val ingredients = parseIngredients(result)
                        callback(PassioResult.Success(ingredients))
                    }

                    override fun onRateLimiting(message: String) {
                        Log.e(
                            AdvisorService::class.java.simpleName,
                            "Rate limiting error: $message, retries: $retries"
                        )
                        if (retries == 0) {
                            onAdvisorFailure(message)
                            return
                        }

                        val newRetries = retries - 1
                        toolsAPI(url, content, newRetries, apiName, languageCode, callback)
                    }

                    override fun onAdvisorFailure(message: String) {
                        val urlPath = url.split("/").last()
                        val errorMessage =
                            "Could not fetch $urlPath for argument $content: $message"
                        PassioLog.e(AdvisorService::class.java.simpleName, errorMessage)
                        callback(PassioResult.Error(errorMessage))
                    }

                    override fun onTokenExpired() {
                        toolsAPI(url, content, retries, apiName, languageCode, callback)
                    }
                },
                apiName
            )
        }
    }

    fun fetchHiddenIngredients(
        foodName: String,
        languageCode: String?,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        toolsAPI(
            HIDDEN_INGREDIENTS_URL,
            foodName,
            2,
            "fetchHiddenIngredients",
            languageCode,
            callback
        )
    }

    fun fetchVisualAlternatives(
        foodName: String,
        languageCode: String?,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        toolsAPI(ALTERNATIVES_URL, foodName, 2, "fetchVisualAlternatives", languageCode, callback)
    }

    fun fetchPossibleIngredients(
        foodName: String,
        languageCode: String?,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        toolsAPI(
            POSSIBLE_INGREDIENTS_URL,
            foodName,
            2,
            "fetchPossibleIngredients",
            languageCode,
            callback
        )
    }

    internal fun runImageRecognition(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        message: String?,
        languageCode: String?,
        callback: (result: List<PassioAdvisorFoodInfo>) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(emptyList())
                return@fetchToken
            }

            val encoded = imageToBase64(bitmap, resolution)
            val body = "data:image/jpeg;base64,$encoded"

            val jsonBody = JSONObject().apply {
                put("image", body)
                if (message != null) {
                    put("message", message)
                }
            }

            val headers = mutableListOf(
                "Content-Type" to "application/json",
                "Authorization" to token,
            )
            if (languageCode != null) {
                headers.add("Localization-ISO" to languageCode)
            }

            val url = NativeUtils.instance.nativeGetAdvisorToolsUrl(dev)
            val task = PostNetworkTask(
                url + IMAGE_URL,
                headers,
                jsonBody.toString(),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.e(
                        AdvisorService::class.java.simpleName,
                        "Could not fetch thread: $message"
                    )
                    callback(emptyList())
                }

                override fun onTokenExpired() {
                    runImageRecognition(bitmap, resolution, message, languageCode, callback)
                }

                override fun onSuccess(result: String) {
                    val list = parseIngredients(result)
                    callback(list)
                }
            }, "recognizeImageRemote")
        }
    }

    @Throws(InterruptedException::class)
    private fun imageToBase64(bitmap: Bitmap, resolution: PassioImageResolution): String {
        var format = Bitmap.CompressFormat.JPEG
        val resized = when (resolution) {
            PassioImageResolution.RES_512 -> bitmap.resize(512)
            PassioImageResolution.RES_1080 -> bitmap.resize(1080)
            PassioImageResolution.FULL -> {
                format = Bitmap.CompressFormat.PNG
                bitmap
            }
        }

        val byteArrayOutputStream = ByteArrayOutputStream()
        resized.compress(format, 100, byteArrayOutputStream)
        val byteArray = byteArrayOutputStream.toByteArray()
        return Base64.encodeToString(byteArray, Base64.NO_WRAP)
    }

    internal fun runPackageFoodRecognition(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        callback: (result: PassioFoodItem?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            val encoded = imageToBase64(bitmap, resolution)
            val body = "data:image/jpeg;base64,$encoded"

            val jsonBody = JSONObject().apply {
                put("image", body)
            }

            val url = NativeUtils.instance.nativeGetAdvisorToolsUrl(dev)
            val task = PostNetworkTask(
                url + PACKAGED_PRODUCT_URL,
                listOf(
                    "Content-Type" to "application/json",
                    "Authorization" to token,
                ),
                jsonBody.toString(),
                readTimeout = 30000,
                connectTimeout = 30000
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.e(
                        AdvisorService::class.java.simpleName,
                        "Could not runPackageFoodRecognition: $message"
                    )
                    callback(null)
                }

                override fun onTokenExpired() {
                    runPackageFoodRecognition(bitmap, resolution, callback)
                }

                override fun onSuccess(result: String) {
                    val response = ResponseIngredient(result)
                    val foodItem = PassioFoodItem.fromUPCResponse(response)
                    callback(foodItem)
                }
            }, "recognizePackagedProductRemote")
        }
    }

    private fun fetchToken(
        callback: (token: String?) -> Unit
    ) {
        TokenService.getInstance().getToken(dev, { token ->
            callback(token)
        }, { _ ->
            callback(null)
        })
    }

    private fun parseSearchIngredientResponse(
        apiName: String,
        result: String
    ): PassioResult<PassioAdvisorResponse> {
        val response = ResponseTextAdvisor(result)
        val ingredientJson = response.optJSONObject("actionResponse")?.optString("data")
        val ingredients = if (ingredientJson != null) parseIngredients(ingredientJson) else null
        val model = PassioAdvisorResponse(
            response.threadId,
            response.messageId,
            response.content,
            MarkdownHelper.removeMarkdownTags(response.content),
            response.contentToolHints,
            ingredients
        )
        return PassioResult.Success(model)
    }

    private fun parseIngredients(json: String): List<PassioAdvisorFoodInfo> {
        val jArray = JSONArray(json)
        val list = mutableListOf<PassioAdvisorFoodInfo>()
        for (i in 0 until jArray.length()) {
            val it = ResponseAdvisorItem(jArray.getJSONObject(i).toString())
            val foodDataInfo = PassioFoodDataInfo(
                it.shortName.ifEmpty { it.displayName },
                it.brandName,
                it.iconId,
                it.score,
                it.scoredName,
                it.labelId,
                it.type,
                it.resultId,
                true,
                it.nutritionPreview.toDataModel(),
                it.tags
            )
            val model = PassioAdvisorFoodInfo(
                it.ingredientName,
                it.portionSize,
                it.weightGrams,
                foodDataInfo,
                null,
                PassioFoodResultType.FOOD_ITEM
            )
            list.add(model)
        }
        return list
    }

    private fun Bitmap.resize(max: Int): Bitmap {
        if (width >= height) {
            if (width <= max) {
                return this
            }

            val ratio = max.toFloat() / width
            val dstHeight = ratio * height
            val matrix =
                ImageUtils.getTransformationMatrix(width, height, max, dstHeight.toInt(), 0, true)
            return Bitmap.createBitmap(this, 0, 0, width, height, matrix, false)
        } else {
            if (height <= max) {
                return this
            }

            val ratio = max.toFloat() / height
            val dstWidth = ratio * width
            val matrix =
                ImageUtils.getTransformationMatrix(width, height, dstWidth.toInt(), max, 0, true)
            return Bitmap.createBitmap(this, 0, 0, width, height, matrix, false)
        }
    }

    fun setAdvisorProfile(
        profileJson: String,
        callback: (result: PassioResult<Any>) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            val jsonBody = JSONObject().apply {
                put("content", profileJson)
            }

            val url = NativeUtils.instance.nativeGetAdvisorToolsUrl(dev)
            val task = PostNetworkTask(
                url + PROFILE_URL,
                listOf(
                    "Content-Type" to "application/json",
                    "Authorization" to token,
                ),
                jsonBody.toString(),
                readTimeout = 20000,
            )

            NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.e(
                        AdvisorService::class.java.simpleName,
                        "Could not setAdvisorProfile: $message"
                    )
                    callback(PassioResult.Error(message))
                }

                override fun onTokenExpired() {
                    setAdvisorProfile(profileJson, callback)
                }

                override fun onSuccess(result: String) {
                    val response = ResponseUserProfile(result)
                    userProfile = UserProfile(response.headerKey, response.profileValue)
                    callback(PassioResult.Success(Any()))
                }
            }, "setAdvisorProfile")
        }
    }
}

internal data class UserProfile(val headerKey: String, val profileValue: String)