package ai.passio.passiosdk.passiofood.data.model

import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.data.measurement.Grams
import ai.passio.passiosdk.passiofood.data.measurement.Milliliters
import ai.passio.passiosdk.passiofood.data.measurement.UnitMass
import ai.passio.passiosdk.passiofood.data.model.internal.ServingUnitHelper
import ai.passio.passiosdk.passiofood.upc.ResponsePortions
import ai.passio.passiosdk.passiofood.upc.ResponseWeight
import androidx.annotation.Keep

data class PassioFoodAmount(
    val servingSizes: List<PassioServingSize>,
    val servingUnits: List<PassioServingUnit>
) {
    var selectedUnit: String = servingSizes.first().unitName
    var selectedQuantity: Double = servingSizes.first().quantity

    companion object {

        @Keep
        const val SERVING_UNIT_NAME = "serving"

        internal fun fromResponse(
            portions: List<ResponsePortions>?,
            ingredientWeight: Double? = null
        ): PassioFoodAmount {
            if (portions == null) {
                // If no portions are found, return 100 grams as the default
                return PassioFoodAmount(
                    listOf(PassioServingSize(100.0, Grams.unitName)),
                    listOf(PassioServingUnit(Grams.unitName, UnitMass(Grams, 1.0)))
                )
            }

            val servingSizes = mutableListOf<PassioServingSize>()
            val servingUnits = mutableListOf<PassioServingUnit>()
            portions.forEach { portion ->
                val servingSizesAndUnit = getServingSizeAndUnit(portion) ?: return@forEach
                servingSizes.addAll(servingSizesAndUnit.first)
                servingUnits.add(servingSizesAndUnit.second)
            }

            if (servingSizes.isEmpty() && ingredientWeight != null) {
                // If there are no serving sizes for a recipe, add "1 serving" with the weight of
                // the summed up weight of the ingredients
                servingUnits.add(
                    PassioServingUnit(
                        SERVING_UNIT_NAME,
                        UnitMass(Grams, ingredientWeight)
                    )
                )
                servingSizes.add(PassioServingSize(1.0, SERVING_UNIT_NAME))
            }

            val conversionUnits = ServingUnitHelper.getConversionUnits(servingUnits)
            if (conversionUnits.isNotEmpty()) {
                servingUnits.addAll(conversionUnits)
            }

            // Grams should always be present in the serving unit
            if (servingUnits.find { it.unitName == Grams.unitName } == null) {
                servingUnits.add(PassioServingUnit(Grams.unitName, UnitMass(Grams, 1.0)))
            }

            // 100 grams should always be present in the serving sizes
            if (servingSizes.find { it.unitName == Grams.unitName && it.quantity == 100.0 } == null) {
                servingSizes.add(PassioServingSize(100.0, Grams.unitName))
            }

            return PassioFoodAmount(servingSizes, servingUnits)
        }

        private fun getServingSizeAndUnit(
            portion: ResponsePortions
        ): Pair<List<PassioServingSize>, PassioServingUnit>? {
            val servings = mutableListOf<PassioServingSize>()
            val unit: PassioServingUnit

            if (portion.name == null) {
                return null
            }

            if (portion.suggestedQuantity != null) {
                val suggestedServings = portion.suggestedQuantity!!.map { suggestedQuantity ->
                    PassioServingSize(suggestedQuantity, portion.name!!)
                }
                servings.addAll(suggestedServings)
            } else if (portion.quantity != null) {
                servings.add(PassioServingSize(portion.quantity!!, portion.name!!))
            } else {
                return null
            }

            if (portion.weight == null) {
                return null
            }

            val unitMass = parseWeight(portion.weight!!) ?: return null
            unit = PassioServingUnit(portion.name!!, unitMass)

            return Pair(servings, unit)
        }

        private fun parseWeight(responseWeight: ResponseWeight): UnitMass? {
            return when (responseWeight.unit) {
                "g", "G", "gr", "GR" -> UnitMass(Grams, responseWeight.value ?: 0.0)
                "ml", "ML" -> UnitMass(Milliliters, responseWeight.value ?: 0.0)
                else -> {
                    PassioLog.e(
                        this::class.java.simpleName,
                        "Unknown unit of weight: ${responseWeight.unit}"
                    )
                    null
                }
            }
        }
    }

    fun weight(): UnitMass {
        val selectedUnit =
            servingUnits.firstOrNull { it.unitName == selectedUnit } ?: return UnitMass(Grams, 0.0)
        return selectedUnit.weight * selectedQuantity
    }

    fun weightGrams(): Double {
        return weight().gramsValue()
    }

    fun setWeightInGrams(grams: Double) {
        selectedQuantity = grams
        selectedUnit = Grams.unitName
    }
}