package ai.passio.passiosdk.passiofood.nutritionfacts

import ai.passio.passiosdk.passiofood.nutritionfacts.dict.EnglishNutritionFactsDict
import ai.passio.passiosdk.passiofood.nutritionfacts.dict.FrenchNutritionFactsDict
import ai.passio.passiosdk.passiofood.nutritionfacts.dict.NutritionFactsDict
import com.google.mlkit.vision.text.Text
import java.util.regex.Pattern
import kotlin.math.max
import kotlin.math.min

private const val TEXT_LENGTH_PER_MISTAKE = 10
private const val BLOCK_VERTICAL_MAX_OFFSET = 8

internal class NutritionFactsReader {

    private data class OCRServingSize(
        val size: String,
        val nominalQuantity: Double?,
        val nominalUnit: String?,
    )

    private val parseGramsUnits = listOf("g", "9", "o")
    private val parseMilligramsUnits = listOf("mg", "mng")

    fun onTextRecognized(visionText: Text): PassioNutritionFacts? {
        val currentNutritionFact = PassioNutritionFacts()
        val currentDict = determineLanguage(visionText)

        visionText.textBlocks.forEach loop@{ textBlock ->
            if (textBlock == null) {
                return@loop
            }

            textBlock.lines.forEach { line ->
                if (line == null) {
                    return@forEach
                }

//                val matchCalories = areKeywordsPresent(currentDict.getCalories(), line.text)
//                if (matchCalories.first && currentNutritionFact.calories == null) {
//                    val horizontalLines = getLinesHorizontalTo(line, visionText)
//                    val value =
//                        parseLine(horizontalLines.toSingleLineString(), matchCalories.second, null)
//                            ?: return@forEach
//                    currentNutritionFact.calories = value
//                    return@forEach
//                }
                currentNutritionFact.calories =
                    matchNutrient(currentDict.getCalories(), line, visionText)
                        ?: currentNutritionFact.calories
//                val matchCarbs = areKeywordsPresent(currentDict.getCarbs(), line.text)
//                if (matchCarbs.first) {
//                    val horizontalLines = getLinesHorizontalTo(line, visionText)
//                    val value =
//                        parseLine(
//                            horizontalLines.toSingleLineString(),
//                            matchCarbs.second,
//                            parseGramsUnits
//                        )
//                            ?: return@forEach
//                    currentNutritionFact.carbs = value
//                    return@forEach
//                }
                currentNutritionFact.carbs =
                    matchNutrient(currentDict.getCarbs(), line, visionText, parseGramsUnits)
                        ?: currentNutritionFact.carbs
                val matchServingSize = areKeywordsPresent(currentDict.getServingSize(), line.text)
                if (matchServingSize.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val servingSize = parseServingSize(
                        horizontalLines.toSingleLineString(),
                        matchServingSize.second
                    ) ?: return@forEach

                    currentNutritionFact.servingSize = servingSize.size.trim()
                    currentNutritionFact.servingSizeQuantity = servingSize.nominalQuantity
                    currentNutritionFact.servingSizeUnitName = servingSize.nominalUnit?.trim()
                    return@forEach
                }
//                val matchFat = areKeywordsPresent(currentDict.getFat(), line.text)
//                if (currentNutritionFact.fat == null && matchFat.first) {
//                    val horizontalLines = getLinesHorizontalTo(line, visionText)
//                    val value =
//                        parseLine(
//                            horizontalLines.toSingleLineString(),
//                            matchFat.second,
//                            parseGramsUnits
//                        )
//                            ?: return@forEach
//                    currentNutritionFact.fat = value
//                    return@forEach
//                }
                currentNutritionFact.fat =
                    matchNutrient(
                        currentDict.getFat(),
                        line,
                        visionText,
                        parseGramsUnits,
                        currentDict.getSatFat() + currentDict.getTransFat()
                    )
                        ?: currentNutritionFact.fat
//                val matchProtein = areKeywordsPresent(currentDict.getProtein(), line.text)
//                if (matchProtein.first) {
//                    val horizontalLines = getLinesHorizontalTo(line, visionText)
//                    val value =
//                        parseLine(
//                            horizontalLines.toSingleLineString(),
//                            matchProtein.second,
//                            parseGramsUnits
//                        )
//                            ?: return@forEach
//                    currentNutritionFact.protein = value
//                    return@forEach
//                }
                currentNutritionFact.protein =
                    matchNutrient(currentDict.getProtein(), line, visionText, parseGramsUnits)
                        ?: currentNutritionFact.protein
                val matchSugars = areKeywordsPresent(currentDict.getSugars(), line.text)
                if (matchSugars.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val value =
                        parseLine(
                            horizontalLines.toSingleLineString(),
                            matchSugars.second,
                            parseGramsUnits
                        )
                            ?: return@forEach
                    currentNutritionFact.sugars = value
                    return@forEach
                }

                val matchSugarAlcohol = areKeywordsPresent(currentDict.getSugarAlcohol(), line.text)
                if (matchSugarAlcohol.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val value =
                        parseLine(
                            horizontalLines.toSingleLineString(),
                            matchSugarAlcohol.second,
                            parseGramsUnits
                        )
                            ?: return@forEach
                    currentNutritionFact.sugarAlcohol = value
                    return@forEach
                }

                val matchSatFat = areKeywordsPresent(currentDict.getSatFat(), line.text)
                if (matchSatFat.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val value =
                        parseLine(
                            horizontalLines.toSingleLineString(),
                            matchSatFat.second,
                            parseGramsUnits
                        )
                            ?: return@forEach
                    currentNutritionFact.saturatedFat = value
                    return@forEach
                }

                val matchTransFat = areKeywordsPresent(currentDict.getTransFat(), line.text)
                if (matchTransFat.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val value =
                        parseLine(
                            horizontalLines.toSingleLineString(),
                            matchTransFat.second,
                            parseGramsUnits
                        )
                            ?: return@forEach
                    currentNutritionFact.transFat = value
                    return@forEach
                }

                val matchCholesterol = areKeywordsPresent(currentDict.getCholesterol(), line.text)
                if (matchCholesterol.first) {
                    val horizontalLines = getLinesHorizontalTo(line, visionText)
                    val value =
                        parseLine(
                            horizontalLines.toSingleLineString(),
                            matchCholesterol.second,
                            parseMilligramsUnits
                        )
                            ?: return@forEach
                    currentNutritionFact.cholesterol = value
                    return@forEach
                }
            }
        }
        return if (currentNutritionFact.isEmpty()) null else currentNutritionFact
    }

    private fun matchNutrient(
        dict: List<String>,
        line: Text.Line,
        visionText: Text,
        units: List<String>? = null,
        exclude: List<String> = listOf()
    ): Double? {
        val match = areKeywordsPresent(dict, line.text)
        if (!match.first) return null
        exclude.forEach { ex ->
            if (line.text.toLowerCase().contains(ex)) {
                return null
            }
        }
        val horizontalLines = getLinesHorizontalTo(line, visionText)
        return parseLine(horizontalLines.toSingleLineString(), match.second, units)
    }

    private fun areKeywordsPresent(
        keywords: List<String>,
        targetText: String
    ): Pair<Boolean, String> {
        keywords.forEach { keyword ->
            if (targetText.length < keyword.length) {
                return@forEach
            }

            val maxMistakes = if (keyword.length < 5) {
                0
            } else {
                max(keyword.length / TEXT_LENGTH_PER_MISTAKE, 1)
            }

            if (targetText.length == keyword.length) {
                return if (distanceBetweenStrings(
                        keyword.toLowerCase(),
                        targetText.toLowerCase()
                    ) <= maxMistakes
                ) {
                    true to targetText.toLowerCase()
                } else {
                    false to ""
                }
            }

            for (i in 0 until targetText.length - keyword.length) {
                val substring = targetText.substring(i, i + keyword.length)
                if (distanceBetweenStrings(
                        keyword.toLowerCase(),
                        substring.toLowerCase()
                    ) <= maxMistakes
                ) {
                    return true to substring.toLowerCase()
                }
            }
        }
        return false to ""
    }

    private fun List<Text.Line>.toSingleLineString(): String {
        var result = ""
        this.sortedBy { it.boundingBox!!.left }.forEach {
            result += "${it.text} "
        }
        return result.substringBeforeLast(" ")
    }

    private fun getLinesHorizontalTo(targetLine: Text.Line, visionText: Text): List<Text.Line> {
        if (targetLine.boundingBox == null) {
            return emptyList()
        }

        val result = mutableListOf(targetLine)
        val centerLine = with(targetLine.boundingBox!!) {
            top + (bottom - top) / 2
        }
        val targetHeight = targetLine.boundingBox!!.height()

        visionText.textBlocks.forEach loop@{ textBlock ->
            if (textBlock == null) {
                return@loop
            }

            // Find line horizontal to targetLine
            textBlock.lines.forEach { line ->
                if (targetLine.boundingBox == line.boundingBox) {
                    return@forEach
                }

                if (line == null || line.boundingBox == null) {
                    return@forEach
                }

                if (line.boundingBox!!.top < centerLine && line.boundingBox!!.bottom > centerLine) {

                    // Do lines have similar height (font size)
                    val lineHeight = line.boundingBox!!.height()

                    if (targetHeight > lineHeight) {
                        if (lineHeight.toDouble() / targetHeight.toDouble() < 0.5) {
                            return@forEach
                        }
                    } else {
                        if (targetHeight.toDouble() / lineHeight.toDouble() < 0.5) {
                            return@forEach
                        }
                    }

                    result.add(line)
                }
            }
        }

        return result.sortedBy { it.boundingBox!!.right }
    }

    private fun parseLine(line: String, keyword: String, units: List<String>?): Double? {
        if (units == null) {
            val pattern = Pattern.compile("$keyword(\\S*)\\s*(\\d+)")
            val matcher = pattern.matcher(line.toLowerCase())
            return if (matcher.find()) {
                matcher.group(2)!!.toDouble()
            } else {
                null
            }
        } else {
            val unitsString = "(" + units.reduce { acc, s -> "$acc|$s" } + ")"
            val patternG = Pattern.compile("$keyword(\\S*)(\\s*)(\\d*\\.?\\d*)\\s*$unitsString")
            val matcherG = patternG.matcher(line.toLowerCase())
            if (!matcherG.find()) {
                return null
            }

            val valueString = matcherG.group(3) ?: return null
            val unitString = matcherG.group(4) ?: return null

            return valueString.toDoubleOrNull()
        }
    }

    private fun parseServingSize(line: String, keyword: String): OCRServingSize? {
        val patternUnit = Pattern.compile("($keyword)([^(]+)\\(([^)]+)\\)")
        val matcherUnit = patternUnit.matcher(line.toLowerCase())
        if (matcherUnit.find()) {
            val servingQuantityString = matcherUnit.group(2)?.trim() ?: return null
            val nominalQuantityString =
                matcherUnit.group(3)?.trim()?.removePrefix("(") ?: return null
            val indexOfFirstDigit = nominalQuantityString.indexOfFirst { it.isDigit() }
            if (indexOfFirstDigit == -1) {
                return null
            }
            val indexOfLastDigit = nominalQuantityString.indexOfLast { it.isDigit() }
            if (indexOfLastDigit == -1) {
                return null
            }
            val quantityString =
                nominalQuantityString.substring(indexOfFirstDigit, indexOfLastDigit + 1)
            val nominalQuantity = quantityString.toDoubleOrNull()
            val nominalUnit = nominalQuantityString.substring(indexOfLastDigit + 1)
            return OCRServingSize(
                servingQuantityString,
                nominalQuantity,
                nominalUnit
            )
        }

        val patternNoUnit = Pattern.compile("$keyword(\\s\\d*\\s.+)")
        val matcherNoUnit = patternNoUnit.matcher(line.toLowerCase())
        if (matcherNoUnit.find()) {
            val servingQuantityString = matcherNoUnit.group(1)!!.trim()
            return OCRServingSize(
                servingQuantityString,
                null,
                null
            )
        }

        return null
    }

    private val gramUnits = listOf('g', 'o', 's')

    private fun parseGrams(text: String): Double? {
        gramUnits.forEach { c ->
            val indexOfG = text.indexOfLast { it == c }
            if (indexOfG != -1) {
                return text.substring(0, indexOfG).toDoubleOrNull()
            }
        }

        if (text.last() == '9') {
            return text.substring(0, text.length - 1).toDoubleOrNull()
        }

        return null
    }

    private fun parseMilliliters(text: String): Double? {
        if (text.endsWith("ml")) {
            return text.substringBefore("ml").toDouble()
        }

        return null
    }

    private fun distanceBetweenStrings(s1: String, s2: String): Int {
        // memoize only previous line of distance matrix
        var prev = IntArray(s2.length + 1)

        for (j in 0 until s2.length + 1) {
            prev[j] = j
        }

        for (i in 1 until s1.length + 1) {

            // calculate current line of distance matrix
            val curr = IntArray(s2.length + 1)
            curr[0] = i
            for (j in 1 until s2.length + 1) {
                val d1 = prev[j] + 1
                val d2 = curr[j - 1] + 1
                var d3 = prev[j - 1]
                if (s1[i - 1] != s2[j - 1]) {
                    d3 += 1
                }
                curr[j] = min(min(d1, d2), d3)
            }

            // define current line of distance matrix as previous
            prev = curr
        }
        return prev[s2.length]
    }

    private fun determineLanguage(visionText: Text): NutritionFactsDict {
        val engDict = EnglishNutritionFactsDict()
        val freDict = FrenchNutritionFactsDict()

        visionText.textBlocks.forEach { block ->
            block.lines.forEach { line ->
                if (line.text.lowercase() in freDict.getTitle()) {
                    return freDict
                }
            }
        }

        return engDict
    }
}