package ai.passio.passiosdk.core.download

import ai.passio.passiosdk.core.event.PassioEventBus
import ai.passio.passiosdk.core.network.NetworkFileTask
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.SimpleNetworkCallback
import ai.passio.passiosdk.core.utils.PassioLog
import android.content.Context
import android.content.Intent
import android.net.Uri
import androidx.core.app.JobIntentService
import java.io.File

internal class PassioDownloadService : JobIntentService() {

    companion object {
        private const val JOB_ID = 17838

        private const val DOWNLOAD_FOLDER_EXTRA = "downloadFolder"
        private const val BASE_URL_EXTRA = "baseUrl"
        private const val MISSING_FILES_EXTRA = "missingFiles"

        const val PASSIO_BROADCAST_EXTRA_SUCCESS = "passioBroadcastExtraSuccess"
        const val PASSIO_BROADCAST_EXTRA_ERROR = "passioBroadcastExtraError"
        const val PASSIO_BROADCAST_EXTRA_FILES_TO_DOWNLOAD = "passioBroadcastExtraFilesToDownload"

        fun downloadMissingFiles(
            context: Context,
            missingFiles: ArrayList<String>,
            baseUrl: String,
            pathToDownloadFolder: String
        ) {
            val intent = Intent(context, PassioDownloadService::class.java).apply {
                putExtra(BASE_URL_EXTRA, baseUrl)
                putExtra(DOWNLOAD_FOLDER_EXTRA, pathToDownloadFolder)
                putStringArrayListExtra(MISSING_FILES_EXTRA, missingFiles)
            }
            enqueueWork(context, PassioDownloadService::class.java, JOB_ID, intent)
        }

        var isRunning: Boolean = false
    }

    override fun onHandleWork(intent: Intent) {
        isRunning = true
        val baseUrl = intent.getStringExtra(BASE_URL_EXTRA)!!
        val pathToDownloadFolder = intent.getStringExtra(DOWNLOAD_FOLDER_EXTRA)!!

        val missingFiles = intent.getStringArrayListExtra(MISSING_FILES_EXTRA)!!
        downloadFiles(missingFiles, baseUrl, pathToDownloadFolder)
    }

    private fun retryFile(
        fileName: String,
        baseUrl: String,
        pathToDownloadFolder: String,
        currentIndex: Int,
        lastIndex: Int
    ) {
        PassioLog.i(this::class.java.simpleName, "Retrying file: $fileName")
        val outPath = pathToDownloadFolder + File.separator + fileName

        NetworkService.instance.doRequest(
            NetworkFileTask(baseUrl + fileName, outPath),
            object : SimpleNetworkCallback<File?>() {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.e(
                        PassioDownloadService::class.java.simpleName,
                        "File $fileName download error: $message"
                    )
                    sendBroadcastErrorMessage(message)
                    if (currentIndex == lastIndex) {
                        isRunning = false
                    }
                }

                override fun onSuccess(result: File?) {
                    PassioLog.i(
                        PassioDownloadService::class.java.simpleName,
                        "File $fileName successfully downloaded!"
                    )
                    sendBroadcastSuccessMessage(result!!.path)
                    if (currentIndex == lastIndex) {
                        isRunning = false
                    }
                }
            }
        )
    }

    private fun downloadFiles(
        filesToDownload: List<String>,
        baseUrl: String,
        pathToDownloadFolder: String
    ) {
        PassioLog.i(
            this::class.java.simpleName,
            "Downloading files: $filesToDownload"
        )
        sendBroadcastNumOfFilesToDownload(filesToDownload.size)
        for (i in filesToDownload.indices) {
            val fileName = filesToDownload[i]
            val outPath = pathToDownloadFolder + File.separator + fileName
            NetworkService.instance.doRequest(
                NetworkFileTask(baseUrl + fileName, outPath),
                object : SimpleNetworkCallback<File?>() {
                    override fun onFailure(code: Int, message: String) {
                        retryFile(
                            fileName,
                            baseUrl,
                            pathToDownloadFolder,
                            i,
                            filesToDownload.lastIndex
                        )
                    }

                    override fun onSuccess(result: File?) {
                        PassioLog.i(
                            PassioDownloadService::class.java.simpleName,
                            "File $fileName successfully downloaded!"
                        )
                        sendBroadcastSuccessMessage(result!!.path)
                        if (i == filesToDownload.lastIndex) {
                            isRunning = false
                        }
                    }
                }
            )
        }
    }

    private fun sendBroadcastSuccessMessage(fileUri: String) {
        PassioEventBus.getInstance().submitEvent(PASSIO_BROADCAST_EXTRA_SUCCESS, fileUri)
    }

    private fun sendBroadcastErrorMessage(errorMessage: String) {
        PassioEventBus.getInstance().submitEvent(PASSIO_BROADCAST_EXTRA_ERROR, errorMessage)
    }

    private fun sendBroadcastNumOfFilesToDownload(filesToDownload: Int) {
        PassioEventBus.getInstance()
            .submitEvent(PASSIO_BROADCAST_EXTRA_FILES_TO_DOWNLOAD, filesToDownload.toString())
    }
}