package ai.passio.passiosdk.core.os

import android.os.Handler
import android.os.Looper
import android.util.Log
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean

private const val TAG = "ProcessingQueue"

internal class ProcessingQueue(threadName: String) : Thread() {

    private lateinit var handler: Handler
    private lateinit var looper: Looper

    private val queueFinished = AtomicBoolean(false)
    private val queueExiting = AtomicBoolean(false)
    private val startupLatch: CountDownLatch
    private val shutdownLatch: CountDownLatch

    init {
        startupLatch = CountDownLatch(1)
        shutdownLatch = CountDownLatch(1)
        name = threadName
    }

    override fun run() {
        try {
            Looper.prepare()
            looper = Looper.myLooper()!!
            handler = Handler(looper)
            startupLatch.countDown()

            Looper.loop()

            queueFinished.set(true)
            shutdownLatch.countDown()
        } catch (t: Throwable) {
            val mainHandler = Handler(Looper.getMainLooper())
            mainHandler.post {
                throw RuntimeException("Processing queue $name halted due to an error: $t", t)
            }
        }
    }

    private fun waitForStartup() {
        try {
            startupLatch.await()
        } catch (e: InterruptedException) {
            Log.d(TAG, "Interrupted while waiting for process to start")
        }
    }

    fun postJob(runnable: Runnable) {
        waitForStartup()

        if (!queueExiting.get()) {
            handler.post(runnable)
        } else {
            Log.w(TAG, "Processing queue is exiting, unable to post job to it")
        }
    }

    fun postJobDelay(runnable: Runnable, delay: Long) {
        waitForStartup()

        if (!queueExiting.get()) {
            handler.postDelayed(runnable, delay)
        } else {
            Log.w(TAG, "Processing queue is exiting, unable to post job to it")
        }
    }

    fun postShutdownJob() {
        val oldVal = queueExiting.getAndSet(true)
        if (!oldVal) {
            handler.post {
                looper.quitSafely()
            }
        }
    }

    fun waitForShutdown(maxWaitTime: Long) {
        if (!queueFinished.get()) {
            try {
                val queueFinished: Boolean
                if (maxWaitTime > 0) {
                    Log.d(
                        TAG,
                        "Waiting max $maxWaitTime miliseconds for processing queue $name to shutdown..."
                    )
                    queueFinished = shutdownLatch.await(maxWaitTime, TimeUnit.MILLISECONDS)
                } else {
                    Log.d(TAG, "Waiting for processing queue $name to shutdown...")
                    shutdownLatch.await()
                    queueFinished = true
                }
                if (!queueFinished) {
                    Log.e(
                        TAG,
                        "Processing queue $name still not exited (after $maxWaitTime miliseconds wait). Giving up..."
                    )
                } else {
                    Log.d(TAG, "Processing queue $name is now terminated...")
                }
            } catch (e: InterruptedException) {
                Log.e(TAG, "Interrupted while waiting for processing queue $name to shutdown: $e")
            }

        } else {
            Log.d(TAG, "Processing queue $name is already terminated.")
        }
    }
}