package ai.passio.passiosdk.core.utils

import ai.passio.passiosdk.core.file.PassioFileManager
import ai.passio.passiosdk.core.migz.PassioMiGzInputStream
import android.content.Context
import android.content.res.AssetManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.drawable.Drawable
import android.net.Uri
import org.json.JSONObject
import java.io.*
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.MappedByteBuffer
import java.nio.channels.FileChannel
import java.nio.charset.Charset
import java.nio.file.Files
import java.nio.file.Paths

internal object FileUtil {

    @Throws(IOException::class)
    internal fun loadLabelsFromAssets(assetManager: AssetManager, filePath: String): List<String> {
        val labels = mutableListOf<String>()
        val inputStream = assetManager.open(filePath)

        inputStream.bufferedReader().useLines { lines ->
            lines.forEach { line ->
                labels.add(line)
            }
        }

        inputStream.close()
        return labels
    }

    @Throws(IOException::class)
    internal fun loadLabelsFromSystem(filePath: String): List<String> {
        val labels = mutableListOf<String>()
        val file = File(filePath)
        val inputStream = FileInputStream(file)

        inputStream.bufferedReader().useLines { lines ->
            lines.forEach { line ->
                labels.add(line)
            }
        }

        inputStream.close()
        return labels
    }

    @Throws(IOException::class)
    internal fun loadMappedFileFromAssets(
        assetManager: AssetManager,
        fileName: String
    ): MappedByteBuffer? {
        val assetFileDescriptor = assetManager.openFd(fileName)
        val inputStream = FileInputStream(assetFileDescriptor.fileDescriptor)
        return try {
            val startOffset = assetFileDescriptor.startOffset
            val declaredLength = assetFileDescriptor.declaredLength
            inputStream.channel.map(
                FileChannel.MapMode.READ_ONLY,
                startOffset,
                declaredLength
            )
        } catch (e: Exception) {
            PassioLog.e(this::class.java.simpleName, e.message ?: "")
            return null
        } finally {
            inputStream.close()
            assetFileDescriptor.close()
        }
    }

    @Throws(IOException::class)
    fun loadFileFromAssets(assetManager: AssetManager, fileName: String): ByteBuffer {
        val inputStream = assetManager.open(fileName)
        val buffer = ByteArray(1024)
        val outputStream = ByteArrayOutputStream()
        var bytesRead = 0
        while ({ bytesRead = inputStream.read(buffer); bytesRead }() != -1) {
            outputStream.write(buffer, 0, bytesRead)
        }
        inputStream.close()
        return ByteBuffer.wrap(outputStream.toByteArray())
    }

    @Throws(IOException::class)
    fun loadFileFromAssetsAsArray(assetManager: AssetManager, fileName: String): ByteArray {
        val inputStream = assetManager.open(fileName)
        val buffer = ByteArray(1024)
        val outputStream = ByteArrayOutputStream()
        var bytesRead = 0
        while ({ bytesRead = inputStream.read(buffer); bytesRead }() != -1) {
            outputStream.write(buffer, 0, bytesRead)
        }
        inputStream.close()
        return outputStream.toByteArray()
    }


    @Throws(IOException::class)
    internal fun loadMappedFileFromSystem(file: File): MappedByteBuffer {
        val inputStream = FileInputStream(file)
        val startOffset = 0L
        val declaredLength = file.length()
        return inputStream.channel.map(FileChannel.MapMode.READ_ONLY, startOffset, declaredLength)
    }

    @Throws(IOException::class)
    internal fun loadFileFromSystem(file: File): ByteArray {
        val bytes = ByteArray(file.length().toInt())
        val inStream = FileInputStream(file)
        val bufferedInputStream = BufferedInputStream(inStream)
        bufferedInputStream.read(bytes, 0, bytes.size)
        bufferedInputStream.close()
        inStream.close()
        return bytes
    }

    @Throws(IOException::class)
    internal fun loadDrawableFromAssets(assetManager: AssetManager, imageName: String): Drawable? {
        val inputStream = assetManager.open(imageName)
        return try {
            Drawable.createFromStream(inputStream, null)
        } catch (e: Exception) {
            return null
        } finally {
            inputStream.close()
        }
    }

    @Throws(IOException::class)
    fun loadBitmapFromAssets(assetManager: AssetManager, imageName: String): Bitmap {
        val inputStream = assetManager.open(imageName)
        return BitmapFactory.decodeStream(inputStream)
    }

    internal fun fileExistsInAssets(assetManager: AssetManager, filePath: String): Boolean {
        var inputStream: InputStream? = null
        var exists = false
        try {
            inputStream = assetManager.open(filePath)
            exists = true
        } catch (ex: IOException) {
            exists = false
        } finally {
            inputStream?.close()
            return exists
        }
    }

    @Throws(IOException::class)
    internal fun loadTextFileFromAssets(assetManager: AssetManager, fileName: String): String {
        val inputStream: InputStream = assetManager.open(fileName)
        val size = inputStream.available()
        val buffer = ByteArray(size)
        inputStream.read(buffer)
        inputStream.close()
        return String(buffer, Charset.defaultCharset())
    }

    @Throws(IOException::class)
    fun writeMetadataFileToCache(context: Context, fileName: String, metadata: String): Uri {
        val root = File(context.cacheDir.absolutePath + File.separator + "metadata")
        if (!root.exists()) {
            root.mkdirs()
        }
        val fileToWrite = File(root, fileName)
        val writer = FileWriter(fileToWrite)
        writer.append(metadata)
        writer.flush()
        writer.close()
        return Uri.fromFile(fileToWrite)
    }

    @Throws(IOException::class)
    fun writeFileToCache(
        context: Context,
        fileName: String,
        rootFolder: String,
        buffer: ByteBuffer
    ): Uri {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        if (!root.exists()) {
            root.mkdirs()
        }

        val fileToWrite = File(root, fileName)
        val fileChannel = FileOutputStream(fileToWrite).channel
        fileChannel.write(buffer)
        fileChannel.close()
        return Uri.fromFile(fileToWrite)
    }

    @Throws(IOException::class)
    fun writeTextFileToCache(
        context: Context,
        fileName: String,
        rootFolder: String,
        contents: String
    ): Uri {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        if (!root.exists()) {
            root.mkdirs()
        }
        val fileToWrite = File(root, fileName)
        val writer = FileWriter(fileToWrite)
        writer.append(contents)
        writer.flush()
        writer.close()
        return Uri.fromFile(fileToWrite)
    }

    @Throws(IOException::class)
    internal fun loadTextFileFromCache(
        context: Context,
        fileName: String,
        rootFolder: String
    ): String? {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        if (!root.exists()) {
            return null
        }
        val file = File(root, fileName)

        val bytes = ByteArray(file.length().toInt())
        val bufferedInputStream = BufferedInputStream(FileInputStream(file))
        bufferedInputStream.read(bytes, 0, bytes.size)
        bufferedInputStream.close()
        return String(bytes)
    }

    fun getFileFromCache(context: Context, fileName: String, rootFolder: String): Uri? {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        val file = File(root, fileName)
        if (!file.exists()) {
            return null
        }

        return Uri.fromFile(file)
    }

    @Throws(IOException::class)
    fun loadBitmapFromCache(context: Context, imageName: String, rootFolder: String): Bitmap? {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        val file = File(root, imageName)
        if (!file.exists()) {
            return null
        }

        val fis = FileInputStream(file)
        val bitmap = BitmapFactory.decodeStream(fis)
        fis.close()
        return bitmap
    }

    fun deleteRecursive(fileOrDirectory: File) {
        if (fileOrDirectory.isDirectory) {
            fileOrDirectory.listFiles()?.forEach { child ->
                deleteRecursive(child)
            }
        }
        fileOrDirectory.delete()
    }

    fun deleteCacheRecursive(context: Context, rootFolder: String) {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        deleteRecursive(root)
    }

    fun listFilesFromCache(
        context: Context,
        rootFolder: String,
        filter: String? = null
    ): List<String> {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        if (!root.isDirectory) {
            return emptyList()
        }

        val filteredFiles = mutableListOf<String>()
        root.listFiles()?.forEach {
            if (!it.isFile) {
                return@forEach
            }

            if (filter == null) {
                filteredFiles.add(it.name)
            } else {
                if (it.name.contains(filter)) {
                    filteredFiles.add(it.name)
                }
            }
        }

        return filteredFiles
    }

    fun copyFileToCache(srcFile: File, rootFolder: String, rename: String? = null): Uri? {
        val root = File(rootFolder)
        if (!root.exists()) {
            root.mkdirs()
        }

        val dstName = rename ?: srcFile.name
        val dstFile = openTempFile(rootFolder + File.separator + dstName) { tempFile ->
            var inputStream: BufferedInputStream? = null
            var outputStream: BufferedOutputStream? = null

            try {
                inputStream = BufferedInputStream(FileInputStream(srcFile))
                outputStream = BufferedOutputStream(FileOutputStream(tempFile, false))

                val buffer = ByteArray(4048)
                var length = 0

                while ({ length = inputStream.read(buffer); length }() > 0) {
                    outputStream.write(buffer, 0, length)
                }

                inputStream.close()
                outputStream.close()
                true
            } catch (e: IOException) {
                e.printStackTrace()
                inputStream?.close()
                outputStream?.close()
                false
            }

        } ?: return null

        return Uri.fromFile(dstFile)
    }

    fun deleteFile(uri: Uri) {
        val file = File(uri.path!!)
        if (file.exists()) {
            file.delete()
        }
    }

    fun writeFileToPath(parentFolder: String, fileName: String, content: ByteBuffer): Uri {
        val rootFolder = File(parentFolder)
        if (!rootFolder.exists()) {
            rootFolder.mkdirs()
        }
        val fileToWrite = File(rootFolder, fileName)
        val fileChannel = FileOutputStream(fileToWrite).channel
        fileChannel.write(content)
        fileChannel.close()
        return Uri.fromFile(fileToWrite)
    }

    fun writeArrayToPath(parentFolder: String, fileName: String, content: ByteArray): Uri {
        val rootFolder = File(parentFolder)
        if (!rootFolder.exists()) {
            rootFolder.mkdirs()
        }
        val fileToWrite = File(rootFolder, fileName)
        val fos = FileOutputStream(fileToWrite)
        fos.write(content)
        fos.close()
        return Uri.fromFile(fileToWrite)
    }

    @Throws(IOException::class)
    fun writeBitmapToCache(
        context: Context,
        fileName: String,
        rootFolder: String,
        bitmap: Bitmap
    ): Uri? {
        val root = File(context.cacheDir.absolutePath + File.separator + rootFolder)
        if (!root.exists()) {
            root.mkdirs()
        }

        val fileToWrite = File(root, fileName)
        val fileOutputStream = FileOutputStream(fileToWrite)
        if (!bitmap.compress(Bitmap.CompressFormat.PNG, 100, fileOutputStream)) {
            fileOutputStream.close()
            return null
        }
        fileOutputStream.close()
        return Uri.fromFile(fileToWrite)
    }

    @Throws(IOException::class)
    fun listAssetFiles(assetManager: AssetManager, path: String): List<String> {
        val list = assetManager.list(path)
        return if (list?.isNotEmpty() == true) {
            list.toList()
        } else {
            listOf()
        }
    }

    @Throws(IOException::class)
    fun loadJsonFromAssets(assetManager: AssetManager, filePath: String): JSONObject {
        var json = ""
        val inputStream = assetManager.open(filePath)

        inputStream.bufferedReader().useLines { lines ->
            lines.forEach { line ->
                json += "$line\n"
            }
        }

        return JSONObject(json)
    }


    internal fun unzipToArray(byteBuffer: ByteBuffer): ByteArray? {
        val array = byteBuffer.array()
        val inputStream = ByteArrayInputStream(array)
        val zipInputStream =
            PassioMiGzInputStream(inputStream)
        val outputStream = ByteArrayOutputStream()

        return try {
            val buffer = ByteArray(4096)
            var count: Int
            while (zipInputStream.read(buffer).also { count = it } != -1) {
                outputStream.write(buffer, 0, count)
            }
            outputStream.toByteArray()
        } catch (e: IOException) {
            e.printStackTrace()
            null
        } finally {
            zipInputStream.close()
            inputStream.close()
        }
    }

    internal fun unzipToArray(inFile: File): ByteArray? {
        val startTime = System.currentTimeMillis()
        val inputStream = FileInputStream(inFile)
        val zipInputStream =
            PassioMiGzInputStream(inputStream)
        val outputStream = ByteArrayOutputStream()

        try {
            val buffer = ByteArray(4096)
            var count: Int
            while (zipInputStream.read(buffer).also { count = it } != -1) {
                outputStream.write(buffer, 0, count)
            }
            return outputStream.toByteArray()
        } catch (e: IOException) {
            e.printStackTrace()
            return null
        } finally {
            zipInputStream.close()
            inputStream.close()
        }
    }

    internal fun unzipToBuffer(fis: FileInputStream): ByteBuffer? {
        val zipInputStream =
            PassioMiGzInputStream(fis)

        val size = uncompressedSize(fis)
        val fileBuffer = ByteBuffer.allocateDirect(size).apply {
            order(ByteOrder.nativeOrder())
        }

        val readBuffer = ByteArray(256 * 1024)
        var len = 0
        while (zipInputStream.read(readBuffer).also { len = it } > 0) {
            fileBuffer.put(readBuffer, 0, len)
        }

        zipInputStream.close()
        fis.close()
        return fileBuffer
    }

    internal fun unzipToBuffer(file: File): ByteBuffer? {
        val startTime = System.currentTimeMillis()
        val inputStream = FileInputStream(file)
        val zipInputStream =
            PassioMiGzInputStream(inputStream)

        val size = uncompressedSize(file)
        val fileBuffer = ByteBuffer.allocateDirect(size).apply {
            order(ByteOrder.nativeOrder())
        }

        val readBuffer = ByteArray(256 * 1024)
        var len = 0
        while (zipInputStream.read(readBuffer).also { len = it } > 0) {
            fileBuffer.put(readBuffer, 0, len)
        }

        zipInputStream.close()
        inputStream.close()
        return fileBuffer
    }

    internal fun unzipToFile(
        inputStream: InputStream,
        outPath: String
    ): Boolean {
        val startTime = System.currentTimeMillis()
        val zipInputStream =
            PassioMiGzInputStream(inputStream)
        val outputStream = FileOutputStream(outPath)
        try {
            val buffer = ByteArray(256 * 1024)
            var len = 0
            while (zipInputStream.read(buffer).also { len = it } > 0) {
                outputStream.write(buffer, 0, len)
            }
        } catch (e: IOException) {
            e.printStackTrace()
            return false
        } finally {
            outputStream.close()
            zipInputStream.close()
            inputStream.close()
        }
        return true
    }

    fun unzipToFile(inFile: File, outPath: String): Boolean {
        val inputStream = FileInputStream(inFile)
        return unzipToFile(inputStream, outPath)
    }

    private fun uncompressedSize(fis: FileInputStream): Int {
        fis.channel.position(fis.channel.size() - 4)
        val a = ByteArray(4)
        fis.read(a)
        val wrapped = ByteBuffer.wrap(byteArrayOf(a[3], a[2], a[1], a[0])).apply {
            order(ByteOrder.nativeOrder())
        }
        return wrapped.int
    }

    private fun uncompressedSize(inFile: File): Int {
        val raf = RandomAccessFile(inFile, "r")
        raf.seek(raf.length() - 4)
        val b4: Byte = raf.read().toByte()
        val b3: Byte = raf.read().toByte()
        val b2: Byte = raf.read().toByte()
        val b1: Byte = raf.read().toByte()
        val wrapped = ByteBuffer.wrap(byteArrayOf(b1, b2, b3, b4)).apply {
            order(ByteOrder.nativeOrder())
        }
        raf.close()
        return wrapped.int
    }

    fun openTempFile(
        path: String,
        filePopulate: (tmpFile: File) -> Boolean
    ): File? {
        val tmpFile = File("${path}.${PassioFileManager.TEMP_EXTENSION}")
        val success = filePopulate(tmpFile)
        if (!success) {
            return null
        }

        val originalFile = File(path)
        tmpFile.renameTo(originalFile)
        return originalFile
    }

    fun printDirectoryTree(folder: File): String {
        if (!folder.isDirectory) {
            return ""
        }
        val indent = 0
        val sb = StringBuilder()
        printDirectoryTree(folder, indent, sb)
        return sb.toString()
    }

    private fun printDirectoryTree(
        folder: File, indent: Int,
        sb: java.lang.StringBuilder
    ) {
        require(folder.isDirectory) { "folder is not a Directory" }
        sb.append(getIndentString(indent))
        sb.append("+--")
        sb.append(folder.name)
        sb.append("/")
        sb.append("\n")
        for (file in folder.listFiles()) {
            if (file.isDirectory) {
                printDirectoryTree(file, indent + 1, sb)
            } else {
                printFile(file, indent + 1, sb)
            }
        }
    }

    private fun printFile(file: File, indent: Int, sb: java.lang.StringBuilder) {
        sb.append(getIndentString(indent))
        sb.append("+--")
        sb.append(file.name)
        sb.append("\n")
    }

    private fun getIndentString(indent: Int): String? {
        val sb = java.lang.StringBuilder()
        for (i in 0 until indent) {
            sb.append("|  ")
        }
        return sb.toString()
    }

    fun copyFromAssets(context: Context, fileName: String, dstPath: String) {
        var myInput: InputStream? = null
        var myOutput: OutputStream? = null
        try {
            myInput = context.assets.open(fileName)
            myOutput = FileOutputStream(File(dstPath))

            //transfer bytes from the inputfile to the outputfile
            val buffer = ByteArray(4048)
            var length: Int = myInput.read(buffer)

            while (length > 0) {
                myOutput.write(buffer, 0, length)
                length = myInput.read(buffer)
            }

            //Close the streams
            myOutput.flush()
            myOutput.close()
            myInput.close()
        } catch (e: IOException) {
            e.printStackTrace()
        }
    }

    fun copyFile(srcFile: File, dstPath: String): File? {
        val dstFile = openTempFile(dstPath) { tempFile ->
            var inputStream: BufferedInputStream? = null
            var outputStream: BufferedOutputStream? = null

            try {
                inputStream = BufferedInputStream(FileInputStream(srcFile))
                outputStream = BufferedOutputStream(FileOutputStream(tempFile, false))

                val buffer = ByteArray(4048)
                var length = 0

                while ({ length = inputStream.read(buffer); length }() > 0) {
                    outputStream.write(buffer, 0, length)
                }

                inputStream.close()
                outputStream.close()
                true
            } catch (e: IOException) {
                e.printStackTrace()
                inputStream?.close()
                outputStream?.close()
                false
            }

        } ?: return null

        return dstFile
    }

    fun stringFromFile(file: File): String? {
        val streamReader = InputStreamReader(FileInputStream(file))
        val reader = BufferedReader(streamReader)
        val stringBuilder = StringBuilder()

        try {
            var inputLine: String? = null
            while ({ inputLine = reader.readLine(); inputLine }() != null) {
                stringBuilder.append(inputLine)
            }
        } catch (e: Exception) {
            return null
        } finally {
            reader.close()
            streamReader.close()
        }

        return stringBuilder.toString()
    }

    fun renameTo(file: File, newName: String): File? {
        val dstFile = File(file.parent, newName)
        return if (file.renameTo(dstFile)) dstFile else null
    }
}
