package ai.passio.passiosdk.passiofood.data

import ai.passio.passiosdk.core.aes.CryptoHandler
import ai.passio.passiosdk.core.file.PassioFileManager.Companion.getTempDir
import ai.passio.passiosdk.core.sharedpreferences.CorePreferencesManager
import ai.passio.passiosdk.core.utils.FileUtil
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.PassioSDKImpl
import ai.passio.passiosdk.passiofood.config.PassioMetadata
import ai.passio.passiosdk.passiofood.data.model.*
import ai.passio.passiosdk.passiofood.data.model.internal.DietAttribute
import android.annotation.SuppressLint
import android.content.Context
import android.database.Cursor
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteException
import android.database.sqlite.SQLiteOpenHelper
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException

private const val DB_TAG = "PassioDBHelper"

internal const val NO_LABEL = "No Label"

internal class DBHelper(
    context: Context,
    private val DB_NAME: String,
    version: Int,
) : SQLiteOpenHelper(context, DB_NAME, null, version) {

    companion object CONSTANTS {
        private const val BACKGROUND = "BKG0001"
    }

    private val DB_PATH = context.applicationInfo.dataDir + "/databases/"
    private lateinit var database: SQLiteDatabase
    private lateinit var tables: List<String>
    private lateinit var headerMaps: HashMap<String, HashMap<String, Int>>
    private lateinit var modelMapper: ModelMapper

    private var dietPreference = DietAttribute.UNRESTRICTED_DIET

    private val attributesCache = mutableMapOf<PassioID, PassioIDAttributes?>()
    private val foodEntityCache = mutableMapOf<PassioID, PassioFoodItemData?>()
    private val nameCache = mutableMapOf<PassioID, String?>()
    private val childrenCache = mutableMapOf<PassioID, List<PassioAlternative>?>()

    private val imageNameList: List<String> by lazy {
        context.applicationContext.assets.list(
            PassioSDKImpl.FOOD_IMAGE_DIRECTORY.substring(
                0,
                PassioSDKImpl.FOOD_IMAGE_DIRECTORY.length - 1
            )
        )!!.asList()
    }

    private val metadata: PassioMetadata

    init {
        val json = FileUtil.loadTextFileFromAssets(context.assets, "metadata.txt")
        metadata = PassioMetadata(json)
    }

    /**
     * Check if the database already exist to avoid re-copying the file each time you open the application.
     * @return true if it exists, false if it doesn't
     */
    private fun checkDatabase(context: Context): Boolean {
        /**
         * Does not open the database instead checks to see if the file exists
         * also creates the databases directory if it does not exists
         * (the real reason why the database is opened, which appears to result in issues)
         */
        val db = File(context.getDatabasePath(DB_NAME).path)
        if (db.exists()) return true


        // Get the parent (directory in which the database file would be)
        val dbdir: File? = db.parentFile
        // If the directory does not exits then make the directory (and higher level directories)
        if (dbdir == null || !dbdir.exists()) {
            db.parentFile?.mkdirs()
            dbdir?.mkdirs()
        }
        return false
    }

    /**
     * If file is from assets, the filename represents the path to the file
     * in the assets folder. If the fromAssets flag is false, the filename
     * is treated as a path to the file on the system memory.
     */
    constructor(
        context: Context,
        dbName: String,
        fileNameOrPath: String,
        fromAssets: Boolean,
        key1: String,
        key2: String,
        version: Int
    ) : this(context, dbName, version) {
        // Check if database exists
        if (checkDatabase(context)) {
            // Check if database needs to be upgraded
            val currentVersion = CorePreferencesManager(context).getDatabaseVersion()
            // Upgrade the database
            if (currentVersion != version) {
                createDatabase(context, fileNameOrPath, fromAssets, key1, key2, version)
            }
        } else {
            createDatabase(context, fileNameOrPath, fromAssets, key1, key2, version)
        }

        openDatabase()
    }

    private fun createDatabase(
        context: Context,
        fileNameOrPath: String,
        fromAssets: Boolean,
        key1: String,
        key2: String,
        version: Int
    ) {
        val inputStream = if (fromAssets) {
            context.assets.open(fileNameOrPath)
        } else {
            FileInputStream(fileNameOrPath)
        }

        val fileNameComponents = fileNameOrPath.split(".")

        if (fileNameComponents.last() == "passiosecure") {
            val dbBytes = ByteArray(inputStream.available())
            inputStream.read(dbBytes)
            inputStream.close()

            val decryptedDB = CryptoHandler.instance.decrypt(
                dbBytes,
                key1.toByteArray(),
                key2.toByteArray()
            )
            copyDatabase(decryptedDB)
        } else if (fileNameComponents.last() == "passio2") {
            val outPath = DB_PATH + DB_NAME
            FileUtil.unzipToFile(inputStream, outPath)
        } else if (fileNameComponents.last() == "passiosecure2") {
            val fileName = fileNameComponents[fileNameComponents.lastIndex - 2] + "." +
                    fileNameComponents[fileNameComponents.lastIndex - 1] + ".passio2"
            val rootPath = context.getTempDir()
            File(rootPath).mkdirs()
            val outPath = rootPath + File.separator + fileName
            CryptoHandler.instance.decrypt(
                inputStream,
                outPath,
                key1.toByteArray(),
                key2.toByteArray()
            )

            val dbPassio2File = File(outPath)
            val dbBytes = FileUtil.unzipToArray(dbPassio2File)!!
            copyDatabase(dbBytes)
            dbPassio2File.delete()
        } else if (fileNameComponents.last() == "passio") {
            val dstPath = DB_PATH + DB_NAME
            FileUtil.copyFile(File(fileNameOrPath), dstPath)
        } else {
            throw IllegalArgumentException("No known file extension: ${fileNameComponents.last()}")
        }

        CorePreferencesManager(context).setDatabaseVersion(version)
    }

    @Throws(SQLiteException::class)
    private fun openDatabase() {
        val path = DB_PATH + DB_NAME

        database = SQLiteDatabase.openDatabase(path, null, SQLiteDatabase.OPEN_READONLY)
        tables = getTableNames()

        headerMaps = hashMapOf()
        tables.forEach { table ->
            val tableMap = getHeaderMap(table)
            headerMaps[table] = tableMap
        }

        modelMapper = ModelMapper(headerMaps)
    }

    @Synchronized
    override fun close() {
        database.close()
        super.close()
    }

    @Throws(IOException::class)
    private fun copyDatabase(database: ByteArray) {
        val outFilename = DB_PATH + DB_NAME
        val outputStream = FileOutputStream(outFilename)
        outputStream.write(database)

        outputStream.flush()
        outputStream.close()
    }

    override fun onCreate(db: SQLiteDatabase?) {

    }

    override fun onUpgrade(db: SQLiteDatabase?, oldVersion: Int, newVersion: Int) {

    }

    override fun onDowngrade(db: SQLiteDatabase?, oldVersion: Int, newVersion: Int) {

    }

    private fun Cursor.runAndClose(run: (cursor: Cursor) -> Unit) {
        run(this)
        this.close()
    }

    @SuppressLint("Recycle")
    private fun getTableNames(): List<String> {
        val tables = arrayListOf<String>()

        database.rawQuery("SELECT * FROM sqlite_master WHERE type='table'", null).runAndClose {
            while (it.moveToNext()) {
                tables.add(it.getString(1))
            }
            it.close()
        }

        return tables
    }

    @SuppressLint("Recycle")
    private fun getHeaderMap(table: String): HashMap<String, Int> {
        val map = hashMapOf<String, Int>()

        database.rawQuery("PRAGMA table_info($table)", null).runAndClose {
            while (it.moveToNext()) {
                val id = it.getInt(0)
                val name = it.getString(1)
                map[name] = id
            }
            it.close()
        }

        return map
    }

    fun getNameFor(passioID: PassioID): String? {
        if (nameCache.containsKey(passioID)) {
            return nameCache[passioID]
        }

        if (passioID == BACKGROUND) {
            return "background"
        }

        val cursor =
            database.rawQuery("SELECT name FROM data where id = ?", arrayOf(passioID))
        if (!cursor.moveToFirst()) {
            cursor.close()
            nameCache[passioID] = null
            return null
        }

        val name = cursor.getString(0)
        cursor.close()
        nameCache[passioID] = name
        return name
    }

    fun lookupPassioIDFor(name: String): PassioID? {
        return getPassioIdFor(name)
    }

    fun getLabelsForSSD(): List<String> {
        val labels = mutableListOf<String>()
        val cursor =
            database.rawQuery("SELECT * FROM data where visual = 2", arrayOf())

        while (cursor.moveToNext()) {
            labels.add(cursor.getString(headerMaps["data"]!!["id"]!!))
        }
        cursor.close()
        return labels.sorted()
    }

    fun getLabelsForHNN(): List<String> {
        val labels = mutableListOf<String>()
        val cursor =
            database.rawQuery("SELECT * FROM data where visual = 1", arrayOf())

        while (cursor.moveToNext()) {
            labels.add(cursor.getString(headerMaps["data"]!!["id"]!!))
        }
        cursor.close()
        return labels.sorted()
    }

    fun getLabelsForLogo(): List<String> {
        val labels = mutableListOf<String>()
        val cursor =
            database.rawQuery("SELECT * FROM data where visual = 4", arrayOf())

        while (cursor.moveToNext()) {
            labels.add(cursor.getString(headerMaps["data"]!!["id"]!!))
        }
        cursor.close()
        return labels.sorted()
    }

    private fun getPassioIdFor(name: String): PassioID? {
        return sqlGetPassioIDFromFoodEnteties(name) ?: sqlGetPassioIDFrom(name)
    }

    private fun sqlGetPassioIDFromFoodEnteties(name: String): PassioID? {
        val cursor = database.rawQuery("SELECT id FROM data WHERE name == ?", arrayOf(name))

        if (!cursor.moveToFirst()) {
            cursor.close()
            return null
        }

        val passioID = cursor.getString(headerMaps["data"]?.get("id") ?: 0)
        cursor.close()
        return passioID
    }

    private fun sqlGetPassioIDFrom(inputSynonymName: String): PassioID? {
        val cursor = database.rawQuery(
            "SELECT id FROM synonyms WHERE name == ?",
            arrayOf(inputSynonymName)
        )

        if (!cursor.moveToFirst()) {
            cursor.close()
            return null
        }

        val result = cursor.getString(headerMaps["synonyms"]?.get("id") ?: 0)
        cursor.close()
        return result
    }

    fun sqlGetAllFoodItems(): List<String> {
        val result = mutableListOf<String>()
        val cursor =
            database.rawQuery(
                "SELECT name FROM synonyms WHERE display == 1 ORDER BY name",
                null
            )


        while (cursor.moveToNext()) {
            result.add(cursor.getString(0))
        }

        cursor.close()

        return result
    }

    fun sqlGetAllPassioIDs(): Map<PassioID, String> {
        val result = mutableMapOf<PassioID, String>()
        val cursor =
            database.rawQuery(
                "SELECT * FROM synonyms WHERE display == 1 ORDER BY name",
                null
            )


        while (cursor.moveToNext()) {
            val synonym = modelMapper.mapSynonymResult(cursor)
            result[synonym.inputPassioID] = synonym.inputName
        }

        cursor.close()

        return result
    }

    fun sqlGetAllVisuallyRecognizableFoodNames(): List<String> {
        val result = mutableListOf<PassioID>()
        val cursor =
            database.rawQuery("SELECT name FROM data WHERE visual != 6", null)


        while (cursor.moveToNext()) {
            result.add(cursor.getString(0))
        }

        cursor.close()

        return result
    }

    fun getChildren(passioID: PassioID): List<PassioAlternative>? {
        if (childrenCache.containsKey(passioID)) {
            return childrenCache[passioID]
        }

        val cursor =
            database.rawQuery("SELECT children FROM data where id = ?", arrayOf(passioID))
        if (!cursor.moveToFirst()) {
            cursor.close()
            childrenCache[passioID] = null
            return null
        }

        val children = mutableListOf<PassioAlternative>()

        val alternativeResult = modelMapper.mapPassioAlternatives(cursor.getString(0))
        if (alternativeResult == null) {
            cursor.close()
            childrenCache[passioID] = null
            return null
        }

        for (passioAlternativeResult in alternativeResult) {
            val name = getNameFor(passioAlternativeResult.passioID) ?: continue
            children.add(PassioAlternative(passioAlternativeResult, name))
        }

        cursor.close()
        childrenCache[passioID] = children
        return children
    }

    private fun getSiblingsForParents(parents: List<PassioID>?): Set<PassioAlternative>? {
        if (parents == null || parents.isEmpty()) {
            return null
        }

        val siblings = mutableSetOf<PassioAlternative>()

        parents.forEach { parentPassioID ->
            val children = getChildren(parentPassioID) ?: return@forEach
            siblings.addAll(children)
        }

        return siblings
    }

    fun getPassioIDAttributesForName(
        name: String
    ): PassioIDAttributes? {
        val cursor =
            database.rawQuery("SELECT id FROM data WHERE name = ?", arrayOf(name))

        if (!cursor.moveToFirst()) {
            cursor.close()
            return null
        }

        val passioID = cursor.getString(0)
        if (passioID == null) {
            cursor.close()
            return null
        }

        cursor.close()
        return getPassioIDAttributesFor(passioID)
    }

    fun getPassioIDAttributesFor(
        passioID: PassioID,
        number: Double? = null,
        unit: String? = null,
        ignoreCache: Boolean = false
    ): PassioIDAttributes? {
        if (!ignoreCache && attributesCache.containsKey(passioID)) {
            return attributesCache[passioID]?.deepCopy()
        }

        val cursor =
            database.rawQuery("SELECT * FROM data WHERE id = ?", arrayOf(passioID))

        if (!cursor.moveToFirst()) {
            cursor.close()
            return null
        }

        val dataResult = modelMapper.mapFoodItemDataResult(cursor)
        cursor.close()

        with(dataResult) {
            val alternativesParents =
                parents?.map { PassioAlternative(it.passioID, getNameFor(it.passioID)!!) }
            val alternativesChildren =
                children?.map { PassioAlternative(it.passioID, getNameFor(it.passioID)!!) }
            val alternativesSiblings =
                getSiblingsForParents(parents?.map { it.passioID })?.filter { it.passioID != passioID }
                    ?.toList()

            // 1. Check nutrition values
            if (nutritionValuesResult != null) {
                val entityType = if (this.children?.isEmpty() == true) {
                    PassioIDEntityType.group
                } else {
                    PassioIDEntityType.item
                }

                val defaultFoodItem = PassioFoodItemData(
                    this,
                    alternativesChildren,
                    alternativesSiblings,
                    alternativesParents,
                    number,
                    unit,
                )

                val attribute = PassioIDAttributes(
                    passioID,
                    name,
                    entityType,
                    alternativesParents,
                    alternativesChildren,
                    alternativesSiblings,
                    defaultFoodItem,
                    metadata.confusionAlternatives[passioID],
                    metadata.invisibleIngredients[passioID]
                )
                if (!ignoreCache) {
                    attributesCache[passioID] = attribute
                }
                return attribute.deepCopy()
            }
            // 2. Check recipe
            if (recipeConstructResult != null) {
                val ingredients = mutableListOf<PassioFoodItemData>()
                recipeConstructResult.forEach { recipeItem ->
                    val ingredientAttribute =
                        getPassioIDAttributesFor(recipeItem.passioID, ignoreCache = true)
                            ?: return@forEach
                    val ingredient = ingredientAttribute.passioFoodItemData!!
                    ingredient.selectedUnit = recipeItem.unit
                    ingredient.selectedQuantity = recipeItem.number
                    ingredients.add(ingredient)
                }
                val foodRecipe = PassioFoodRecipe(
                    passioID,
                    name,
                    passioServingSizes,
                    passioServingUnits,
                    ingredients,
                    number,
                    unit
                )
                val attribute = PassioIDAttributes(
                    passioID,
                    name,
                    PassioIDEntityType.recipe,
                    alternativesParents,
                    alternativesChildren,
                    alternativesSiblings,
                    foodRecipe,
                    metadata.confusionAlternatives[passioID],
                    metadata.invisibleIngredients[passioID]
                )
                if (!ignoreCache) {
                    attributesCache[passioID] = attribute
                }
                return attribute.deepCopy()
            }
            // 3. Check first child
            val firstChild = children?.first()
            if (firstChild != null) {
                val childAttributes = getPassioIDAttributesFor(
                    firstChild.passioID,
                    number ?: firstChild.number,
                    unit ?: firstChild.unit
                )
                val attribute = if (childAttributes != null) {
                    PassioIDAttributes(
                        childAttributes,
                        passioID,
                        dataResult.name,
                        alternativesChildren,
                        alternativesParents,
                        alternativesSiblings,
                        metadata.confusionAlternatives[passioID],
                        metadata.invisibleIngredients[passioID]
                    )
                } else {
                    null
                }
                if (!ignoreCache) {
                    attributesCache[passioID] = attribute
                }
                return attribute?.deepCopy()
            }
            // 4. Return without nutrition data
            val attribute = PassioIDAttributes(
                passioID,
                name,
                PassioIDEntityType.item,
                alternativesParents,
                alternativesChildren,
                alternativesSiblings,
                null,
                null,
                null,
                metadata.confusionAlternatives[passioID],
                metadata.invisibleIngredients[passioID]
            )
            if (!ignoreCache) {
                attributesCache[passioID] = attribute
            }
            return attribute.deepCopy()
        }
    }

    private val synonyms: Map<String, PassioID> by lazy {
        val cursor =
            database.rawQuery("SELECT * FROM synonyms ORDER BY name", null)
        val result = mutableMapOf<String, PassioID>()

        while (cursor.moveToNext()) {
            val passioID = cursor.getString(0)
            val synonym = cursor.getString(1)
            result[synonym] = passioID
        }
        cursor.close()
        result
    }

    fun getFoodNamesWithPassioIDs(
        byText: String,
        regex: Regex? = null,
        replaceWith: String? = null
    ): List<Pair<PassioID, String>> {
        val ngrams = mutableListOf<String>()

        val sanitizedQuery = regex?.replace(byText, replaceWith!!) ?: byText

        if (byText.length < 5) {
            for (i in 0 until sanitizedQuery.length - 1) {
                ngrams.add(sanitizedQuery.substring(i, i + 2))
            }
        } else if (sanitizedQuery.length < 8) {
            for (i in 0 until sanitizedQuery.length - 2) {
                ngrams.add(sanitizedQuery.substring(i, i + 3))
            }
        } else if (sanitizedQuery.length < 13) {
            for (i in 0 until sanitizedQuery.length - 3) {
                ngrams.add(sanitizedQuery.substring(i, i + 4))
            }
        } else {
            for (i in 0 until sanitizedQuery.length - 4) {
                ngrams.add(sanitizedQuery.substring(i, i + 5))
            }
        }

        val result = mutableListOf<Pair<PassioID, String>>()

        synonyms.iterator().forEach loop@ { synonym ->
            ngrams.forEach { ngram ->
                val sanitizedSynonym =
                    regex?.replace(synonym.key, replaceWith!!) ?: synonym.key
                if (sanitizedSynonym.contains(ngram)) {
                    result.add(synonym.value to synonym.key)
                    return@loop
                }
            }
        }
        return result
    }

    fun getSynonymsForNames(names: List<String>): List<Pair<PassioID, String>> {
        return names.map { name ->
            synonyms[name]!! to name
        }
    }
}