package ai.passio.passiosdk.passiofood.data

import ai.passio.passiosdk.core.utils.getDoubleOptional
import ai.passio.passiosdk.core.utils.getStringOptional
import ai.passio.passiosdk.passiofood.data.measurement.Grams
import ai.passio.passiosdk.passiofood.data.measurement.UnitMass
import ai.passio.passiosdk.passiofood.data.model.PassioFoodOrigin
import ai.passio.passiosdk.passiofood.data.model.PassioIDEntityType
import ai.passio.passiosdk.passiofood.data.model.PassioServingSize
import ai.passio.passiosdk.passiofood.data.model.PassioServingUnit
import ai.passio.passiosdk.passiofood.data.model.internal.CondimentResult
import ai.passio.passiosdk.passiofood.data.model.internal.FoodEntityResult
import ai.passio.passiosdk.passiofood.data.model.internal.FoodItemDataResult
import ai.passio.passiosdk.passiofood.data.model.internal.FoodItemsResult
import ai.passio.passiosdk.passiofood.data.model.internal.NutritionValuesResult
import ai.passio.passiosdk.passiofood.data.model.internal.PassioAlternativeResult
import ai.passio.passiosdk.passiofood.data.model.internal.RecipeConstructorResult
import ai.passio.passiosdk.passiofood.data.model.internal.SynonymResult
import android.database.Cursor
import org.json.JSONArray
import org.json.JSONObject

internal class ModelMapper(private val headerMaps: HashMap<String, HashMap<String, Int>>) {

    fun mapFoodEntityResult(cursor: Cursor): FoodEntityResult {
        val foodEntityMap = headerMaps["foodEntities"]!!

        return FoodEntityResult(
            cursor.getString(foodEntityMap["passioID"]!!),
            cursor.getString(foodEntityMap["name"]!!),
            cursor.getString(foodEntityMap["filename_icon"]!!),
            PassioIDEntityType.fromString(
                cursor.getString(
                    foodEntityMap["entity_type"]!!
                )
            )
        )
    }

    fun mapCondimentResult(cursor: Cursor): CondimentResult {
        val condimentMap = headerMaps["foodCondiments"]!!

        return CondimentResult(
            cursor.getString(condimentMap["condiment"]!!),
            cursor.getDouble(condimentMap["number"]!!),
            cursor.getString(condimentMap["unit"]!!).emptyToNull(),
            cursor.getString(condimentMap["size"]!!).emptyToNull()
        )
    }

    fun mapFoodItemsResult(cursor: Cursor): FoodItemsResult {
        val foodItemsMap = headerMaps["foodItems"]!!

        return FoodItemsResult(
            cursor.getString(foodItemsMap["passioID"]!!),
            cursor.getString(foodItemsMap["display_name"]!!),
            cursor.getString(foodItemsMap["filename_icon"]!!),
            cursor.getString(foodItemsMap["nutritionID"]!!),
            cursor.getDouble(foodItemsMap["default_number"]!!),
            cursor.getString(foodItemsMap["default_unit"]!!).emptyToNull(),
            cursor.getString(foodItemsMap["default_size"]!!).emptyToNull()
        )
    }

    fun mapSynonymResult(cursor: Cursor): SynonymResult {
        val synonymMap = headerMaps["synonyms"]!!

        return SynonymResult(
            cursor.getString(synonymMap["id"]!!),
            cursor.getString(synonymMap["name"]!!)
        )
    }

    fun mapFoodItemDataResult(cursor: Cursor): FoodItemDataResult {
        val foodItemDataMap = headerMaps["data"]!!
        val originsColumn = foodItemDataMap["origins"]

        val licenseCopy = cursor.getStringOptional(foodItemDataMap["licensecopy"])
        val origins = if (originsColumn != null) {
            mapFoodItemOrigins(cursor.getString(originsColumn), licenseCopy)
        } else {
            null
        }

        return FoodItemDataResult(
            cursor.getString(foodItemDataMap["id"]!!),
            cursor.getString(foodItemDataMap["name"]!!),
            cursor.getInt(foodItemDataMap["visual"]!!),
            mapPassioAlternatives(cursor.getString(foodItemDataMap["children"]!!)),
            mapPassioAlternatives(cursor.getString(foodItemDataMap["parents"]!!)),
            mapNutritionValuesResult(cursor.getString(foodItemDataMap["nutrition"]!!)),
            mapServingUnits(cursor.getString(foodItemDataMap["units"]!!)),
            mapServingSizes(cursor.getString(foodItemDataMap["servings"]!!)),
            cursor.getStringOptional(foodItemDataMap["product"]!!),
            mapRecipeConstruct(cursor.getString(foodItemDataMap["recipe"]!!)),
            origins
        )
    }

    private fun mapRecipeConstruct(json: String?): List<RecipeConstructorResult>? {
        if (json == null) {
            return null
        }
        val recipeItems = mutableListOf<RecipeConstructorResult>()
        val jArray = JSONArray(json)
        for (i in 0 until jArray.length()) {
            val jsonObject = jArray.getJSONObject(i)
            val passioID = jsonObject.getString("id")
            val number = jsonObject.getDouble("number")
            val unit = jsonObject.getString("unit")
            val recipeItem = RecipeConstructorResult(passioID, number, unit)
            recipeItems.add(recipeItem)
        }
        return recipeItems
    }

    private fun mapServingSizes(json: String?): List<PassioServingSize>? {
        if (json == null) {
            return null
        }
        val servingSizes = mutableListOf<PassioServingSize>()
        val jArray = JSONArray(json)
        if (jArray.length() == 0) {
            return null
        }

        for (i in 0 until jArray.length()) {
            val jsonObject = jArray.getJSONObject(i)
            val number = jsonObject.getDouble("number")
            val unit = jsonObject.getString("unit")
            servingSizes.add(PassioServingSize(number, unit))
        }
        return servingSizes
    }

    private fun mapServingUnits(json: String?): List<PassioServingUnit>? {
        if (json.isNullOrEmpty()) {
            return null
        }
        val servingUnits = mutableListOf<PassioServingUnit>()
        val jArray = JSONArray(json)
        if (jArray.length() == 0) {
            return null
        }


        for (i in 0 until jArray.length()) {
            val item = jArray.getJSONObject(i)
            val unit = item.getString("unit")
            val weight = item.getDouble("weight")
            servingUnits.add(PassioServingUnit(unit, UnitMass(Grams, weight)))

        }
        return servingUnits
    }

    private fun mapNutritionValuesResult(json: String?): NutritionValuesResult? {
        if (json == null) {
            return null
        }
        val jsonObject = JSONObject(json)
        val cal = jsonObject.getDoubleOptional("calories")
        val carbs = jsonObject.getDoubleOptional("carbs")
        val fat = jsonObject.getDoubleOptional("fat")
        val protein = jsonObject.getDoubleOptional("protein")
        val satFat = jsonObject.getDoubleOptional("satFat")
        val monoSatFat = jsonObject.getDoubleOptional("monounsaturatedFat")
        val polySatFat = jsonObject.getDoubleOptional("polyunsaturatedFat")
        val calcium = jsonObject.getDoubleOptional("calcium")
        val potassium = jsonObject.getDoubleOptional("potassium")
        val sodium = jsonObject.getDoubleOptional("sodium")
        val iron = jsonObject.getDoubleOptional("iron")
        val vitaminC = jsonObject.getDoubleOptional("vitaminC")
        val vitaminA = jsonObject.getDoubleOptional("vitaminA")
        val cholesterol = jsonObject.getDoubleOptional("cholesterol")
        val fiber = jsonObject.getDoubleOptional("fiber")
        val transFat = jsonObject.getDoubleOptional("transFat")
        val sugars =
            jsonObject.getDoubleOptional("sugars") ?: jsonObject.getDoubleOptional("sugarTotal")
        val alcohol = jsonObject.getDoubleOptional("alcohol")
        val sugarAdded = jsonObject.getDoubleOptional("sugarAdded")
        val vitaminD = jsonObject.getDoubleOptional("vitaminD")
        val sugarAlcohol = jsonObject.getDoubleOptional("sugarAlcohol")
        val vitaminB12Added = jsonObject.getDoubleOptional("vitaminB12Added")
        val vitaminB12 = jsonObject.getDoubleOptional("vitaminB12")
        val vitaminB6 = jsonObject.getDoubleOptional("vitaminB6")
        val vitaminE = jsonObject.getDoubleOptional("vitaminE")
        val vitaminEAdded = jsonObject.getDoubleOptional("vitaminEAdded")
        val magnesium = jsonObject.getDoubleOptional("magnesium")
        val phosphorus = jsonObject.getDoubleOptional("phosphorus")
        val iodine = jsonObject.getDoubleOptional("iodine")

        return NutritionValuesResult(
            100,
            cal,
            fat,
            satFat,
            monoSatFat,
            polySatFat,
            cholesterol,
            sodium,
            carbs,
            protein,
            iron,
            vitaminC,
            vitaminA,
            calcium,
            potassium,
            fiber,
            transFat,
            sugars,
            alcohol,
            sugarAdded,
            vitaminD,
            sugarAlcohol,
            vitaminB12Added,
            vitaminB12,
            vitaminB6,
            vitaminE,
            vitaminEAdded,
            magnesium,
            phosphorus,
            iodine
        )
    }

    fun mapPassioAlternatives(json: String?): List<PassioAlternativeResult>? {
        if (json == null) {
            return null
        }

        val alternatives = mutableListOf<PassioAlternativeResult>()
        val jArray = JSONArray(json)
        for (i in 0 until jArray.length()) {
            val item = jArray.getJSONObject(i)
            val passioID = item.getString("id")
            val number = item.getDoubleOptional("number")
            val unit = item.getStringOptional("unit")
            alternatives.add(PassioAlternativeResult(passioID, number, unit))
        }
        return alternatives
    }

    private fun mapFoodItemOrigins(
        json: String?,
        licenseCopy: String?
    ): List<PassioFoodOrigin>? {
        if (json == null) return null
        val origins = mutableListOf<PassioFoodOrigin>()

        val jsonArray = JSONArray(json)
        for (i in 0 until jsonArray.length()) {
            val jsonObject = jsonArray.getJSONObject(i)
            val id = jsonObject.getString("id")
            val source = jsonObject.getString("source")
            if (source == "openfood") {
                origins.add(PassioFoodOrigin(id, source, licenseCopy))
            } else {
                origins.add(PassioFoodOrigin(id, source, null))
            }
        }
        return origins
    }

    fun mapChildren(cursor: Cursor): List<PassioAlternativeResult>? {
        val foodDataMap = headerMaps["foodData"]!!
        val childrenJson = cursor.getString(foodDataMap["children"]!!) ?: return null

        return mapPassioAlternatives(childrenJson)
    }

    private fun Cursor.getDoubleOptional(columnIndex: Int?): Double? {
        if (columnIndex == null) {
            return null
        }

        if (this.isNull(columnIndex)) {
            return null
        }

        return this.getDouble(columnIndex)
    }

    private fun Cursor.getStringOptional(columnIndex: Int?): String? {
        if (columnIndex == null) {
            return null
        }

        if (this.isNull(columnIndex)) {
            return null
        }

        return this.getString(columnIndex)
    }

    private fun String.emptyToNull(): String? {
        return if (this.isNotEmpty()) {
            this
        } else {
            null
        }
    }
}