package ai.passio.passiosdk.passiofood.data.model

import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.data.measurement.Grams
import ai.passio.passiosdk.passiofood.data.measurement.UnitMass

private const val SERVING_UNIT_NAME = "serving"

/**
 * Data class the holds the nutritional information of a food
 * item that is classified as a recipe. A recipe has multiple
 * ingredients, but their quantity is controlled by a single
 * [selectedUnit] and [selectedQuantity].
 *
 * @property passioID identifier of food item.
 * @property name recipe name.
 * @property filenameIcon icon name.
 * @property foodItems list of ingredients of the recipe.
 * @property servingSizes list of predefined serving sizes.
 * @property servingUnits list of predefined serving units.
 * @property selectedQuantity currently selected quantity.
 * @property selectedUnit currently selected unit.
 */
internal data class PassioFoodRecipe(
    var passioID: PassioID,
    var name: String,
    private val _Passio_servingSizes: List<PassioServingSize>?,
    private val _Passio_servingUnits: List<PassioServingUnit>?,
    var foodItems: MutableList<PassioFoodItemData>,
    private val preselectedQuantity: Double? = null,
    private val preselectedUnit: String? = null,
    private val gramName: String = Grams.unitName
) {
    var servingSizes: List<PassioServingSize>
    var servingUnits: List<PassioServingUnit>
    var selectedUnit: String
    var selectedQuantity: Double

    init {
        val preselectedServingSize = if (preselectedQuantity != null && preselectedUnit != null) {
            PassioServingSize(preselectedQuantity, preselectedUnit)
        } else {
            null
        }
        var ingredientWeight = 0.0
        if (foodItems.isNotEmpty()) {
            ingredientWeight =
                foodItems.map { it.computedWeight().gramsValue() }.reduce { acc, d -> acc + d }
        }

        val tempServingUnits = mutableListOf<PassioServingUnit>()
        if (_Passio_servingUnits != null) {
            tempServingUnits.addAll(_Passio_servingUnits)
        } else {
            tempServingUnits.add(
                PassioServingUnit(
                    SERVING_UNIT_NAME,
                    UnitMass(Grams, ingredientWeight)
                )
            )
            tempServingUnits.add(PassioServingUnit(gramName, UnitMass(Grams, 1.0)))
        }
        servingUnits = tempServingUnits

        val servingSizes = (if (_Passio_servingSizes == null && _Passio_servingUnits == null) {
            listOf(
                PassioServingSize(1.0, SERVING_UNIT_NAME),
                PassioServingSize(
                    ingredientWeight, gramName
                )
            )
        } else if (_Passio_servingSizes == null && _Passio_servingUnits != null) {
            listOf(PassioServingSize(1.0, _Passio_servingUnits[0].unitName))
        } else {
            _Passio_servingSizes!!
        }).toMutableList()

        if (preselectedServingSize != null) {
            servingSizes.add(0, preselectedServingSize)
        }

        this.servingSizes = servingSizes
        selectedUnit = servingSizes.first().unitName
        selectedQuantity = servingSizes.first().quantity

        computeNewAmountNumbersForIngredients()
    }

    /**
     * Returns true if this food item originates from the
     * open food database.
     *
     * @see [open food link](https://world.openfoodfacts.org/)
     */
    fun isOpenFood(): Boolean {
        foodItems.forEach { foodItem ->
            if (foodItem.isOpenFood()) {
                return true
            }
        }
        return false
    }

    private fun computeNewAmountNumbersForIngredients() {
        if (foodItems.isEmpty()) {
            return
        }

        val totalWeight =
            foodItems.map { it.computedWeight().gramsValue() }.reduce { acc, d -> acc + d }
        val ratioMultiply = computedWeight().gramsValue() / totalWeight
        foodItems.forEach {
            it.selectedQuantity *= ratioMultiply
        }
    }

    private fun computedWeight(): UnitMass {
        val servingUnit = servingUnits.firstOrNull { it.unitName == selectedUnit }
        return if (servingUnit == null) {
            UnitMass(Grams, 0.0)
        } else {
            UnitMass(Grams, servingUnit.weight.gramsValue() * selectedQuantity)
        }
    }

    internal fun deepCopy(): PassioFoodRecipe {
        return PassioFoodRecipe(
            passioID,
            name,
            _Passio_servingSizes,
            _Passio_servingUnits,
            foodItems.map { it.deepCopy() }.toMutableList(),
            preselectedQuantity,
            preselectedUnit
        )
    }
}