package ai.passio.passiosdk.passiofood.search

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.data.DataUtils
import ai.passio.passiosdk.passiofood.token.TokenUsageTracker
import android.net.Uri
import android.util.Log

internal object SearchService {

    fun fetchSearchResult(
        term: String,
        dev: Boolean,
        languageCode: String? = null,
        callback: (response: SearchResponse?) -> Unit
    ) {
        fetchToken(dev) { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            searchWithToken(token, term, dev, languageCode, callback)
        }
    }

    private fun fetchToken(
        dev: Boolean = false,
        callback: (token: String?) -> Unit
    ) {
        TokenService.getInstance().getToken(dev,
            { token ->
                callback(token)
            }, { _ ->
                callback(null)
            }
        )
    }

    private fun searchWithToken(
        token: String,
        term: String,
        dev: Boolean,
        languageCode: String?,
        callback: (response: SearchResponse?) -> Unit
    ) {
        val encoded = Uri.encode(term)
        val url = NativeUtils.instance.nativeGetSearchURL(dev)
        val headers = mutableListOf("Authorization" to token)
        if (languageCode != null) {
            headers.add("Localization-ISO" to languageCode)
        }
        val task = NetworkStringTask("$url?term=$encoded", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    SearchService::class.java.simpleName,
                    "Could not fetch search result: $message"
                )
                callback(null)
            }

            override fun onTokenExpired() {
                fetchSearchResult(term, dev, languageCode, callback)
            }

            override fun onSuccess(result: String) {
                val response = SearchResponse(result)
                callback(response)
            }
        }, "searchForFood")
    }

    fun fetchFoodItem(
        labelId: String,
        type: String,
        resultId: String,
        useShortName: Boolean,
        dev: Boolean,
        languageCode: String?,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        fetchToken(false) { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            foodItemWithToken(
                token,
                labelId,
                type,
                resultId,
                useShortName,
                dev,
                languageCode,
                callback
            )
        }
    }

    private fun foodItemWithToken(
        token: String,
        labelId: String,
        type: String,
        resultId: String,
        useShortName: Boolean,
        dev: Boolean,
        languageCode: String?,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val encoded = DataUtils.encodeMetadata(useShortName)
        val url = NativeUtils.instance.nativeGetFoodFetchURL(dev)
        val headers = mutableListOf("Authorization" to token)
        if (languageCode != null) {
            headers.add("Localization-ISO" to languageCode)
        }
        val task = NetworkStringTask("$url$labelId/$type/$resultId?metadata=$encoded", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                callback(null)
            }

            override fun onTokenExpired() {
                fetchFoodItem(
                    labelId,
                    type,
                    resultId,
                    useShortName,
                    dev,
                    languageCode,
                    callback
                )
            }

            override fun onSuccess(result: String) {
                val response = ResponseFood(result)
                val responseItem = response.results.first()
                val foodItem = PassioFoodItem.fromSearchResponse(responseItem, useShortName)
                callback(foodItem)
            }
        }, "fetchFoodItemForDataInfo")
    }

    fun fetchFoodItemForRefCode(
        refCode: String,
        dev: Boolean,
        languageCode: String?,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            fetchFoodItemForRefCodeInternal(token, refCode, dev, languageCode, callback)
        }
    }

    private fun fetchFoodItemForRefCodeInternal(
        token: String,
        refCode: String,
        dev: Boolean,
        languageCode: String?,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val url = NativeUtils.instance.nativeGetRefCodeURL(dev)
        val headers = mutableListOf("Authorization" to token)
        if (languageCode != null) {
            headers.add("Localization-ISO" to languageCode)
        }
        val task = NetworkStringTask("$url$refCode", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                callback(null)
            }

            override fun onTokenExpired() {
                fetchFoodItemForRefCode(refCode, dev, languageCode, callback)
            }

            override fun onSuccess(result: String) {
                val response = ResponseFood(result)
                val responseItem = response.results.first()
                val shortName = DataUtils.metadataHasShortName(refCode)
                val foodItem = PassioFoodItem.fromSearchResponse(responseItem, shortName)
                callback(foodItem)
            }
        }, "fetchFoodItemForRefCode")
    }
}