package ai.passio.passiosdk.core.authentication

import ai.passio.passiosdk.core.aes.CryptoHandler
import ai.passio.passiosdk.core.aes.hexToByteArray
import ai.passio.passiosdk.core.aes.toHexUpper
import ai.passio.passiosdk.core.network.NetworkBinaryTask
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.SimpleNetworkCallback
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.sharedpreferences.CorePreferencesManager
import ai.passio.passiosdk.core.utils.PassioLog
import android.content.Context
import org.json.JSONArray
import org.json.JSONException
import java.io.IOException
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.concurrent.TimeUnit
import javax.crypto.BadPaddingException

internal class AuthenticationService(
    context: Context,
    private val allowInternet: Boolean
) {

    interface AuthenticationListener {
        fun onAuthorized()
        fun onError(type: Error, message: String)
    }

    companion object {
        const val KEY_LENGTH_LEGACY = 44
        const val KEY_LENGTH = 40
        private const val LOCAL_LICENSE_NAME = "PassioSDK.passiosecure"
    }

    enum class Error {
        TOKEN_FETCH_ERROR,
        KEY_LENGTH_ERROR,
        DECODING_ERROR,
        NETWORK_ERROR,
        KEY_EXPIRED
    }

    private val sharedPrefs = CorePreferencesManager(context)

    val dateFormat = SimpleDateFormat("yyyy-MM-dd", Locale.US)
    private lateinit var developerKey: String
    private lateinit var authenticationListener: AuthenticationListener
    private var currentConfiguration: LicenseConfiguration? = null

    fun checkLicense(
        context: Context,
        key: String,
        metadata: String,
        dev: Boolean,
        authenticationListener: AuthenticationListener
    ) {
        if (key.length == KEY_LENGTH_LEGACY) {
            checkLicenseLegacy(context, key, metadata, dev, authenticationListener)
        } else if (key.length == KEY_LENGTH) {
            checkLicenseToken(context, key, dev, authenticationListener)
        } else {
            authenticationListener.onError(Error.KEY_LENGTH_ERROR, "Key length is wrong")
        }
    }

    private fun checkLicenseToken(
        context: Context,
        key: String,
        dev: Boolean,
        authenticationListener: AuthenticationListener
    ) {
        TokenService.create(context, key)
        TokenService.getInstance().getToken(dev, { _ ->
            authenticationListener.onAuthorized()
        }, { errorMessage ->
            authenticationListener.onError(
                Error.TOKEN_FETCH_ERROR,
                "Could not fetch token: $errorMessage"
            )
        })
    }

    private fun checkLicenseLegacy(
        context: Context,
        key: String,
        metadata: String,
        dev: Boolean,
        authenticationListener: AuthenticationListener
    ) {
        this.developerKey = key
        this.authenticationListener = authenticationListener
        TokenService.create(context, developerKey)

        if (checkLocalLicense(context)) {
            return
        }

        val url = "${NativeUtils.instance.nativeGetLicenseURL(dev)}$key.license"
        val storedConfig = sharedPrefs.getLicense(key)

        if (storedConfig == null) {
            tryFetchLicense(url, metadata)
            return
        }

        currentConfiguration = decryptLicense(storedConfig.first.hexToByteArray())
        if (currentConfiguration == null) {
            PassioLog.e(
                this::class.java.simpleName,
                "PassioConfigurations Decoding error"
            )
            sharedPrefs.removeLicense()
            tryFetchLicense(url, metadata)
            return
        }

        val expirationDate = dateFormat.parse(currentConfiguration!!.until)!!.time
        PassioLog.i(
            this::class.java.simpleName,
            "License expiration date: ${dateFormat.format(Date(expirationDate))}"
        )
        if (expirationDate < System.currentTimeMillis()) {
            PassioLog.e(
                this::class.java.simpleName,
                "License expired, please renew it"
            )
            tryFetchLicense(url, metadata)
            return
        }

        val checkDate = storedConfig.second
        PassioLog.i(
            this::class.java.simpleName,
            "License saved date: ${dateFormat.format(Date(checkDate))}"
        )
        if (System.currentTimeMillis() - checkDate <
            TimeUnit.DAYS.toMillis(currentConfiguration!!.daysToCheckLicense.toLong())
        ) {
            PassioLog.i(
                this::class.java.simpleName,
                "License is valid!"
            )
            authenticationListener.onAuthorized()
            return
        }

        tryFetchLicense(url, metadata)
    }

    private fun fetchLicense(
        url: String,
        metadata: String
    ) {
        PassioLog.i(this::class.java.simpleName, "Fetch license")
        NetworkService.instance.doRequest(
            NetworkBinaryTask(url, metadata),
            downloadCallback
        )
    }

    private val downloadCallback = object : SimpleNetworkCallback<ByteArray>() {

        override fun onFailure(code: Int, message: String) {
            PassioLog.e(
                this::class.java.simpleName,
                "Could not fetch license: $message"
            )
            callbackOldLicense(Error.NETWORK_ERROR, "Could not fetch license: $message")
        }

        override fun onSuccess(result: ByteArray) {
            onLicenseDownloaded(result)
        }
    }

    private fun decryptLicense(encryptedLicense: ByteArray): LicenseConfiguration? {
        val keys = NativeUtils.instance.nativeGetPassioKeys(developerKey)
        val keyHex = keys[0]
        val ivHex = keys[1]

        return try {
            val decryptedBytes =
                CryptoHandler.instance.decrypt(
                    encryptedLicense,
                    keyHex.toByteArray(),
                    ivHex.toByteArray()
                )
            val license = String(decryptedBytes)
            val licenseJson = JSONArray(license)
            LicenseConfiguration(licenseJson[0].toString())
        } catch (e: JSONException) {
            null
        } catch (e: BadPaddingException) {
            null
        }
    }

    private fun onLicenseDownloaded(encryptedLicense: ByteArray) {
        val passioConfiguration = decryptLicense(encryptedLicense)
        if (passioConfiguration == null) {
            authenticationListener.onError(Error.DECODING_ERROR, "License Decoding error")
            return
        }

        PassioLog.i(
            this::class.java.simpleName,
            "Key Expiration Date = ${passioConfiguration.until}"
        )
        val expirationDate = dateFormat.parse(passioConfiguration.until)!!.time
        if (expirationDate < System.currentTimeMillis()) {
            PassioLog.e(
                this::class.java.simpleName,
                "License expired, please renew it"
            )
            callbackOldLicense(
                Error.KEY_EXPIRED,
                "Your Key has expired on: ${passioConfiguration.until}"
            )
            return
        }

        sharedPrefs.setLicense(developerKey, encryptedLicense.toHexUpper())
        PassioLog.i(
            AuthenticationService::class.java.simpleName,
            "License is valid"
        )
        authenticationListener.onAuthorized()
    }

    private fun callbackOldLicense(type: Error, errorMessage: String) {
        if (currentConfiguration != null) {
            val expirationDate = dateFormat.parse(currentConfiguration!!.until)!!.time
            if (expirationDate < System.currentTimeMillis()) {
                authenticationListener.onError(type, errorMessage)
                return
            } else {
                PassioLog.i(
                    this::class.java.simpleName,
                    "Proceeding with the old license"
                )
                authenticationListener.onAuthorized()
            }
        } else {
            authenticationListener.onError(type, errorMessage)
        }
    }

    private fun tryFetchLicense(url: String, metadata: String) {
        if (!allowInternet) {
            authenticationListener.onError(
                Error.NETWORK_ERROR,
                "Could not fetch license, networking not allowed."
            )
        } else {
            fetchLicense(url, metadata)
        }
    }

    private fun checkLocalLicense(context: Context): Boolean {
        return try {
            val licenseIS = context.assets.open(LOCAL_LICENSE_NAME)
            val licenseBytes = ByteArray(licenseIS.available())
            licenseIS.read(licenseBytes)
            licenseIS.close()
            onLicenseDownloaded(licenseBytes)
            true
        } catch (e: IOException) {
            PassioLog.d(this::class.java.simpleName, "No local license found")
            false
        }
    }
}