package ai.passio.passiosdk.passiofood.time

import ai.passio.passiosdk.passiofood.ClassificationCandidate
import ai.passio.passiosdk.passiofood.DebugCandidate
import ai.passio.passiosdk.passiofood.ObjectDetectionCandidate
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.metadata.MetadataManager
import ai.passio.passiosdk.passiofood.voting.HNNKNNVotingLayer
import android.graphics.Bitmap
import android.graphics.RectF
import android.util.Log
import java.lang.ref.WeakReference
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.pow
import kotlin.math.sqrt

internal class VotingFrameAveraging {

    private data class RecognitionResult(
        val objectDetectionCandidate: ObjectDetectionCandidate,
        val hnnCandidate: ClassificationCandidate,
        val knnCandidate: ClassificationCandidate,
        val bitmapRef: WeakReference<Bitmap>,
        val frame: Long
    )

    private var frames: Int = 2
    private var currentFrame: Long = 0L
    private val queue = mutableListOf<RecognitionResult>()

    private val votingLayer = HNNKNNVotingLayer()

    fun setFramesToAnalyze(frames: Int) {
        this.frames = frames
    }

    fun onResult(
        objectDetectionCandidate: ObjectDetectionCandidate,
        hnnCandidate: ClassificationCandidate,
        knnCandidate: ClassificationCandidate,
        bitmap: Bitmap
    ) {
        queue.add(
            RecognitionResult(
                objectDetectionCandidate,
                hnnCandidate,
                knnCandidate,
                WeakReference(bitmap),
                currentFrame
            )
        )
    }

    internal fun onFrameEnd(labelManager: MetadataManager): List<DebugCandidate> {
        Log.i(VotingFrameAveraging::class.java.simpleName, "On Frame End")
        queue.removeAll { currentFrame - it.frame >= frames }
        val result = mutableMapOf<PassioID, DebugCandidate>()

        val odResults = queue.map { it.objectDetectionCandidate }
        val hnnResults = queue.map { it.hnnCandidate }
        val bitmapMap = queue.map { it.hnnCandidate to it.bitmapRef }.toMap()
        val knnResults = queue.map { it.knnCandidate }

        odResults.forEach { odResult ->
            hnnResults.forEach { hnnResult ->
                loop@ for (knnResult in knnResults) {
                    val votedCandidate =
                        votingLayer.vote(odResult, hnnResult, knnResult) ?: continue@loop
                    val label =
                        labelManager.getMappedPassioID(votedCandidate.passioID) ?: continue@loop
                    result[votedCandidate.passioID] = DebugCandidate(
                        label.first,
                        label.second,
                        votedCandidate,
                        odResult,
                        hnnResult,
                        knnResult,
                        bitmapMap[hnnResult]?.get()
                    )
                }
            }
        }

        currentFrame++
        return result.values.toList()
    }

    private fun areBoundingBoxesSimilar(box1: RectF, box2: RectF, screenWidth: Int): Boolean {
        val maxWith = max(box1.width(), box2.width())
        if (abs(box1.width() - box2.width()) > maxWith * 0.15) {
            return false
        }
        val maxHeight = max(box1.height(), box2.height())
        if (abs(box1.height() - box2.height()) > maxHeight * 0.15) {
            return false
        }

        val minDistance = 0.05f * screenWidth
        val deltaX = box1.centerX() - box2.centerX()
        val deltaY = box2.centerY() - box2.centerY()
        val delta = sqrt(deltaX.pow(2) + deltaY.pow(2))
        if (delta > minDistance) {
            return false
        }

        return true
    }

}