package ai.passio.passiosdk.passiofood.data.model

import ai.passio.passiosdk.passiofood.data.measurement.Grams
import ai.passio.passiosdk.passiofood.data.measurement.UnitMass
import ai.passio.passiosdk.passiofood.search.ResponseFoodItem
import ai.passio.passiosdk.passiofood.upc.ResponseIngredient

data class PassioFoodItem(
    val id: String,
    val refCode: String,
    val name: String,
    val details: String,
    val iconId: String,
    val amount: PassioFoodAmount,
    val ingredients: List<PassioIngredient>,
) {

    internal companion object {
        fun fromSearchResponse(item: ResponseFoodItem, useShortName: Boolean): PassioFoodItem {
            val id = item.internalId
            val refCode = item.refCode ?: ""
            val name = if (useShortName) {
                item.internalName.ifEmpty { item.displayName }
            } else {
                item.displayName.ifEmpty { item.internalName }
            }
            val details = if (useShortName) {
                item.displayName.ifEmpty {
                    if (item.ingredients.size == 1) item.ingredients.first().branded.owner
                        ?: "" else ""
                }
            } else {
                if (item.ingredients.size == 1) item.ingredients.first().branded.owner?.ifEmpty { item.displayName }
                    ?: "" else ""
            }
            val iconId = item.iconId
            val ingredients = item.ingredients.map { PassioIngredient.fromResponse(it, iconId) }
                .toMutableList()
            val amount = if (item.ingredients.size == 1) {
                PassioFoodAmount.fromResponse(item.ingredients.first().portions)
            } else {
                val ingredientsWeight =
                    ingredients.map { it.weight() }.reduce { acc, unitMass -> acc + unitMass }
                        .gramsValue()
                PassioFoodAmount.fromResponse(item.portions, ingredientsWeight)
            }


            return PassioFoodItem(id, refCode, name, details, iconId, amount, ingredients)
        }

        fun fromUPCResponse(item: ResponseIngredient): PassioFoodItem {
            val id = item.id ?: ""
            val refCode = item.refCode ?: ""
            val name = item.name ?: ""
            val details = item.branded.owner ?: ""
            val iconId = item.iconId ?: ""
            val ingredient = PassioIngredient.fromResponse(item, iconId)
            val amount = PassioFoodAmount.fromResponse(item.portions)

            return PassioFoodItem(id, refCode, name, details, iconId, amount, listOf(ingredient))
        }
    }

    fun nutrients(weight: UnitMass): PassioNutrients {
        val currentWeight = ingredientWeight()
        val ingredientNutrients = ingredients.map { ingredient ->
            Pair(ingredient.referenceNutrients, ingredient.weight() / currentWeight)
        }
        return PassioNutrients(ingredientNutrients, weight)
    }

    fun nutrientsSelectedSize(): PassioNutrients {
        val currentWeight = ingredientWeight()
        val ingredientNutrients = ingredients.map { ingredient ->
            Pair(ingredient.referenceNutrients, amount.weight() / currentWeight)
        }
        return PassioNutrients(ingredientNutrients, currentWeight)
    }

    fun nutrientsReference(): PassioNutrients {
        val currentWeight = ingredientWeight()
        val ingredientNutrients = ingredients.map { ingredient ->
            Pair(ingredient.referenceNutrients, ingredient.weight() / currentWeight)
        }
        return PassioNutrients(ingredientNutrients, UnitMass(Grams, 100.0))
    }

    fun ingredientWeight(): UnitMass {
        return ingredients.map { it.weight() }.reduce { acc, unitMass -> acc + unitMass }
    }

    fun isOpenFood(): String? {
        ingredients.forEach { ingredient ->
            if (ingredient.metadata.openFoodLicense() != null) {
                return ingredient.metadata.openFoodLicense()
            }
        }
        return null
    }

    fun setWeightGrams(grams: Double) {
        amount.selectedQuantity = grams
        amount.selectedUnit = Grams.unitName

        if (ingredients.size == 1) {
            ingredients[0].amount.apply {
                selectedQuantity = grams
                selectedUnit = Grams.unitName
            }
            return
        }

        val ratio = grams / ingredientWeight().gramsValue()
        ingredients.forEach {
            it.amount.selectedQuantity *= ratio
        }
    }
}