package ai.passio.passiosdk.passiofood.upc

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.InflammatoryEffectData
import ai.passio.passiosdk.passiofood.data.DataUtils
import ai.passio.passiosdk.passiofood.data.model.PassioIDEntityType
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.search.ResponseFood
import ai.passio.passiosdk.passiofood.utils.DIIHelper
import android.util.Base64
import android.util.Log
import org.json.JSONException
import java.lang.IllegalStateException
import java.nio.charset.Charset

internal class UPCService(
    private val allowInternet: Boolean
) {

    private val foodItemCache = mutableMapOf<String, PassioFoodItem?>()
    private val tagsCache = mutableMapOf<String, List<String>?>()

    fun getFoodItemForPassioID(
        passioID: PassioID,
        dev: Boolean,
        languageCode: String?,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (foodItemCache.containsKey(passioID)) {
            onResult(foodItemCache[passioID])
            return
        }

        fetchFoodItemForPassioID(passioID, dev, languageCode) { response, shouldCache ->
            if (response == null || response.results.isEmpty()) {
                if (shouldCache) {
                    foodItemCache[passioID] = null
                }
                onResult(null)
                return@fetchFoodItemForPassioID
            }

            try {
                val foodItem = PassioFoodItem.fromSearchResponse(response.results.first(), true)
                foodItemCache[passioID] = foodItem
                onResult(foodItem)
            } catch (e: Exception) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Could not parse upcResponse: $passioID"
                )
                onResult(null)
            }
        }
    }

    private fun fetchFoodItemForPassioID(
        passioID: PassioID,
        dev: Boolean,
        languageCode: String?,
        onResult: (response: ResponseFood?, shouldCache: Boolean) -> Unit
    ) {
        val encoded = DataUtils.encodeMetadata(true)
        val url = NativeUtils.instance.nativeGetScanURL(dev)


        TokenService.getInstance().getToken(
            dev,
            { token ->
                val headers = mutableListOf("Authorization" to token)
                if (languageCode != null) {
                    headers.add("Localization-ISO" to languageCode)
                }
                NetworkService.instance.doRequestTrackTokens(
                    NetworkStringTask(
                        "$url?passioID=$passioID&metadata=$encoded",
                        headers
                    ),
                    object : NetworkCallback<String> {
                        override fun onFailure(code: Int, message: String) {
                            Log.e(UPCService::class.java.simpleName, message)
                            onResult(null, false)
                        }

                        override fun onSuccess(result: String) {
                            if (result == "null") {
                                onResult(null, true)
                                return
                            }

                            try {
                                val response = ResponseFood(result)
                                onResult(response, true)
                            } catch (jsonException: JSONException) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing JSON for passioID: $passioID"
                                )
                                onResult(null, true)
                            } catch (e: Exception) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing json result for: $passioID"
                                )
                                onResult(null, false)
                            }
                        }

                        override fun onTokenExpired() {
                            fetchFoodItemForPassioID(passioID, dev, languageCode, onResult)
                        }
                    }, "fetchFoodItemForPassioID"
                )
            }, {
                onResult(null, false)
            })
    }

    fun getFoodItemForProductCode(
        upcCode: String,
        dev: Boolean = false,
        languageCode: String?,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val cleanedUPC = cleanUPCCode(upcCode)

        // TODO test with removing UPC_A
        if (cleanedUPC.length == 13 && cleanedUPC.first() == '0') {
            cleanedUPC.substring(1, cleanedUPC.length)
        }

        if (foodItemCache.containsKey(cleanedUPC)) {
            onResult(foodItemCache[cleanedUPC])
            return
        }

        if (!allowInternet) {
            onResult(null)
            return
        }

        fetchUPSCode(
            cleanedUPC,
            dev,
            "fetchFoodItemForProductCode",
            null,
            languageCode
        ) { upcProduct, shouldCache ->
            if (upcProduct == null) {
                if (shouldCache) {
                    foodItemCache[upcCode] = null
                }
                onResult(null)
                return@fetchUPSCode
            }

            try {
                val foodItem = PassioFoodItem.fromUPCResponse(upcProduct)
                foodItemCache[upcCode] = foodItem
                onResult(foodItem)
            } catch (e: Exception) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Could not parse upcResponse: $upcProduct"
                )
                onResult(null)
            }

        }
    }

    private fun cleanUPCCode(upcCode: String): String {
        return upcCode.replace(PassioIDEntityType.barcode.value, "")
            .replace(PassioIDEntityType.packagedFoodCode.value, "")
    }

    private fun fetchUPSCode(
        upcCode: String,
        dev: Boolean,
        apiName: String,
        customUrl: String?,
        languageCode: String?,
        onUPSCode: (responseIngredient: ResponseIngredient?, shouldCache: Boolean) -> Unit
    ) {
        val url = customUrl ?: NativeUtils.instance.nativeGetUPCUrl(dev)

        val encoded = DataUtils.encodeMetadata(false)

        TokenService.getInstance().getToken(
            dev,
            { token ->
                val headers = mutableListOf("Authorization" to token)
                if (languageCode != null) {
                    headers.add("Localization-ISO" to languageCode)
                }
                NetworkService.instance.doRequestTrackTokens(
                    NetworkStringTask(
                        "$url$upcCode?metadata=$encoded",
                        headers
                    ),
                    object : NetworkCallback<String> {
                        override fun onFailure(code: Int, message: String) {
                            Log.e(UPCService::class.java.simpleName, message)
                            onUPSCode(null, false)
                        }

                        override fun onSuccess(result: String) {
                            if (result == "null") {
                                onUPSCode(null, true)
                                return
                            }

                            try {
                                val responseIngredient = ResponseIngredient(result)
                                onUPSCode(responseIngredient, true)
                            } catch (jsonException: JSONException) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing JSON for code: $upcCode"
                                )
                                onUPSCode(null, true)
                            }
                        }

                        override fun onTokenExpired() {
                            fetchUPSCode(upcCode, dev, apiName, customUrl, languageCode, onUPSCode)
                        }
                    },
                    apiName
                )
            }, {
                onUPSCode(null, false)
            })
    }

    fun fetchTags(
        passioID: PassioID,
        dev: Boolean,
        onFetched: (tags: List<String>?) -> Unit
    ) {
        if (tagsCache.containsKey(passioID)) {
            onFetched(tagsCache[passioID])
            return
        }

        val url = NativeUtils.instance.nativeGetTagsURL(dev)
        fetchUPSCode(passioID, dev, "fetchTagsFor", url, null) { upcProduct, shouldCache ->
            if (shouldCache) {
                tagsCache[passioID] = upcProduct?.tags
            }
            onFetched(upcProduct?.tags)
        }
    }

    fun fetchNutrients(
        passioID: PassioID,
        dev: Boolean,
        onFetched: (tags: List<InflammatoryEffectData>?) -> Unit
    ) {
        val url = NativeUtils.instance.nativeGetTagsURL(dev)
        fetchUPSCode(passioID, dev, "fetchInflammatoryEffectData", url, null) { upcProduct, _ ->
            if (upcProduct == null) {
                onFetched(null)
                return@fetchUPSCode
            }

            val nutrients = upcProduct.nutrients?.mapNotNull { upcNutrient ->
                val shortName = upcNutrient.nutrient?.shortName ?: ""
                val longName = upcNutrient.nutrient?.name ?: ""
                val name = DIIHelper.shortNames.firstOrNull { it == shortName }
                    ?: DIIHelper.longNames.firstOrNull { it == longName }
                if (name == null) {
                    null
                } else {
                    InflammatoryEffectData(
                        name,
                        upcNutrient.amount ?: 0.0,
                        upcNutrient.nutrient?.unit ?: "",
                        DIIHelper.inflammationScores[name] ?: 0.0
                    )
                }

            }
            onFetched(nutrients)
        }
    }
}