package ai.passio.passiosdk.core.config

import ai.passio.passiosdk.core.annotation.ExperimentalAPI
import ai.passio.passiosdk.core.annotation.PassioRestrictedApi
import android.content.Context

/**
 * A [PassioConfiguration] object must be constructed to configure the SDK. The configuration
 * process depends on the location of the files needed for the SDK to work. There are three
 * different ways to configure the SDK depending on the attributes of the object.
 *
 * @param appContext Application context needed for the license fetch service.
 * @param key Developer key provided by Passio Inc. For obtaining this key contact
 * support@passiolife.com.
 */
class PassioConfiguration(
    appContext: Context,
    key: String
) : BaseConfiguration(appContext, key) {
    /**
     * SDK Downloads Models indicates that the SDK will try to fetch the latest files every 2 weeks.
     * This mode works with the provided [customURL] or without it. If customURL is not null, the
     * SDK will go to that URL to try to fetch the latest files. Contrastingly, it will go to the
     * BaseUrl (Passio's server) to try to fetch the files. The flow of the configuration process is
     * the same like in the [customURL] case.
     */
    var sdkDownloadsModels: Boolean = true

    var allowInternetConnection: Boolean = true

    var remoteOnly: Boolean = false

    var proxyUrl: String? = null

    var proxyHeaders: Map<String, String>? = null

    var imageModel: String? = null

    var textModel: String? = null

    @PassioRestrictedApi
    var bridge: Bridge = Bridge.NONE

    override fun toString(): String {
        return "localFiles: $localFiles, autoUpdate: $sdkDownloadsModels, debugMode: $debugMode"
    }
}

/**
 * Object that is returned as a result of the configuration process.
 * @see [ai.passio.passiosdk.passiofood.PassioSDK.configure]
 */
class PassioStatus {

    internal companion object {
        fun error(errorType: PassioSDKError, message: String): PassioStatus {
            return PassioStatus().apply {
                mode = PassioMode.FAILED_TO_CONFIGURE
                error = errorType
                debugMessage = message
            }
        }

        fun configuring(): PassioStatus =
            PassioStatus().apply { mode = PassioMode.IS_BEING_CONFIGURED }

        fun downloading(): PassioStatus =
            PassioStatus().apply { mode = PassioMode.IS_DOWNLOADING_MODELS }
    }

    /**
     * Indicates the state of the configuration process.
     */
    var mode: PassioMode = PassioMode.NOT_READY
        internal set

    /**
     * If not null, lists all of the missing files needed for the SDK to achieve full operation.
     */
    var missingFiles: List<String>? = null
        internal set

    /**
     * Message that gives some detail about the result of the configuration process. Should not be
     * used programmatically!
     */
    var debugMessage: String? = null
        internal set

    /**
     * Indicates that current version of the installed files.
     */
    var activeModels: Int? = null
        internal set

    /**
     * If the SDK configuration process run unsuccessfully the [error] will contain the reason.
     */
    var error: PassioSDKError? = null
        internal set

    var metadataHash: String? = null
        internal set

    override fun toString(): String {
        return "mode: $mode, missingFiles: $missingFiles, debugMessage: $debugMessage, activeModels: $activeModels, metadataHash: $metadataHash"
    }
}

enum class PassioMode {
    /**
     * Indicates that the configuration process has not started yet.
     */
    NOT_READY,

    /**
     * The configuration process has encountered an error and was not completed successfully.
     */
    FAILED_TO_CONFIGURE,

    /**
     * The configuration process is still running.
     */
    IS_BEING_CONFIGURED,

    /**
     * A newer version of files needed by the SDK are being downloaded.
     */
    IS_DOWNLOADING_MODELS,

    /**
     * All the files needed by the SDK are present and the SDK can be used to its full extent.
     */
    IS_READY_FOR_DETECTION
}

enum class Bridge {
    NONE,
    FLUTTER,
    REACT_NATIVE
}