package ai.passio.passiosdk.passiofood.upc

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.InflammatoryEffectData
import ai.passio.passiosdk.passiofood.config.SDKProperties
import ai.passio.passiosdk.passiofood.data.DataUtils
import ai.passio.passiosdk.passiofood.data.model.PassioIDEntityType
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.search.ResponseFood
import ai.passio.passiosdk.passiofood.utils.DIIHelper
import android.util.Base64
import android.util.Log
import org.json.JSONException
import java.lang.IllegalStateException
import java.nio.charset.Charset

internal class UPCService(
    private val allowInternet: Boolean
) {

    private val foodItemCache = mutableMapOf<String, PassioFoodItem?>()
    private val tagsCache = mutableMapOf<String, List<String>?>()

    fun getFoodItemForPassioID(
        passioID: PassioID,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (foodItemCache.containsKey(passioID)) {
            onResult(foodItemCache[passioID])
            return
        }

        fetchFoodItemForPassioID(passioID) { response, shouldCache ->
            if (response == null || response.results.isEmpty()) {
                if (shouldCache) {
                    foodItemCache[passioID] = null
                }
                onResult(null)
                return@fetchFoodItemForPassioID
            }

            try {
                val foodItem = PassioFoodItem.fromSearchResponse(response.results.first(), true)
                foodItemCache[passioID] = foodItem
                onResult(foodItem)
            } catch (e: Exception) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Could not parse upcResponse: $passioID"
                )
                onResult(null)
            }
        }
    }

    private fun fetchFoodItemForPassioID(
        passioID: PassioID,
        onResult: (response: ResponseFood?, shouldCache: Boolean) -> Unit
    ) {
        val encoded = DataUtils.encodeMetadata(true)
        val url = NativeUtils.instance.getScanURL(SDKProperties.dev)


        TokenService.getInstance().getToken(
            { token ->
                val headers = mutableMapOf("Authorization" to token)
                SDKProperties.languageCode?.let {
                    headers["Localization-ISO"] = it
                }
                NetworkService.instance.doRequestTrackTokens(
                    NetworkStringTask(
                        "$url?passioID=$passioID&metadata=$encoded",
                        headers
                    ),
                    object : NetworkCallback<String> {
                        override fun onFailure(code: Int, message: String) {
                            Log.e(UPCService::class.java.simpleName, message)
                            onResult(null, false)
                        }

                        override fun onSuccess(result: String) {
                            if (result == "null") {
                                onResult(null, true)
                                return
                            }

                            try {
                                val response = ResponseFood(result)
                                onResult(response, true)
                            } catch (jsonException: JSONException) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing JSON for passioID: $passioID, ${jsonException.message}"
                                )
                                onResult(null, true)
                            } catch (e: Exception) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing json result for: $passioID, ${e.message}"
                                )
                                onResult(null, false)
                            }
                        }

                        override fun onTokenExpired() {
                            fetchFoodItemForPassioID(passioID, onResult)
                        }
                    }, "fetchFoodItemForPassioID"
                )
            }, {
                onResult(null, false)
            })
    }

    fun getFoodItemForProductCode(
        upcCode: String,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val cleanedUPC = cleanUPCCode(upcCode)

        // TODO test with removing UPC_A
        if (cleanedUPC.length == 13 && cleanedUPC.first() == '0') {
            cleanedUPC.substring(1, cleanedUPC.length)
        }

        if (foodItemCache.containsKey(cleanedUPC)) {
            onResult(foodItemCache[cleanedUPC])
            return
        }

        if (!allowInternet) {
            onResult(null)
            return
        }

        fetchUPSCode(
            cleanedUPC,
            "fetchFoodItemForProductCode",
            null,
        ) { upcProduct, shouldCache ->
            if (upcProduct == null) {
                if (shouldCache) {
                    foodItemCache[upcCode] = null
                }
                onResult(null)
                return@fetchUPSCode
            }

            try {
                val foodItem = PassioFoodItem.fromUPCResponse(upcProduct)
                foodItemCache[upcCode] = foodItem
                onResult(foodItem)
            } catch (e: Exception) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Could not parse upcResponse: $upcProduct"
                )
                onResult(null)
            }

        }
    }

    private fun cleanUPCCode(upcCode: String): String {
        return upcCode.replace(PassioIDEntityType.barcode.value, "")
            .replace(PassioIDEntityType.packagedFoodCode.value, "")
    }

    private fun fetchUPSCode(
        upcCode: String,
        apiName: String,
        customUrl: String?,
        onUPSCode: (responseIngredient: ResponseIngredient?, shouldCache: Boolean) -> Unit
    ) {
        val url = customUrl ?: NativeUtils.instance.getUPCUrl(SDKProperties.dev)

        val encoded = DataUtils.encodeMetadata(false)

        TokenService.getInstance().getToken(
            { token ->
                val headers = mutableMapOf("Authorization" to token)
                SDKProperties.languageCode?.let {
                    headers["Localization-ISO"] = it
                }
                NetworkService.instance.doRequestTrackTokens(
                    NetworkStringTask(
                        "$url$upcCode?metadata=$encoded",
                        headers
                    ),
                    object : NetworkCallback<String> {
                        override fun onFailure(code: Int, message: String) {
                            Log.e(UPCService::class.java.simpleName, message)
                            onUPSCode(null, false)
                        }

                        override fun onSuccess(result: String) {
                            if (result == "null") {
                                onUPSCode(null, true)
                                return
                            }

                            try {
                                val responseIngredient = ResponseIngredient(result)
                                onUPSCode(responseIngredient, true)
                            } catch (jsonException: JSONException) {
                                PassioLog.e(
                                    UPCService::class.java.simpleName,
                                    "Error parsing JSON for code: $upcCode, ${jsonException.message}"
                                )
                                onUPSCode(null, true)
                            }
                        }

                        override fun onTokenExpired() {
                            fetchUPSCode(upcCode, apiName, customUrl, onUPSCode)
                        }
                    },
                    apiName
                )
            }, {
                onUPSCode(null, false)
            })
    }

    fun fetchTags(
        passioID: PassioID,
        dev: Boolean,
        onFetched: (tags: List<String>?) -> Unit
    ) {
        if (tagsCache.containsKey(passioID)) {
            onFetched(tagsCache[passioID])
            return
        }

        val url = NativeUtils.instance.getTagsURL(dev)
        fetchUPSCode(passioID, "fetchTagsFor", url) { upcProduct, shouldCache ->
            if (shouldCache) {
                tagsCache[passioID] = upcProduct?.tags
            }
            onFetched(upcProduct?.tags)
        }
    }
}