package ai.passio.passiosdk.core.os

import android.content.Context
import android.os.Build
import android.os.Handler
import android.util.Log
import java.io.*
import java.util.concurrent.Executor
import java.util.concurrent.RejectedExecutionException

internal fun <T> measure(
    prefix: String = "",
    tag: String = "HHHH",
    block: () -> T,
): T {
    val startTime = System.currentTimeMillis()
    val t = block()
    val time = System.currentTimeMillis() - startTime
    Log.d(tag, "$prefix$time ms")
    return t
}

internal fun Context.getPassioMainExecutor(): PassioMainHandlerExecutor {
    return PassioMainHandlerExecutor(Handler(mainLooper))
}

internal class PassioMainHandlerExecutor constructor(private val mHandler: Handler) : Executor {
    override fun execute(command: Runnable) {
        if (!mHandler.post(command)) {
            throw RejectedExecutionException("$mHandler is shutting down")
        }
    }

    fun executeDelayed(command: Runnable, delayMillis: Long) {
        if (!mHandler.postDelayed(command, delayMillis)) {
            throw RejectedExecutionException("$mHandler is shutting down")
        }
    }
}

internal object ProcUtils {

    const val NO_CPU_INFO = -1

    private val CPU_FILTER = FileFilter { pathname ->
        val path = pathname.name
        //regex is slow, so checking char by char.
        if (path.startsWith("cpu")) {
            for (i in 3 until path.length) {
                if (!Character.isDigit(path[i])) {
                    return@FileFilter false
                }
            }
            return@FileFilter true
        }
        false
    }


    private fun getNumberOfCPUCores(): Int {
        var cores: Int
        try {
            cores = getCoresFromFileInfo("/sys/devices/system/cpu/possible")
            if (cores == NO_CPU_INFO) {
                cores = getCoresFromFileInfo("/sys/devices/system/cpu/present")
            }
            if (cores == NO_CPU_INFO) {
                cores = getCoresFromCPUFileList()
            }
        } catch (e: SecurityException) {
            cores = NO_CPU_INFO
        } catch (e: NullPointerException) {
            cores = NO_CPU_INFO
        }
        return cores
    }

    private fun getCoresFromFileInfo(fileLocation: String): Int {
        var inputStream: InputStream? = null
        return try {
            inputStream = FileInputStream(fileLocation)
            val buf = BufferedReader(InputStreamReader(inputStream))
            val fileContents = buf.readLine()
            buf.close()
            getCoresFromFileString(fileContents)
        } catch (e: IOException) {
            NO_CPU_INFO
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close()
                } catch (e: IOException) {
                    // Do nothing.
                }
            }
        }
    }

    private fun getCoresFromFileString(str: String?): Int {
        return if (str == null || !str.matches(Regex.fromLiteral("0-[\\d]+$"))) {
            NO_CPU_INFO
        } else Integer.valueOf(str.substring(2)) + 1
    }

    @Throws(IOException::class)
    fun getCPUInfo(): Map<String, String> {
        val br = BufferedReader(FileReader("/proc/cpuinfo"))
        var str: String? = null
        val output: MutableMap<String, String> = HashMap()
        while ({ str = br.readLine(); str }() != null) {
            val data = str!!.split(":").toTypedArray()
            if (data.size > 1) {
                var key = data[0].trim { it <= ' ' }.replace(" ", "_")
                if (key == "model_name") key = "cpu_model"
                output[key] = data[1].trim { it <= ' ' }
            }
        }
        br.close()
        return output
    }

    private fun getCoresFromCPUFileList(): Int {
        return File("/sys/devices/system/cpu/").listFiles(CPU_FILTER)?.size ?: 0
    }

    fun getCPUMaxFreqKHz(): Int {
        var maxFreq: Int = NO_CPU_INFO
        try {
            for (i in 0 until getNumberOfCPUCores()) {
                val filename = "/sys/devices/system/cpu/cpu$i/cpufreq/cpuinfo_max_freq"
                val cpuInfoMaxFreqFile = File(filename)
                if (cpuInfoMaxFreqFile.exists() && cpuInfoMaxFreqFile.canRead()) {
                    val buffer = ByteArray(128)
                    val stream = FileInputStream(cpuInfoMaxFreqFile)
                    try {
                        stream.read(buffer)
                        var endIndex = 0
                        //Trim the first number out of the byte buffer.
                        while (Character.isDigit(buffer[endIndex].toChar()) && endIndex < buffer.size) {
                            endIndex++
                        }
                        val str = String(buffer, 0, endIndex)
                        val freqBound = str.toInt()
                        if (freqBound > maxFreq) {
                            maxFreq = freqBound
                        }
                    } catch (e: NumberFormatException) {
                        //Fall through and use /proc/cpuinfo.
                    } finally {
                        stream.close()
                    }
                }
            }
            if (maxFreq == NO_CPU_INFO) {
                val stream = FileInputStream("/proc/cpuinfo")
                try {
                    var freqBound: Int = parseFileForValue("cpu MHz", stream)
                    freqBound *= 1000 //MHz -> kHz
                    if (freqBound > maxFreq) maxFreq = freqBound
                } finally {
                    stream.close()
                }
            }
        } catch (e: IOException) {
            maxFreq = NO_CPU_INFO //Fall through and return unknown.
        }
        return maxFreq
    }

    private fun parseFileForValue(textToMatch: String, stream: FileInputStream): Int {
        val buffer = ByteArray(1024)
        try {
            val length = stream.read(buffer)
            var i = 0
            while (i < length) {
                if (buffer[i].toChar() == '\n' || i == 0) {
                    if (buffer[i].toChar() == '\n') i++
                    for (j in i until length) {
                        val textIndex = j - i
                        //Text doesn't match query at some point.
                        if (buffer[j].toChar() != textToMatch[textIndex]) {
                            break
                        }
                        //Text matches query here.
                        if (textIndex == textToMatch.length - 1) {
                            return extractValue(buffer, j)
                        }
                    }
                }
                i++
            }
        } catch (e: IOException) {
            //Ignore any exceptions and fall through to return unknown value.
        } catch (e: java.lang.NumberFormatException) {
        }
        return NO_CPU_INFO
    }

    private fun extractValue(buffer: ByteArray, index: Int): Int {
        var myIndex = index
        while (myIndex < buffer.size && buffer[myIndex].toChar() != '\n') {
            if (Character.isDigit(buffer[myIndex].toChar())) {
                val start = myIndex
                myIndex++
                while (myIndex < buffer.size && Character.isDigit(buffer[myIndex].toChar())) {
                    myIndex++
                }
                val str = String(buffer, start, myIndex - start)
                return str.toInt()
            }
            myIndex++
        }
        return NO_CPU_INFO
    }
}