package ai.passio.passiosdk.passiofood

import ai.passio.passiosdk.BuildConfig
import ai.passio.passiosdk.R
import ai.passio.passiosdk.core.authentication.AuthenticationService
import ai.passio.passiosdk.core.camera.PassioCameraConfigurator
import ai.passio.passiosdk.core.camera.PassioCameraProxy
import ai.passio.passiosdk.core.camera.PassioCameraViewProvider
import ai.passio.passiosdk.core.camera.PassioCameraXProxy
import ai.passio.passiosdk.core.config.Bridge
import ai.passio.passiosdk.core.config.PassioConfiguration
import ai.passio.passiosdk.core.config.PassioMode
import ai.passio.passiosdk.core.config.PassioSDKError
import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.core.download.PassioDownloadManager
import ai.passio.passiosdk.core.file.PassioFileManager
import ai.passio.passiosdk.core.icons.IconFileManager
import ai.passio.passiosdk.core.icons.IconService
import ai.passio.passiosdk.core.icons.IconSize
import ai.passio.passiosdk.core.report.ActivationEvent
import ai.passio.passiosdk.core.report.ReportingService
import ai.passio.passiosdk.core.sharedpreferences.CorePreferencesManager
import ai.passio.passiosdk.core.sharedpreferences.DeviceIDPreferencesManager
import ai.passio.passiosdk.core.utils.FileUtil
import ai.passio.passiosdk.core.utils.ImageUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.core.version.ConfigurationListener
import ai.passio.passiosdk.passiofood.advisor.AdvisorService
import ai.passio.passiosdk.passiofood.config.PassioSDKFileConfig
import ai.passio.passiosdk.passiofood.config.passio_dataset
import ai.passio.passiosdk.passiofood.config.passio_metadata
import ai.passio.passiosdk.passiofood.data.model.IconDefaults
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorFoodInfo
import ai.passio.passiosdk.core.camera.PassioCameraData
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorResponse
import ai.passio.passiosdk.passiofood.data.model.PassioIDEntityType
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioFoodResultType
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlan
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlanItem
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.file.PassioFoodFileManager
import ai.passio.passiosdk.passiofood.metadata.MetadataManager
import ai.passio.passiosdk.passiofood.recognition.PassioRecognizer
import ai.passio.passiosdk.passiofood.recommend.MealPlanService
import ai.passio.passiosdk.passiofood.search.SearchService
import ai.passio.passiosdk.passiofood.recommend.SuggestionService
import ai.passio.passiosdk.passiofood.search.NutritionPreviewResult
import ai.passio.passiosdk.passiofood.upc.UPCService
import ai.passio.passiosdk.passiofood.version.PassioVersionManager
import ai.passio.passiosdk.passiofood.data.model.PassioSpeechRecognitionModel
import ai.passio.passiosdk.passiofood.data.model.PassioTokenBudget
import ai.passio.passiosdk.passiofood.token.TokenUsageListener
import ai.passio.passiosdk.passiofood.token.TokenUsageTracker
import ai.passio.passiosdk.passiofood.user.UserService
import ai.passio.passiosdk.passiofood.voice.remote.SpeechService
import android.content.Context
import android.graphics.Bitmap
import android.graphics.RectF
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Build
import android.provider.Settings
import android.util.Size
import androidx.annotation.WorkerThread
import androidx.core.content.ContextCompat
import org.json.JSONObject
import java.io.IOException
import java.lang.ref.WeakReference
import java.util.concurrent.CompletableFuture
import java.util.Locale
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.atomic.AtomicReference

internal class PassioSDKImpl : PassioSDK, PassioSDKDebug,
    PassioDownloadManager.PassioDownloadListener,
    ConfigurationListener, TokenUsageListener, NutritionAdvisor {

    companion object {
        var FILE_VERSION = 20240207

        const val FOOD_IMAGE_DIRECTORY = "foodIcons/"

        const val SDK_VERSION = "3.2.2"
        private const val PRODUCT_ID = "PassioSDK-Nutrition-AI"
        const val ARCHITECTURE = "hnn0"
    }

    private val sdkFileConfig = PassioSDKFileConfig()
    private val passioFileManager = PassioFoodFileManager(sdkFileConfig)

    private lateinit var authenticationService: AuthenticationService
    private lateinit var sharedPreferencesManager: CorePreferencesManager
    private var reportingService: ReportingService? = null
    private var reportMetadata: JSONObject? = null

    private var downloadManager = PassioDownloadManager(passioFileManager, this)

    private var cameraProxy: PassioCameraProxy = PassioCameraXProxy().apply {
        setTimeBetweenFrames(500L)
    }
    // private var dbHelper: DBHelper? = null

    private var executor = Executors.newSingleThreadExecutor()
    private lateinit var mainExecutor: Executor
    private var passioRecognizer: PassioRecognizer? = null
    private val metadataManager = MetadataManager(passioFileManager)
    private val iconService = IconService(metadataManager)

    private var statusListener: PassioStatusListener? = null

    private var upcService: UPCService? = null

    private val passioStatusRef = AtomicReference(PassioStatus())
    private val versionManager = PassioVersionManager(passioFileManager, this)

    private var configurationCompletion: (status: PassioStatus) -> Unit = {}
    private var allowInternet = false
    private val advisorService = AdvisorService(BuildConfig.DEV_API)

    private var contextRef: WeakReference<Context?>? = null
    private var tokenUsageListener: PassioAccountListener? = null
    private var languageCode: String? = null
    private var remoteOnly = false

    override fun configure(
        passioConfiguration: PassioConfiguration,
        onComplete: (status: PassioStatus) -> Unit
    ) {
        mainExecutor = ContextCompat.getMainExecutor(passioConfiguration.appContext)
        TokenUsageTracker.listener = this

        val passioStatus = passioStatusRef.get()
        if (passioStatus.mode == PassioMode.IS_BEING_CONFIGURED ||
            passioStatus.mode == PassioMode.IS_DOWNLOADING_MODELS
        ) {
            return
        }

        this.allowInternet = passioConfiguration.allowInternetConnection

        PassioLog.setupMode(passioConfiguration.debugMode)
        PassioLog.d(PassioSDKImpl::class.java.simpleName, "Configure: $passioConfiguration")

        passioStatus.mode = PassioMode.IS_BEING_CONFIGURED
        updatePassioStatus(passioStatus)

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            val errorStatus = PassioStatus.error(
                PassioSDKError.MIN_SDK_VERSION,
                "Min Android API supported: ${Build.VERSION_CODES.O}"
            )
            updatePassioStatus(errorStatus)
            onComplete(errorStatus)
            return
        }

        internalConfigure(
            passioConfiguration.appContext,
            passioConfiguration.key,
            passioConfiguration.localFiles,
            passioConfiguration.sdkDownloadsModels,
            passioConfiguration.bridge,
            passioConfiguration.remoteOnly,
            onComplete
        )
    }

    private fun isDBReady(): Boolean =
        passioStatusRef.get().mode == PassioMode.IS_READY_FOR_DETECTION

    private fun getCachedDeviceID(context: Context): String {
        val prefs = DeviceIDPreferencesManager(context)
        val cachedID = prefs.getDeviceID()
        if (cachedID == null) {
            val newID = prefs.generateRandomID(16)
            prefs.setDeviceID(newID)
            return newID
        }
        return cachedID
    }

    private fun internalConfigure(
        context: Context,
        developerKey: String,
        withFiles: List<Uri>?,
        autoUpdate: Boolean,
        bridge: Bridge,
        remoteOnly: Boolean,
        onComplete: (status: PassioStatus) -> Unit
    ) {
        contextRef = WeakReference(context)
        configurationCompletion = onComplete
        this.remoteOnly = remoteOnly

        passioRecognizer = PassioRecognizer(
            cameraProxy,
            mainExecutor,
            passioFileManager,
            metadataManager,
            remoteOnly
        )
        cameraProxy.setCameraListener(passioRecognizer!!)

        val udid = try {
            Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID)
                ?: getCachedDeviceID(context)
        } catch (e: SecurityException) {
            PassioLog.e(this::class.java.simpleName, "Could not fetch ANDROID_ID")
            getCachedDeviceID(context)
        }

        val licenseMetadata = "$PRODUCT_ID/${SDK_VERSION}/Android/$udid"
        val platform = when (bridge) {
            Bridge.NONE -> "Android"
            Bridge.FLUTTER -> "Flutter-Android"
            Bridge.REACT_NATIVE -> "RN-Android"
        }

        reportingService = ReportingService(context)
        reportMetadata = JSONObject().apply {
            put("installid", udid)
            put("key", developerKey)
            put("platform", platform)
            put("product", PRODUCT_ID)
            put("version", SDK_VERSION)
        }

        sharedPreferencesManager = CorePreferencesManager(context)
        authenticationService = AuthenticationService(context, allowInternet)

        downloadManager.onStart()

        executor.submit {
            authenticationService.checkLicense(
                context,
                developerKey,
                licenseMetadata,
                BuildConfig.DEV_API,
                object : AuthenticationService.AuthenticationListener {
                    override fun onAuthorized() {
                        executor.submit auth@{
                            PassioLog.d(this::class.java.simpleName, "On license fetched")

                            val errorMsg = checkDependencies()
                            if (errorMsg != null) {
                                val status =
                                    PassioStatus.error(PassioSDKError.MISSING_DEPENDENCY, errorMsg)
                                updatePassioStatus(status)
                                mainExecutor.execute {
                                    onComplete(status)
                                }
                                return@auth
                            }

                            if (autoUpdate) {
                                val status =
                                    versionManager.configureSDKWithRemoteFiles(context, remoteOnly)
                                updatePassioStatus(status)
                                passioRecognizer!!.onPassioStatus(status)
                                mainExecutor.execute {
                                    configurationCompletion.invoke(status)
                                }
                            } else {
                                val status = versionManager.configureSDKWithExternalFiles(
                                    context,
                                    withFiles ?: listOf(),
                                    remoteOnly
                                )
                                PassioLog.d(
                                    this::class.java.simpleName,
                                    "External file config finished - $status"
                                )
                                updatePassioStatus(status)
                                passioRecognizer!!.onPassioStatus(status)
                                mainExecutor.execute {
                                    configurationCompletion.invoke(status)
                                }
                            }
                        }
                    }

                    override fun onError(type: AuthenticationService.Error, message: String) {
                        val status = when (type) {
                            AuthenticationService.Error.DECODING_ERROR ->
                                PassioStatus.error(PassioSDKError.LICENSE_DECODING_ERROR, message)

                            AuthenticationService.Error.NETWORK_ERROR ->
                                PassioStatus.error(PassioSDKError.NETWORK_ERROR, message)

                            AuthenticationService.Error.KEY_EXPIRED ->
                                PassioStatus.error(PassioSDKError.LICENSE_KEY_HAS_EXPIRED, message)

                            AuthenticationService.Error.KEY_LENGTH_ERROR -> {
                                PassioStatus.error(
                                    PassioSDKError.KEY_NOT_VALID,
                                    context.getString(R.string.key_not_valid_message)
                                )
                            }

                            AuthenticationService.Error.TOKEN_FETCH_ERROR -> {
                                PassioStatus.error(
                                    PassioSDKError.KEY_NOT_VALID,
                                    context.getString(R.string.key_not_valid_message)
                                )
                            }
                        }

                        updatePassioStatus(status)
                        PassioLog.e(this::class.java.simpleName, "License error - $status")
                        mainExecutor.execute {
                            onComplete(status)
                        }
                    }

                })
        }
    }

    private fun checkDependencies(): String? {
        try {
            Class.forName("org.tensorflow.lite.Interpreter")
        } catch (e: ClassNotFoundException) {
            val msg = "Could not find tflite dependency"
            PassioLog.e(this::class.java.simpleName, msg)
            return msg
        }

        try {
            Class.forName("org.tensorflow.lite.support.metadata.MetadataExtractor")
        } catch (e: ClassNotFoundException) {
            val msg = "Could not find tflite metadata dependency"
            PassioLog.e(this::class.java.simpleName, msg)
            return msg
        }

        try {
            Class.forName("com.google.mlkit.vision.barcode.BarcodeScanner")
        } catch (e: ClassNotFoundException) {
            val msg = "Could not find mlkit barcode dependency"
            PassioLog.e(this::class.java.simpleName, msg)
            return msg
        }

        try {
            Class.forName("com.google.mlkit.vision.text.TextRecognizer")
        } catch (e: ClassNotFoundException) {
            val msg = "Could not find mlkit text recognizer dependency"
            PassioLog.e(this::class.java.simpleName, msg)
            return msg
        }

        return null
    }

    private fun updatePassioStatus(newStatus: PassioStatus) {
        PassioLog.i(this::class.java.simpleName, "Update passio status: $newStatus")
        passioStatusRef.set(newStatus)
        mainExecutor.execute {
            statusListener?.onPassioStatusChanged(newStatus)
        }
        if (newStatus.mode == PassioMode.IS_READY_FOR_DETECTION) {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.ACTIVATION,
                BuildConfig.DEV_API
            )
        }
    }

    override fun isSDKReady(): Boolean {
        return passioStatusRef.get().mode == PassioMode.IS_READY_FOR_DETECTION
    }

    override fun shutDownPassioSDK() {
        super.shutDownPassioSDK()
        downloadManager.onStop()
        contextRef?.clear()
        contextRef = null
    }

    override fun startFoodDetection(
        foodRecognitionListener: FoodRecognitionListener,
        detectionConfig: FoodDetectionConfiguration
    ): Boolean {
        if (passioRecognizer == null) {
            PassioLog.e(
                PassioSDK::class.java.simpleName,
                "Could not start food detection, configure not called"
            )
            return false
        }
        if (remoteOnly && detectionConfig.detectVisual) {
            PassioLog.e(
                PassioSDK::class.java.simpleName,
                "SDK set up as remote only, visual food detection won't work"
            )
        }
        if (remoteOnly && detectionConfig.detectPackagedFood) {
            PassioLog.e(
                PassioSDK::class.java.simpleName,
                "SDK set up as remote only, packaged food detection won't work"
            )
        }

        executor.submit {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.DETECT_IN_VIDEO,
                BuildConfig.DEV_API
            )
        }

        PassioLog.d(this::class.java.simpleName, "Start food detection - $detectionConfig")
        when (detectionConfig.framesPerSecond) {
            PassioSDK.FramesPerSecond.ONE -> runFrameEvery(1000L)
            PassioSDK.FramesPerSecond.TWO -> runFrameEvery(500L)
            PassioSDK.FramesPerSecond.MAX -> runFrameEvery(0L)
        }
        passioRecognizer!!.startFoodDetection(
            foodRecognitionListener,
            detectionConfig.detectVisual,
            detectionConfig.detectBarcodes,
            detectionConfig.detectPackagedFood,
        )

        return true
    }

    override fun stopFoodDetection(): Boolean {
        if (passioRecognizer == null) {
            return false
        }
        passioRecognizer!!.stopFoodDetection()
        return true
    }

    override fun enableFrameAveraging(enabled: Boolean): Boolean {
        if (passioRecognizer == null) {
            return false
        }

        passioRecognizer!!.enableFrameAveraging(enabled)
        return true
    }

    override fun setCurrentVersion(version: Int) {
        FILE_VERSION = version
    }

    override fun runFrameEvery(timeForAnalysis: Long) {
        cameraProxy.setTimeBetweenFrames(timeForAnalysis)
    }

    override fun getFrameSize(): Size {
        return cameraProxy.getFrameSize()
    }

    override fun startCamera(
        passioCameraViewProvider: PassioCameraViewProvider,
        displayRotation: Int,
        cameraFacing: Int,
        tapToFocus: Boolean,
        onCameraReady: (data: PassioCameraData) -> Unit,
    ) {
        executor.submit {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.CAMERA,
                BuildConfig.DEV_API
            )
        }

        PassioLog.i(this::class.java.simpleName, "Start camera")
        cameraProxy.setCameraViewProvider(passioCameraViewProvider)
        passioCameraViewProvider.requestPreviewView().post {
            cameraProxy.startCamera(displayRotation, cameraFacing) { data ->
                val frameWidth = cameraProxy.getFrameSize().width
                val frameHeight = cameraProxy.getFrameSize().height
                val rotation = cameraProxy.getSensorRotation() ?: 0
                PassioLog.i(
                    this::class.java.simpleName,
                    "Camera ready, preview: ${frameWidth}x${frameHeight}, rotation: $rotation"
                )
                onCameraReady(data)
            }
            if (tapToFocus) {
                cameraProxy.enableTapToFocus()
            }
        }
    }

    override fun startCamera(
        viewProvider: PassioCameraViewProvider,
        configurator: PassioCameraConfigurator
    ) {
        executor.submit {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.CAMERA,
                BuildConfig.DEV_API
            )
        }

        PassioLog.i(this::class.java.simpleName, "Start camera")
        cameraProxy.setCameraViewProvider(viewProvider)
        viewProvider.requestPreviewView().post {
            cameraProxy.startCamera(configurator) {
                val frameWidth = cameraProxy.getFrameSize().width
                val frameHeight = cameraProxy.getFrameSize().height
                val rotation = cameraProxy.getSensorRotation() ?: 0
                PassioLog.i(
                    this::class.java.simpleName,
                    "Camera ready, preview: ${frameWidth}x${frameHeight}, rotation: $rotation"
                )
            }
            if (configurator.tapToFocus()) {
                cameraProxy.enableTapToFocus()
            }
        }
    }

    override fun stopCamera() {
        cameraProxy.stopCamera()
    }


    override fun enableFlashlight(enabled: Boolean) {
        val success = cameraProxy.enableFlashlight(enabled)
        if (!success) {
            PassioLog.e(
                this::class.java.simpleName,
                "Camera not bounded, could not enable flashlight"
            )
        }
    }

    override fun setCameraZoomLevel(zoomLevel: Float) {
        val success = cameraProxy.setZoomLevel(zoomLevel)
        if (!success) {
            PassioLog.e(
                this::class.java.simpleName,
                "Camera not bounded, could not set zoom level"
            )
        }
    }

    override fun getMinMaxCameraZoomLevel(): Pair<Float?, Float?> {
        val levels = cameraProxy.getMinMaxCameraZoomLevel()
        if (levels == null) {
            PassioLog.e(
                this::class.java.simpleName,
                "Camera not bounded, could not get zoom levels"
            )
            return null to null
        }
        return levels
    }

    @Deprecated("No longer supported", replaceWith = ReplaceWith("recognizeImageRemote"))
    override fun detectFoodIn(
        bitmap: Bitmap,
        foodDetectionConfiguration: FoodDetectionConfiguration?,
        onDetectionCompleted: (candidates: FoodCandidates?) -> Unit
    ): Boolean {
        if (passioRecognizer == null) {
            return false
        }

        if (remoteOnly) {
            PassioLog.e(
                this::class.java.simpleName,
                "Detect food in doesn't work with remoteOnly configuration!"
            )
            return false
        }

        executor.submit {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.DETECT_IN_IMAGE,
                BuildConfig.DEV_API
            )
        }

        passioRecognizer!!.detectFoodIn(bitmap, foodDetectionConfiguration, onDetectionCompleted)
        return true
    }

    override fun lookupIconsFor(
        context: Context,
        passioID: PassioID,
        iconSize: IconSize,
        type: PassioIDEntityType
    ): Pair<Drawable, Drawable?> {
        val defaultImageName = IconDefaults.entityTypeToIcon(type).iconName
        val defaultImage = lookupImageForFilename(context, defaultImageName) ?: BitmapDrawable(
            context.resources,
            Bitmap.createBitmap(0, 0, Bitmap.Config.ARGB_8888)
        )

        val localImage = getLocalImage(context, passioID, iconSize)
        return defaultImage to localImage
    }

    override fun fetchIconFor(
        context: Context,
        passioID: PassioID,
        iconSize: IconSize,
        callback: (drawable: Drawable?) -> Unit
    ) {
        val localImage = getLocalImage(context, passioID, iconSize)
        if (localImage != null) {
            callback(localImage)
            return
        }

        // If possible, fetch the icon from Passio's backend
        if (allowInternet) {
            iconService.fetchIcon(
                passioID,
                iconSize
            ) { bitmap ->
                if (bitmap != null) {
                    IconFileManager.storeIcon(context, passioID, bitmap, iconSize)
                    mainExecutor.execute {
                        callback(BitmapDrawable(context.resources, bitmap))
                    }
                } else {
                    // Could not fetch icon, return default icon for type
                    mainExecutor.execute {
                        callback(null)
                    }
                }
            }
        } else {
            // Could not fetch icon, return default icon for type
            callback(null)
        }
    }

    private fun getLocalImage(
        context: Context,
        passioID: PassioID,
        iconSize: IconSize,
    ): Drawable? {
        // Check local icons package
        val localImage = lookupImageForFilename(context, passioID)
        if (localImage != null) {
            return localImage
        }

        // Check the icon cache
        val cachedImage = IconFileManager.loadIcon(
            context,
            passioID,
            iconSize
        )
        if (cachedImage != null) {
            return BitmapDrawable(context.resources, cachedImage)
        }

        return null
    }

    private fun lookupImageForFilename(context: Context, imageName: String): Drawable? {
        return try {
            FileUtil.loadDrawableFromAssets(context.assets, "$FOOD_IMAGE_DIRECTORY$imageName.jpg")
        } catch (e: IOException) {
            null
        }
    }

    override fun fetchFoodItemForProductCode(
        productCode: String,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            onResult(null)
            return
        }

        executor.submit {
            reportingService?.reportSDKUsage(
                reportMetadata,
                ActivationEvent.PRODUCT_CODE,
                BuildConfig.DEV_API
            )
        }

        upcService!!.getFoodItemForProductCode(
            productCode,
            BuildConfig.DEV_API,
            languageCode
        ) { foodItem ->
            mainExecutor.execute {
                onResult(foodItem)
            }
        }
    }

    override fun fetchFoodItemForPassioID(
        passioID: PassioID,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            onResult(null)
            return
        }

        upcService!!.getFoodItemForPassioID(
            passioID,
            BuildConfig.DEV_API,
            languageCode
        ) { foodItem ->
            mainExecutor.execute {
                onResult(foodItem)
            }
        }
    }

    override fun fetchSuggestions(
        mealTime: PassioMealTime,
        callback: (results: List<PassioFoodDataInfo>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(emptyList())
            return
        }

        executor.execute {
            SuggestionService.fetchSuggestions(
                mealTime,
                BuildConfig.DEV_API,
                languageCode
            ) { list ->
                val searchResults = list.map {
                    PassioFoodDataInfo(
                        it.refCode,
                        it.shortName,
                        it.brandName,
                        it.iconId,
                        it.score,
                        it.scoredName,
                        it.labelId,
                        it.type,
                        it.resultId,
                        true,
                        it.nutritionPreview.toDataModel(),
                        it.tags,
                    )
                }
                mainExecutor.execute {
                    callback(searchResults)
                }
            }
        }
    }

    override fun fetchMealPlans(callback: (result: List<PassioMealPlan>) -> Unit) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(emptyList())
            return
        }

        executor.execute {
            MealPlanService.fetchMealPlans(BuildConfig.DEV_API) { mealPlans ->
                mainExecutor.execute {
                    callback(mealPlans)
                }
            }
        }
    }

    override fun fetchMealPlanForDay(
        mealPlanLabel: String,
        day: Int,
        callback: (result: List<PassioMealPlanItem>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(emptyList())
            return
        }

        executor.execute {
            MealPlanService.fetchDayOf(
                mealPlanLabel,
                day,
                BuildConfig.DEV_API,
                languageCode
            ) { items ->
                mainExecutor.execute {
                    callback(items)
                }
            }
        }
    }

    override fun recognizeSpeechRemote(
        text: String,
        callback: (result: List<PassioSpeechRecognitionModel>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(emptyList())
            return
        }

        executor.submit {
            SpeechService.fetchSpeechRecognitionResult(
                text,
                BuildConfig.DEV_API,
                languageCode
            ) { result ->
                mainExecutor.execute {
                    callback(result)
                }
            }
        }
    }

    override fun fetchFoodItemForRefCode(
        refCode: String,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(null)
            return
        }

        SearchService.fetchFoodItemForRefCode(
            refCode,
            BuildConfig.DEV_API,
            languageCode
        ) { foodItem ->
            mainExecutor.execute {
                callback(foodItem)
            }
        }
    }

    override fun startNutritionFactsDetection(
        listener: NutritionFactsRecognitionListener
    ): Boolean {
        if (passioRecognizer == null) {
            PassioLog.e(
                PassioSDK::class.java.simpleName,
                "Could not start nutrition facts detection, configure not called"
            )
            return false
        }

        passioRecognizer!!.setNutritionFactsListener(listener)
        return true
    }

    override fun stopNutritionFactsDetection() {
        passioRecognizer?.setNutritionFactsListener(null)
    }

    override fun fetchFoodItemLegacy(
        passioID: PassioID,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (passioID.startsWith("barcode")) {
            val id = passioID.removePrefix("barcode")
            fetchFoodItemForProductCode(id, callback)
            return
        }

        if (passioID.startsWith("packagedFoodCode")) {
            val id = passioID.removePrefix("packagedFoodCode")
            fetchFoodItemForProductCode(id, callback)
            return
        }

        fetchFoodItemForPassioID(passioID, callback)
    }

    override fun recognizeNutritionFactsRemote(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(null)
            return
        }

        executor.execute {
            advisorService.runPackageFoodRecognition(bitmap, resolution) { foodItem ->
                mainExecutor.execute {
                    callback(foodItem)
                }
            }
        }
    }

    override fun fetchHiddenIngredients(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(PassioResult.Error("SDK isn't configured"))
            return
        }

        executor.execute {
            advisorService.fetchHiddenIngredients(foodName, languageCode) { response ->
                mainExecutor.execute {
                    callback(response)
                }
            }
        }
    }

    override fun fetchVisualAlternatives(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(PassioResult.Error("SDK isn't configured"))
            return
        }

        executor.execute {
            advisorService.fetchVisualAlternatives(foodName, languageCode) { response ->
                mainExecutor.execute {
                    callback(response)
                }
            }
        }
    }

    override fun fetchPossibleIngredients(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    ) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            PassioLog.e(this::class.java.simpleName, "SDK isn't configured")
            callback(PassioResult.Error("SDK isn't configured"))
            return
        }
        executor.execute {
            advisorService.fetchPossibleIngredients(foodName, languageCode) { response ->
                mainExecutor.execute {
                    callback(response)
                }
            }
        }
    }

    override fun setAccountListener(listener: PassioAccountListener?) {
        executor.execute {
            tokenUsageListener = listener
        }
    }

    override fun getPassioStatus(): PassioStatus {
        return passioStatusRef.get()
    }

    override fun updateLanguage(languageCode: String): Boolean {
        if (languageCode.length != 2) {
            PassioLog.e(
                this::class.java.simpleName,
                "Could not set language code, only 2 character codes (ISO 639-1) are accepted. Please use locale.language()"
            )
            return false
        }

        try {
            Locale(languageCode).isO3Language
        } catch (e: Exception) {
            PassioLog.e(this::class.java.simpleName, "No known locale code: $languageCode")
            return false
        }

        this.languageCode = languageCode
        return true
    }

    override fun reportFoodItem(
        refCode: String,
        productCode: String,
        notes: List<String>?,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        UserService.reportFoodItem(refCode, productCode, notes, BuildConfig.DEV_API) { result ->
            mainExecutor.execute {
                callback(result)
            }
        }
    }

    override fun submitUserCreatedFoodItem(
        foodItem: PassioFoodItem,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        UserService.submitUserFood(foodItem, BuildConfig.DEV_API) { result ->
            mainExecutor.execute {
                callback(result)
            }
        }
    }

    override fun predictNextIngredients(
        currentIngredients: List<String>,
        callback: (result: List<PassioFoodDataInfo>) -> Unit
    ) {
        SearchService.fetchPredictNextIngredients(
            currentIngredients,
            BuildConfig.DEV_API,
            languageCode
        ) { response ->
            mainExecutor.execute {
                 callback(response)
            }
        }
    }

    private fun barcodeAsync(barcode: String): CompletableFuture<PassioFoodItem?> {
        val future = CompletableFuture<PassioFoodItem?>()
        upcService?.getFoodItemForProductCode(
            barcode,
            BuildConfig.DEV_API,
            languageCode
        ) { foodItem ->
            future.complete(foodItem)
        }
        return future
    }

    private fun barcodeRecognition(
        bitmap: Bitmap,
        callback: (result: List<PassioAdvisorFoodInfo>) -> Unit
    ) {
        val config = FoodDetectionConfiguration(detectVisual = false, detectBarcodes = true)
        passioRecognizer!!.detectFoodIn(bitmap, config) { candidates ->
            executor.execute {
                if (candidates != null && candidates.barcodeCandidates!!.isNotEmpty()) {
                    val futures = candidates.barcodeCandidates.map {
                        barcodeAsync(it.barcode)
                    }
                    val combinedFuture =
                        CompletableFuture.allOf(*futures.toTypedArray()).thenApply {
                            futures.map { it.join() }.mapNotNull { fi ->
                                if (fi == null) {
                                    null
                                } else {
                                    PassioAdvisorFoodInfo(
                                        fi.name,
                                        "${fi.amount.selectedQuantity} ${fi.amount.selectedUnit}",
                                        fi.ingredientWeight().gramsValue(),
                                        null,
                                        fi,
                                        PassioFoodResultType.BARCODE
                                    )
                                }
                            }
                        }
                    val combinedResult = combinedFuture.join()
                    callback(combinedResult)
                } else {
                    callback(emptyList())
                }
            }
        }
    }

    private fun nutriFactsRecognition(
        bitmap: Bitmap,
        callback: (result: List<PassioAdvisorFoodInfo>) -> Unit
    ) {
        passioRecognizer!!.runJustOCR(bitmap) { text ->
            executor.execute {
                if (text.isEmpty()) {
                    callback(emptyList())
                    return@execute
                }

                if (isTextNutritionFact(text.toLowerCase())) {
                    advisorService.runPackageFoodRecognition(
                        bitmap,
                        PassioImageResolution.RES_1080
                    ) { fi ->
                        if (fi == null) {
                            callback(emptyList())
                            return@runPackageFoodRecognition
                        }

                        val info = PassioAdvisorFoodInfo(
                            fi.name,
                            "${fi.amount.selectedQuantity} ${fi.amount.selectedUnit}",
                            fi.ingredientWeight().gramsValue(),
                            null,
                            fi,
                            PassioFoodResultType.NUTRITION_FACTS
                        )

                        callback(listOf(info))
                    }
                } else {
                    callback(emptyList())
                    return@execute
                }
            }
        }
    }

    override fun recognizeImageRemote(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        message: String?,
        callback: (result: List<PassioAdvisorFoodInfo>) -> Unit
    ) {
        executor.execute toplevel@{
            if (passioRecognizer == null) {
                mainExecutor.execute {
                    callback(emptyList())
                }
            }
            barcodeRecognition(bitmap) { barcodeResult ->
                if (barcodeResult.isNotEmpty()) {
                    mainExecutor.execute {
                        callback(barcodeResult)
                    }
                    return@barcodeRecognition
                }

                nutriFactsRecognition(bitmap) { nutriFactsResult ->
                    if (nutriFactsResult.isNotEmpty()) {
                        mainExecutor.execute {
                            callback(nutriFactsResult)
                        }
                        return@nutriFactsRecognition
                    }

                    advisorService.runImageRecognition(
                        bitmap,
                        resolution,
                        message,
                        languageCode
                    ) { result ->
                        mainExecutor.execute {
                            callback(result)
                        }
                    }
                }
            }
        }
    }

    private fun isTextNutritionFact(text: String): Boolean {
        val keywords = listOf("calories", "carbs", "protein", "fat", "serving", "ingredients")
        var score = 0
        val words = text.replace("\n", " ").split(" ")
        words.forEach { word ->
            if (word in keywords) {
                score++
                if (score >= 3) {
                    return true
                }
            }
        }
        return false
    }

    override fun changeBoundingBoxFilter(mode: Int?): Boolean {
        if (passioRecognizer == null) {
            return false
        }
        passioRecognizer!!.changeBoundingBoxFilter(mode)
        return true
    }

    override fun setPassioStatusListener(statusListener: PassioStatusListener?) {
        this.statusListener = statusListener
    }

    override fun boundingBoxToViewTransform(
        boundingBox: RectF,
        viewWidth: Int,
        viewHeight: Int,
        displayAngle: Int,
        barcode: Boolean
    ): RectF {
        return if (barcode) {
            val frameRect = RectF(
                boundingBox.left * cameraProxy.getFrameSize().width,
                boundingBox.top * cameraProxy.getFrameSize().height,
                boundingBox.right * cameraProxy.getFrameSize().width,
                boundingBox.bottom * cameraProxy.getFrameSize().height
            )

            val transform = ImageUtils.getTransformationMatrix(
                cameraProxy.getFrameSize().height,
                cameraProxy.getFrameSize().width,
                viewWidth,
                viewHeight,
                0,
                false
            )

            val resultRectF = RectF()
            transform.mapRect(resultRectF, frameRect)
            resultRectF
        } else {
            val transformMatrix = ImageUtils.getTransformationMatrix(
                1,
                1,
                cameraProxy.getFrameSize().width,
                cameraProxy.getFrameSize().height,
                0,
                false
            )
            val frameRect = RectF()
            transformMatrix.mapRect(frameRect, boundingBox)

            val transform = ImageUtils.getTransformationMatrix(
                cameraProxy.getFrameSize().width,
                cameraProxy.getFrameSize().height,
                viewWidth,
                viewHeight,
                (cameraProxy.getSensorRotation() ?: 0) - displayAngle,
                false
            )

            val resultRectF = RectF()
            transform.mapRect(resultRectF, frameRect)

            resultRectF
        }
    }

    override fun iconURLFor(passioID: PassioID, size: IconSize): String {
        return iconService.getIconURL(passioID, size)
    }

    override fun fetchTagsFor(refCode: String, onTagsFetched: (tags: List<String>?) -> Unit) {
        SearchService.fetchTagsForRefCode(refCode, BuildConfig.DEV_API) { tags ->
            mainExecutor.execute {
                onTagsFetched(tags)
            }
        }
    }

    override fun fetchInflammatoryEffectData(
        refCode: String,
        onResult: (nutrients: List<InflammatoryEffectData>?) -> Unit
    ) {
        SearchService.fetchNutrientsForRefCode(refCode, BuildConfig.DEV_API) { nutrients ->
            mainExecutor.execute {
                onResult(nutrients)
            }
        }
    }

    override fun searchForFood(
        term: String,
        callback: (result: List<PassioFoodDataInfo>, suggestions: List<String>) -> Unit
    ) {
        SearchService.fetchSearchResult(term, BuildConfig.DEV_API, languageCode) { response ->
            if (response == null) {
                mainExecutor.execute {
                    callback(emptyList(), emptyList())
                }
                return@fetchSearchResult
            }

            val searchResults = response.results.map {
                PassioFoodDataInfo(
                    it.refCode,
                    it.displayName,
                    it.brandName,
                    it.iconId,
                    it.score,
                    it.scoredName,
                    it.labelId,
                    it.type,
                    it.resultId,
                    false,
                    it.nutritionPreview.toDataModel(),
                    it.tags
                )
            }
            mainExecutor.execute {
                callback(searchResults, response.alternateNames)
            }
        }
    }

    override fun searchForFoodSemantic(
        term: String,
        callback: (result: List<PassioFoodDataInfo>, suggestions: List<String>) -> Unit
    ) {
        SearchService.fetchSemanticSearchResult(term, BuildConfig.DEV_API, languageCode) { response ->
            if (response == null) {
                mainExecutor.execute {
                    callback(emptyList(), emptyList())
                }
                return@fetchSemanticSearchResult
            }

            val searchResults = response.results.map {
                PassioFoodDataInfo(
                    it.refCode,
                    it.displayName,
                    it.brandName,
                    it.iconId,
                    it.score,
                    it.scoredName,
                    it.labelId,
                    it.type,
                    it.resultId,
                    false,
                    it.nutritionPreview.toDataModel(),
                    it.tags
                )
            }
            mainExecutor.execute {
                callback(searchResults, response.alternateNames)
            }
        }
    }

    override fun fetchFoodItemForDataInfo(
        dataInfo: PassioFoodDataInfo,
        servingQuantity: Double?,
        servingUnit: String?,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val labelId = if (dataInfo.type == "reference") {
            "00000000-0000-0000-0000-000000000000"
        } else {
            dataInfo.labelId
        }
        SearchService.fetchFoodItem(
            labelId,
            dataInfo.type,
            dataInfo.resultId,
            dataInfo.isShortName,
            BuildConfig.DEV_API,
            languageCode
        ) { response ->
            if (servingUnit == null && servingQuantity != null) {
                response?.setWeightGrams(servingQuantity)
            } else if (servingUnit != null && servingQuantity != null) {
                response?.setQuantityAndUnit(servingQuantity, servingUnit)
            }
            mainExecutor.execute {
                callback(response)
            }
        }
    }

    private fun Map<Uri, Pair<Boolean, String>>.isSuccessful(): Boolean {
        this.entries.forEach { entry ->
            if (!entry.value.first) {
                return false
            }
        }

        return true
    }

    private fun Map<Uri, Pair<Boolean, String>>.getErrorMessage(): String {
        var errorMessage = ""
        this.entries.forEach { entry ->
            if (!entry.value.first) {
                errorMessage += "File ${entry.key} update unsuccessful, errorMessage: ${entry.value.second}\n"
            }
        }
        if (errorMessage.isNotEmpty()) {
            errorMessage = errorMessage.substring(0, errorMessage.length - 1)
        }
        return errorMessage
    }

    @WorkerThread
    override fun initializeSDKWithExternalModels(
        context: Context,
        version: Int
    ): PassioStatus {

        val status = PassioStatus()
        val missingModels = mutableListOf<String>()
        passioFileManager.copyFromTempToPassioWithPrune(context)

        // If the version is 0, that means no files were delivered and the SDK will try to
        // initialize the current SDK version of the files
        val deliveredVersion = if (version == 0) {
            FILE_VERSION
        } else {
            version
        }

        try {
            val hash = metadataManager.configure(
                context,
                useAssets = false,
                remoteOnly = false,
                dev = BuildConfig.DEV_API
            )
            PassioLog.i(this::class.java.simpleName, "Metadata labels configured")
            if (hash == null) {
                missingModels.add("${passio_metadata.name}.$deliveredVersion")
                val passioStatus =
                    PassioStatus.error(PassioSDKError.NO_MODELS_FILES_FOUND, "No metadata found")
                passioStatus.missingFiles = missingModels
                return passioStatus
            }
            status.metadataHash = hash
        } catch (e: Exception) {
            val passioStatus =
                PassioStatus.error(PassioSDKError.METADATA_ERROR, e.message ?: "")
            passioStatus.missingFiles = missingModels
            return passioStatus
        }

        upcService = UPCService(allowInternet)

        try {
            val missingRecognitionModels =
                if (passioFileManager.isSDKUsingCompressedModels(context)) {
                    passioRecognizer!!.initializeFromCompressedExternalModels(
                        context,
                        deliveredVersion
                    )
                } else if (passioFileManager.isSDKUsingSecuredModels(context)) {
                    passioRecognizer!!.initializeFromSecuredExternalModels(
                        context,
                        deliveredVersion
                    )
                } else {
                    passioRecognizer!!.initializeFromExternalModels(
                        context,
                        deliveredVersion
                    )
                }

            if (missingRecognitionModels.isEmpty()) {
                status.mode = PassioMode.IS_READY_FOR_DETECTION
                status.activeModels = deliveredVersion
            } else {
                status.error = PassioSDKError.NO_MODELS_FILES_FOUND
                missingModels.addAll(missingRecognitionModels)
                status.activeModels = null
            }

            val allFiles = passioFileManager.getAvailableFileTypes().map {
                "${it.name}.$FILE_VERSION.passiosecure"
            }
            if (version < FILE_VERSION) {
                status.missingFiles = allFiles
            } else if (missingModels.isNotEmpty()) {
                status.missingFiles = missingModels
            }
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }

        return status
    }

    override fun initializeRemoteOnly(context: Context): PassioStatus {
        val status = PassioStatus()
//        try {
//            val hash = metadataManager.configure(context,
//                useAssets = false,
//                remoteOnly = true,
//                dev = BuildConfig.DEV_API
//            )
//            PassioLog.i(this::class.java.simpleName, "Metadata labels configured")
//            if (hash == null) {
//                val passioStatus =
//                    PassioStatus.error(PassioSDKError.METADATA_ERROR, "No metadata found")
//                return passioStatus
//            }
//            status.metadataHash = hash
//        } catch (e: Exception) {
//            val passioStatus =
//                PassioStatus.error(PassioSDKError.METADATA_ERROR, e.message ?: "")
//            return passioStatus
//        }

        passioRecognizer!!.initializeRemoteOnly(context)
        upcService = UPCService(allowInternet)
        status.mode = PassioMode.IS_READY_FOR_DETECTION
        status.activeModels = 2
        return status
    }

    private var downloadedFiles = mutableListOf<Uri>()

    override fun downloadFiles(context: Context, version: Int, files: List<String>) {
        downloadManager.downloadFiles(context, version, ArrayList(files))
        downloadedFiles.clear()
    }

    override fun onFileDownloaded(fileUri: Uri, filesDownloaded: Int, filesToDownload: Int) {
        downloadedFiles.add(fileUri)
        mainExecutor.execute {
            statusListener?.onCompletedDownloadingFile(fileUri, filesToDownload - filesDownloaded)
        }
    }

    override fun onDownloadError(errorMessage: String) {
        mainExecutor.execute {
            statusListener?.onDownloadError(errorMessage)
        }
    }

    override fun onDownloadComplete() {
        mainExecutor.execute {
            statusListener?.onCompletedDownloadingAllFiles(downloadedFiles)
        }

        executor.submit {
            val context = contextRef?.get()

            if (context == null) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Context is null, can't continue with update"
                )
                return@submit
            }

            val isDownloadSuccess = passioFileManager.getDownloadedVersion(context) == FILE_VERSION

            if (!isDownloadSuccess && !isSDKReady()) {
                val requiredFiles = passioFileManager.getAvailableFileTypes()
                val missingFiles = passioFileManager.getFilesMissingFromFolder(
                    context,
                    PassioFileManager.CACHE_DOWNLOAD,
                    requiredFiles
                )
                val status = PassioStatus().apply {
                    debugMessage = "Files are missing, can't configure"
                    this.missingFiles = missingFiles
                    mode = PassioMode.FAILED_TO_CONFIGURE
                    error = PassioSDKError.MODELS_DOWNLOAD_FAILED
                }
                updatePassioStatus(status)
                mainExecutor.execute {
                    configurationCompletion.invoke(status)
                }
                return@submit
            }

            passioFileManager.transferFromDownloadToTemp(context, listOf(passio_dataset))
            sharedPreferencesManager.setAutoUpdateRan()

            if (!isSDKReady()) {
                val status = initializeSDKWithExternalModels(context, FILE_VERSION)
                updatePassioStatus(status)
                passioRecognizer!!.onPassioStatus(status)
                mainExecutor.execute {
                    configurationCompletion.invoke(status)
                }
            }
        }
    }

    @WorkerThread
    override fun initializeSDKWithAssetModels(
        context: Context,
        version: Int
    ): PassioStatus {
        val status = passioStatusRef.get()
        val missingModels = mutableListOf<String>()
        val requiredFiles = passioFileManager.getAvailableFileTypes()
        val requiredFileNames = requiredFiles.map {
            "${it.name}.$FILE_VERSION.passiosecure"
        }
        try {
            val hash = metadataManager.configure(
                context,
                useAssets = true,
                remoteOnly = false,
                dev = BuildConfig.DEV_API
            )
            PassioLog.i(this::class.java.simpleName, "Metadata labels configured")
            if (hash == null) {
                missingModels.add("${passio_metadata.name}.$FILE_VERSION")
                val passioStatus =
                    PassioStatus.error(PassioSDKError.NO_MODELS_FILES_FOUND, "No metadata found")
                passioStatus.missingFiles = missingModels
                return passioStatus
            }
            status.metadataHash = hash
        } catch (e: Exception) {
            val passioStatus =
                PassioStatus.error(PassioSDKError.METADATA_ERROR, e.message ?: "")
            passioStatus.missingFiles = missingModels
            return passioStatus
        }

        upcService = UPCService(allowInternet)

        val assetFiles = passioFileManager.getFilesFromAssets(context, requiredFiles)
        // Model loading
        val missingFiles = passioRecognizer!!.initializeFromAssetModels(
            context,
            version,
            assetFiles
        )

        if (missingFiles.isEmpty()) {
            status.mode = PassioMode.IS_READY_FOR_DETECTION
            status.activeModels = version
        } else {
            status.mode = PassioMode.FAILED_TO_CONFIGURE
            status.error = PassioSDKError.MODELS_NOT_VALID
            status.activeModels = null
        }
        if (version < FILE_VERSION) {
            status.missingFiles = requiredFileNames
        } else if (missingFiles.isNotEmpty()) {
            status.missingFiles = missingFiles
        }

        return status
    }

    protected fun finalize() {
        downloadManager.onStop()
        executor.shutdown()
    }

    override fun lookupAllSiblingsFor(passioID: PassioID): List<PassioID>? {
//        if (!isDBReady()) {
//            return null
//        }
//
//        val item = dbHelper!!.getPassioIDAttributesFor(passioID) ?: return null
//        return item.siblings?.map { it.passioID }
        return null
    }

    override fun onTokensUsed(usage: PassioTokenBudget) {
        if (tokenUsageListener == null) return
        if (!::mainExecutor.isInitialized) return
        mainExecutor.execute {
            tokenUsageListener?.onTokenBudgetUpdate(usage)
        }
    }

    override fun initConversation(callback: (result: PassioResult<Any>) -> Unit) {
        if (passioStatusRef.get().mode != PassioMode.IS_READY_FOR_DETECTION) {
            callback(PassioResult.Error("PassioSDK must be configured before accessing NutritionAdvisor"))
            return
        }

        advisorService.initConversation {
            mainExecutor.execute {
                callback(it)
            }
        }
    }

    override fun sendMessage(
        message: String,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        advisorService.sendMessage(message) {
            mainExecutor.execute {
                callback(it)
            }
        }
    }

    override fun sendImage(
        bitmap: Bitmap,
        resolution: PassioImageResolution,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        advisorService.sendImage(bitmap, resolution, languageCode) {
            mainExecutor.execute {
                callback(it)
            }
        }
    }

    override fun fetchIngredients(
        response: PassioAdvisorResponse,
        callback: (response: PassioResult<PassioAdvisorResponse>) -> Unit
    ) {
        advisorService.fetchIngredients(response, languageCode) {
            mainExecutor.execute {
                callback(it)
            }
        }
    }

    override fun setAdvisorProfile(
        profileJson: String,
        callback: (result: PassioResult<Any>) -> Unit
    ) {
        advisorService.setAdvisorProfile(profileJson) {
            mainExecutor.execute {
                callback(it)
            }
        }
    }
}

internal fun NutritionPreviewResult.toDataModel(): PassioSearchNutritionPreview {
    val servingUnit = portion.name ?: ""
    val servingQuantity = if (portion.suggestedQuantity != null) {
        portion.suggestedQuantity!!.first()
    } else {
        portion.quantity ?: 0.0
    }
    val weightUnit = portion.weight?.unit ?: ""
    val weightQuantity = (portion.weight?.value ?: 0.0) * servingQuantity

    return PassioSearchNutritionPreview(
        calories.toInt(),
        carbs,
        protein,
        fat,
        fiber,
        servingUnit,
        servingQuantity,
        weightUnit,
        weightQuantity
    )
}