package ai.passio.passiosdk.core.report

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.sharedpreferences.ReportPreferencesManager
import ai.passio.passiosdk.core.utils.PassioLog
import android.content.Context
import org.json.JSONObject
import java.util.Calendar
import java.util.Date
import java.util.TimeZone

internal class ReportingService(context: Context, private val isEnabled: Boolean) {

    private val prefs = ReportPreferencesManager(context)
    private var reportedToday = mutableMapOf<ActivationEvent, Boolean>()
    private var inProgress = mutableMapOf<ActivationEvent, Boolean>()

    fun reportSDKUsage(
        metadata: JSONObject?,
        event: ActivationEvent,
        dev: Boolean
    ) {
        if (!isEnabled) {
            return
        }

        if (reportedToday[event] == true || metadata == null) {
            return
        }

        if (inProgress[event] == true) {
            return
        }

        val lastReportedTime = prefs.getLastReported(event.eventName)
        val lastReportedDate = Date(lastReportedTime)
        val lastReportedCal = Calendar.getInstance(TimeZone.getTimeZone("Europe/Paris"))
        lastReportedCal.time = lastReportedDate
        val lastReportedYear = lastReportedCal.get(Calendar.YEAR)
        val lastReportedDay = lastReportedCal.get(Calendar.DAY_OF_YEAR)

        val currentTime = System.currentTimeMillis()
        val currentDate = Date(currentTime)
        val currentCal = Calendar.getInstance(TimeZone.getTimeZone("Europe/Paris"))
        currentCal.time = currentDate
        val currentYear = currentCal.get(Calendar.YEAR)
        val currentDay = currentCal.get(Calendar.DAY_OF_YEAR)

        if (lastReportedYear == currentYear && lastReportedDay == currentDay) {
            reportedToday[event] = true
            return
        }

        inProgress[event] = true
        metadata.put("event", event.eventName)
        prepareEvent(metadata.toString(), event, dev)
    }

    private fun prepareEvent(
        metadata: String,
        event: ActivationEvent,
        dev: Boolean
    ) {
        TokenService.getInstance().getToken(dev,
            { token ->
                sendEvent(metadata, token, event, dev)
            }, { errorMessage ->
                PassioLog.w(ReportingService::class.java.simpleName, "Error retrieving token")
                inProgress[event] = false
            })
    }

    private fun sendEvent(
        metadata: String,
        token: String,
        event: ActivationEvent,
        dev: Boolean
    ) {
        val headers = mapOf(
            "Content-Type" to "application/json",
            "Authorization" to token
        )
        NetworkService.instance.doRequest(
            PostJsonNetworkTask(NativeUtils.instance.getUsageURL(dev), headers, metadata),
            object : NetworkCallback<Boolean> {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.w(
                        ReportingService::class.java.simpleName,
                        "Activation ${event.eventName} failed with: $message"
                    )
                    inProgress[event] = false
                }

                override fun onSuccess(result: Boolean) {
                    PassioLog.d(ReportingService::class.java.simpleName, "Event sent")
                    inProgress[event] = false
                    if (result) {
                        prefs.setLastReported(event.eventName, System.currentTimeMillis())
                        reportedToday[event] = true

                    }
                }

                override fun onTokenExpired() {
                    prepareEvent(metadata, event, dev)
                }
            })
    }
}