package ai.passio.passiosdk.passiofood.recognition

import ai.passio.passiosdk.core.annotation.CameraThread
import ai.passio.passiosdk.core.annotation.PassioThread
import ai.passio.passiosdk.core.camera.PassioCameraProxy
import ai.passio.passiosdk.core.config.PassioMode
import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.core.config.SDKFileType
import ai.passio.passiosdk.core.utils.BitmapUtil
import ai.passio.passiosdk.core.utils.DimenUtil
import ai.passio.passiosdk.core.utils.ImageUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.*
import ai.passio.passiosdk.passiofood.file.PassioFoodFileManager
import ai.passio.passiosdk.passiofood.metadata.MetadataManager
import ai.passio.passiosdk.passiofood.mlkit.BarcodeDetectorFactory
import ai.passio.passiosdk.passiofood.mlkit.BarcodeProxy
import ai.passio.passiosdk.passiofood.mlkit.TextRecognitionFactory
import ai.passio.passiosdk.passiofood.nutritionfacts.NutritionFactsReader
import ai.passio.passiosdk.passiofood.nutritionfacts.PassioNutritionFacts
import ai.passio.passiosdk.passiofood.time.VotingFrameAveraging
import ai.passio.passiosdk.passiofood.utils.AggregateHelper
import ai.passio.passiosdk.passiofood.voting.*
import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Matrix
import android.graphics.RectF
import android.util.Log
import androidx.camera.core.ImageProxy
import com.google.mlkit.vision.common.InputImage
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicReference

@Suppress("PrivatePropertyName")
internal open class PassioRecognizer(
    private val cameraProxy: PassioCameraProxy,
    private val mainExecutor: Executor,
    private val fileManager: PassioFoodFileManager,
    private val labelManager: MetadataManager,
    private val remoteOnly: Boolean,
) : PassioCameraProxy.PassioCameraListener {

    companion object {
        internal const val TF_OD_API_INPUT_SIZE = 360
        internal const val TF_HNN_API_INPUT_SIZE = 260
        internal const val TF_YOLO_INPUT_SIZE = 320
    }

    private var isReady = AtomicBoolean(false)
    private val recognitionQueue: Executor by lazy { Executors.newSingleThreadExecutor() }

    private var frameWidth = 0
    private var frameHeight = 0
    private var orientation = 0

    private var objectDetectionListener = AtomicReference<ObjectDetectionListener>()
    private var classificationListener = AtomicReference<ClassificationListener>()
    private var foodDetectionListener = AtomicReference<FoodRecognitionListener>()
    private var nutritionFactsListener = AtomicReference<NutritionFactsRecognitionListener?>(null)

    private var isFoodEnabled = AtomicBoolean(false)
    private var isBarcodeEnabled = AtomicBoolean(false)
    private var isOCREnabled = AtomicBoolean(false)

    private val frameAveraging = VotingFrameAveraging()
    private var frameAveragingRef = AtomicReference<VotingFrameAveraging?>(null)

    private val resourcesAnalyzingImage = AtomicInteger(0)

    private val yuvBytes = DimenUtil.init2DByteArray(3, null)
    private var rgbBytes: IntArray? = null

    private var objectDetectionStrategy: ObjectDetectionStrategy? = BiggerBoundingBoxStrategy()

    private val croppedODBitmap: Bitmap by lazy {
        Bitmap.createBitmap(
            modelHolder.getObjectDetectionInputSize(),
            modelHolder.getObjectDetectionInputSize(),
            Bitmap.Config.ARGB_8888
        )
    }
    private val croppedClassBitmap: Bitmap = Bitmap.createBitmap(
        TF_HNN_API_INPUT_SIZE,
        TF_HNN_API_INPUT_SIZE,
        Bitmap.Config.ARGB_8888
    )

    private val frameToODTransform: Matrix by lazy {
        ImageUtils.getTransformationMatrix(
            frameWidth,
            frameHeight,
            modelHolder.getObjectDetectionInputSize(),
            modelHolder.getObjectDetectionInputSize(),
            orientation,
            false
        )
    }

    private val ODToFrameTransform: Matrix by lazy {
        val mat = Matrix()
        frameToODTransform.invert(mat)
        mat
    }

    private lateinit var frameToClassTransform: Matrix

    protected lateinit var modelHolder: PassioModelHolder
    protected val votingLayer: VotingLayer = HNNKNNVotingLayer()
    private val nutritionFactsReader = NutritionFactsReader()

    @PassioThread
    fun initializeFromCompressedExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        try {
            modelHolder = PassioExternalCompressedModelHolder(labelManager, fileManager)
            return modelHolder.initializeModels(context, version)
        } catch (e: Exception) {
            return emptyList()
        }
    }

    @PassioThread
    fun initializeFromSecuredExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioExternalSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    @PassioThread
    fun initializeFromExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioExternalModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    @PassioThread
    fun initializeFromAssetModels(
        context: Context,
        version: Int,
        assetFiles: List<String>
    ): List<String> {
        return if (assetFiles[0].endsWith("passiosecure2")) {
            initializeFromAssetCompressedModels(
                context,
                version
            )
        } else if (assetFiles[0].endsWith("passiosecure")) {
            initializeFromAssetSecuredModels(
                context,
                version
            )
        } else {
            initializeFromAssetModels(
                context,
                version
            )
        }
    }

    @PassioThread
    fun initializeRemoteOnly(context: Context) {
        modelHolder = object : PassioModelHolder(labelManager, fileManager) {

            override fun initializeModels(context: Context, version: Int): List<String> {
                barcodeDetector = BarcodeDetectorFactory.create()
                ocrDetector = TextRecognitionFactory.getTextRecognizer()
                return emptyList()
            }

            override fun initializeModel(
                context: Context,
                fileType: SDKFileType,
                version: Int
            ): Boolean {
                throw IllegalAccessException("Should not be invoked")
            }

            override fun getExtension(): String = ""
        }
        modelHolder.initializeModels(context, 0)
    }

    private fun initializeFromAssetCompressedModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetCompressedSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    private fun initializeFromAssetSecuredModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    private fun initializeFromAssetModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    fun onPassioStatus(status: PassioStatus) {
        isReady.set(status.mode == PassioMode.IS_READY_FOR_DETECTION)
    }

    fun detectFoodIn(
        bitmap: Bitmap,
        detectionConfig: FoodDetectionConfiguration?,
        onDetectionCompleted: (candidates: FoodCandidates?) -> Unit
    ) {
        if (!isReady.get()) {
            onDetectionCompleted(null)
            return
        }

        recognitionQueue.execute {
            val shouldScanFood = detectionConfig?.detectVisual ?: true
            val shouldScanBarcode = detectionConfig?.detectBarcodes ?: false
            val shouldScanOCR = detectionConfig?.detectPackagedFood ?: false

            val numOfDetections =
                shouldScanFood.toInt() +
                        shouldScanBarcode.toInt() +
                        shouldScanOCR.toInt()

            val listener = object : FoodRecognitionListener {
                override fun onRecognitionResults(
                    candidates: FoodCandidates?,
                    image: Bitmap?,
                ) {
                    onDetectionCompleted(candidates)
                }
            }
            val listenerRef = AtomicReference<FoodRecognitionListener>(listener)

            val aggregateHelper = AggregateHelper(
                bitmap,
                numOfDetections,
                mainExecutor,
                listenerRef
            )

            if (shouldScanFood) {
                val frameToODTransform = ImageUtils.getTransformationMatrix(
                    bitmap.width,
                    bitmap.height,
                    modelHolder.getObjectDetectionInputSize(),
                    modelHolder.getObjectDetectionInputSize(),
                    0,
                    false
                )

                val oDToFrameTransform = Matrix()
                frameToODTransform.invert(oDToFrameTransform)

                if (modelHolder.objectDetector == null || modelHolder.hnnDetector == null) {
                    aggregateHelper.onFoodCandidates(listOf(), listener)
                } else {
                    runVoting(
                        bitmap,
                        false,
                        frameToODTransform,
                        oDToFrameTransform
                    ) { debugCandidates ->
                        val transformedResult = debugCandidates.mapNotNull {
                            val relativeBoundingBox = RectF(
                                it.boundingBox.left / bitmap.width,
                                it.boundingBox.top / bitmap.height,
                                it.boundingBox.right / bitmap.width,
                                it.boundingBox.bottom / bitmap.height
                            )
                            val transformedObjectDetection = ObjectDetectionCandidate(
                                it.odCandidate.passioID,
                                it.odCandidate.confidence,
                                relativeBoundingBox
                            )
                            val label = labelManager.getMappedPassioID(it.votedCandidate.passioID)
                                ?: return@mapNotNull null
                            DebugCandidate(
                                label.first,
                                label.second,
                                it.votedCandidate,
                                transformedObjectDetection,
                                it.hnnCandidate,
                                it.knnCandidate,
                                it.croppedImage
                            )
                        }

                        aggregateHelper.onFoodCandidates(transformedResult, listener)
                    }
                }
            }

            if (shouldScanBarcode) {
                if (modelHolder.barcodeDetector == null) {
                    aggregateHelper.onBarcodeCandidates(listOf(), listener)
                } else {
                    runBarcodeDetection(bitmap) { barcodes ->
                        // Prune barcodes
                        val filtered = barcodes.filter { it.boundingBox != null }

                        val transformedResult = filtered.map {
                            val relativeBoundingBox = RectF(
                                it.boundingBox!!.left.toFloat() / bitmap.width,
                                it.boundingBox!!.top.toFloat() / bitmap.height,
                                it.boundingBox!!.right.toFloat() / bitmap.width,
                                it.boundingBox!!.bottom.toFloat() / bitmap.height
                            )
                            BarcodeCandidate(it.value, relativeBoundingBox)
                        }
                        aggregateHelper.onBarcodeCandidates(transformedResult, listener)
                    }
                }
            }

            if (shouldScanOCR) {
                if (modelHolder.ocrDetector == null) {
                    aggregateHelper.onPackagedFoodCandidate(null, listener)
                } else {
                    runOCR(
                        bitmap,
                        true,
                        { packagedFoodCandidate, _ ->
                            aggregateHelper.onPackagedFoodCandidate(packagedFoodCandidate, listener)
                        }
                    )
                }
            }
        }
    }

    private fun runHNNDetection(
        image: Bitmap,
        onDetectionResult: (result: List<ClassificationCandidate>) -> Unit
    ) {
        val classCanvas = Canvas(croppedClassBitmap)
        classCanvas.drawBitmap(image, frameToClassTransform, null)

        val recognitionResult = modelHolder.hnnDetector!!.recognizeImage(croppedClassBitmap)

        onDetectionResult(recognitionResult)
    }

    private fun runBarcodeDetection(
        inputImage: InputImage,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        modelHolder.barcodeDetector!!.process(inputImage,
            { barcodes -> onResult(barcodes) },
            { onResult(listOf()) })
    }

    @SuppressLint("UnsafeOptInUsageError")
    protected fun runBarcodeDetection(
        imageProxy: ImageProxy,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        try {
            val image = InputImage.fromMediaImage(
                imageProxy.image!!,
                imageProxy.imageInfo.rotationDegrees
            )
            runBarcodeDetection(image, onResult)
        } catch (e: Exception) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            onResult(listOf())
        }
    }

    private fun runBarcodeDetection(
        bitmap: Bitmap,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        try {
            val image = InputImage.fromBitmap(bitmap, 0)
            runBarcodeDetection(image, onResult)
        } catch (e: java.lang.IllegalStateException) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            onResult(listOf())
        }
    }

    private fun runObjectDetection(
        image: Bitmap,
        frameToODMatrix: Matrix? = null,
        odToFrameMatrix: Matrix? = null,
        onDetectionResult: (result: List<ObjectDetectionCandidate>) -> Unit
    ) {
        val canvas = Canvas(croppedODBitmap)
        canvas.drawBitmap(image, frameToODMatrix ?: frameToODTransform, null)

        val recognitionResult = modelHolder.objectDetector!!.recognizeImage(croppedODBitmap)

        val filtered = recognitionResult.filter { result ->
            val validResult = result.confidence >= PassioSDK.minimumConfidence()
            if (validResult) {
                if (odToFrameMatrix != null) {
                    odToFrameMatrix.mapRect(result.boundingBox)
                } else {
                    ODToFrameTransform.mapRect(result.boundingBox)
                }

            }
            validResult
        }
        val strategyFiltered = objectDetectionStrategy?.process(filtered) ?: filtered

        onDetectionResult(strategyFiltered)
    }

    protected fun runVoting(
        image: Bitmap,
        shouldRunFilter: Boolean,
        frameToODMatrix: Matrix? = null,
        odToFrameMatrix: Matrix? = null,
        callback: (result: List<DebugCandidate>) -> Unit
    ) {
        runObjectDetection(image, frameToODMatrix, odToFrameMatrix) { objectDetectionResults ->
            if (objectDetectionResults.isEmpty()) {
                callback(listOf())
                return@runObjectDetection
            }

            val currentFrameAveraging = frameAveragingRef.get()
            val votingLayerResult =
                ArrayList<DebugCandidate>(objectDetectionResults.size)

            objectDetectionResults.forEach { objectDetectionCandidate ->
                val classRect = objectDetectionCandidate.boundingBox

                val normalizedRect =
                    ImageUtils.normalizeRectF(classRect, image.width, image.height)

                val croppedBitmap = Bitmap.createBitmap(
                    image,
                    normalizedRect.left,
                    normalizedRect.top,
                    normalizedRect.width(),
                    normalizedRect.height()
                )

                var targetBitmap = Bitmap.createScaledBitmap(
                    croppedBitmap,
                    TF_HNN_API_INPUT_SIZE,
                    TF_HNN_API_INPUT_SIZE,
                    true
                )

                if (frameToODMatrix == null) {
                    val matrix = Matrix()
                    matrix.postRotate(cameraProxy.getSensorRotation()?.toFloat() ?: 0f)

                    targetBitmap = Bitmap.createBitmap(
                        targetBitmap,
                        0,
                        0,
                        targetBitmap.width,
                        targetBitmap.height,
                        matrix,
                        false
                    )
                }

                val hnnCandidates =
                    modelHolder.hnnDetector!!.recognizeImage(targetBitmap, listOf(shouldRunFilter))
                        .filter {
                            it.passioID != PassioSDK.BKG_PASSIO_ID
                        }

                if (hnnCandidates.isEmpty()) {
                    return@forEach
                }

                if (currentFrameAveraging != null) {
                    PassioLog.i(this::class.java.simpleName, "Frame averaging on result")
                    currentFrameAveraging.onResult(
                        objectDetectionCandidate,
                        hnnCandidates.first(),
                        hnnCandidates.first(),
                        targetBitmap
                    )
                } else {
                    if (shouldRunFilter && hnnCandidates.size > 1) {
                        val hnnCandidate = hnnCandidates.first()
                        val confusionAlternatives = mutableListOf<DebugCandidate>()
                        for (i in 1 until hnnCandidates.size) {
                            val candidate = hnnCandidates[i]
                            val label =
                                labelManager.getMappedPassioID(candidate.passioID) ?: continue
                            val debug = DebugCandidate(
                                label.first,
                                label.second,
                                VotedCandidate(
                                    candidate.passioID,
                                    candidate.confidence,
                                    objectDetectionCandidate.boundingBox
                                ),
                                objectDetectionCandidate,
                                candidate,
                                candidate,
                                targetBitmap,
                            )
                            confusionAlternatives.add(debug)
                        }
                        val label =
                            labelManager.getMappedPassioID(hnnCandidate.passioID) ?: return@forEach
                        val contextAlternatives =
                            labelManager.getAlternatives(hnnCandidate.passioID)?.map {
                                DetectedCandidate(
                                    it.first,
                                    it.second,
                                    1f,
                                    RectF(),
                                    null,
                                    emptyList()
                                )
                            } ?: emptyList()
                        votingLayerResult.add(
                            DebugCandidate(
                                label.first,
                                label.second,
                                VotedCandidate(
                                    hnnCandidate.passioID,
                                    hnnCandidate.confidence,
                                    objectDetectionCandidate.boundingBox
                                ),
                                objectDetectionCandidate,
                                hnnCandidate,
                                hnnCandidate,
                                targetBitmap,
                                alternatives = confusionAlternatives + contextAlternatives
                            )
                        )
                    } else {
                        hnnCandidates.forEach loop@{ hnnCandidate ->
                            val label =
                                labelManager.getMappedPassioID(hnnCandidate.passioID) ?: return@loop
                            val alternatives =
                                labelManager.getAlternatives(hnnCandidate.passioID)?.map {
                                    DetectedCandidate(
                                        it.first,
                                        it.second,
                                        1f,
                                        RectF(),
                                        null,
                                        emptyList()
                                    )
                                } ?: emptyList()
                            votingLayerResult.add(
                                DebugCandidate(
                                    label.first,
                                    label.second,
                                    VotedCandidate(
                                        hnnCandidate.passioID,
                                        hnnCandidate.confidence,
                                        objectDetectionCandidate.boundingBox
                                    ),
                                    objectDetectionCandidate,
                                    hnnCandidate,
                                    hnnCandidate,
                                    targetBitmap,
                                    alternatives = alternatives
                                )
                            )
                        }
                    }
                }
            }


            val votingResult = currentFrameAveraging?.onFrameEnd(labelManager) ?: votingLayerResult
            callback(votingResult)
        }
    }

    private fun runOCR(
        inputImage: InputImage,
        runPackagedFoodDetection: Boolean = false,
        onOCRResult: (candidates: List<PackagedFoodCandidate>, text: String) -> Unit = { _, _ -> },
        runNutritionFactsDetection: Boolean = false,
        onNutritionFactsDetection: (facts: PassioNutritionFacts?, text: String) -> Unit = { _, _ -> }
    ) {
        if (!runPackagedFoodDetection && !runNutritionFactsDetection) {
            return
        }

        if (runPackagedFoodDetection && modelHolder.ocrMatcher == null) {
            onOCRResult(emptyList(), "")
            if (!runNutritionFactsDetection) {
                return
            }
        }

        modelHolder.ocrDetector!!.process(inputImage).addOnSuccessListener {
            recognitionQueue.execute {
                val recognizedText = it.text
                if (recognizedText.isEmpty()) {
                    if (runPackagedFoodDetection) {
                        onOCRResult(emptyList(), "")
                    }
                    if (runNutritionFactsDetection) {
                        onNutritionFactsDetection(null, "")
                    }
                    return@execute
                }

                if (runNutritionFactsDetection) {
                    onNutritionFactsDetection(
                        nutritionFactsReader.onTextRecognized(it),
                        recognizedText
                    )
                }

                if (runPackagedFoodDetection) {
                    val results = modelHolder.ocrMatcher!!.matchText(it.text)
                    if (results.isEmpty()) {
                        onOCRResult(emptyList(), recognizedText)
                    } else {
                        onOCRResult(results.map { pair ->
                            PackagedFoodCandidate(pair.first, pair.second)
                        }, recognizedText)
                    }
                }
            }
        }.addOnFailureListener {
            if (runPackagedFoodDetection) {
                onOCRResult(emptyList(), "")
            }
            if (runNutritionFactsDetection) {
                onNutritionFactsDetection(null, "")
            }
        }
    }

    @SuppressLint("UnsafeOptInUsageError")
    protected fun runOCR(
        imageProxy: ImageProxy,
        runPackagedFoodDetection: Boolean = false,
        onOCRResult: (candidates: List<PackagedFoodCandidate>, text: String) -> Unit = { _, _ -> },
        runNutritionFactsDetection: Boolean = false,
        onNutritionFactsDetection: (facts: PassioNutritionFacts?, text: String) -> Unit = { _, _ -> }
    ) {
        try {
            for (plane in imageProxy.planes) {
                plane.buffer.rewind()
            }

            val inputImage = InputImage.fromMediaImage(
                imageProxy.image!!,
                imageProxy.imageInfo.rotationDegrees
            )

            runOCR(
                inputImage,
                runPackagedFoodDetection,
                onOCRResult,
                runNutritionFactsDetection,
                onNutritionFactsDetection
            )
        } catch (e: IllegalStateException) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            if (runPackagedFoodDetection) {
                onOCRResult(emptyList(), "")
            }
            if (runNutritionFactsDetection) {
                onNutritionFactsDetection(null, "")
            }
        }
    }

    fun runOCR(
        bitmap: Bitmap,
        runPackagedFoodDetection: Boolean = false,
        onOCRResult: (candidates: List<PackagedFoodCandidate>, text: String) -> Unit = { _, _ -> },
        runNutritionFactsDetection: Boolean = false,
        onNutritionFactsDetection: (facts: PassioNutritionFacts?, text: String) -> Unit = { _, _ -> }
    ) {
        try {
            val inputImage = InputImage.fromBitmap(bitmap, 0)
            runOCR(
                inputImage,
                runPackagedFoodDetection,
                onOCRResult,
                runNutritionFactsDetection,
                onNutritionFactsDetection
            )
        } catch (e: IllegalStateException) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            if (runPackagedFoodDetection) {
                onOCRResult(emptyList(), "")
            }
            if (runNutritionFactsDetection) {
                onNutritionFactsDetection(null, "")
            }
        }
    }

    override fun onFrameSize(
        frameWidth: Int,
        frameHeight: Int,
        previewWidth: Int,
        previewHeight: Int,
        orientation: Int
    ) {
        this.frameWidth = frameWidth
        this.frameHeight = frameHeight
        this.orientation = orientation

        frameToClassTransform = ImageUtils.getTransformationMatrix(
            frameWidth,
            frameHeight,
            TF_HNN_API_INPUT_SIZE,
            TF_HNN_API_INPUT_SIZE,
            orientation,
            false
        )
    }

    @CameraThread
    override fun analyzeImage(imageProxy: ImageProxy) {
        if (!isReady.get()) {
            imageProxy.close()
            return
        }

        if (resourcesAnalyzingImage.get() != 0) {
            imageProxy.close()
            return
        }

        if (imageProxy.width != frameWidth && imageProxy.height != frameHeight) {
            imageProxy.close()
            return
        }

        val currentObjectDetectionListener = objectDetectionListener.get()
        val currentHNNDetectionListener = classificationListener.get()
        val currentFoodDetectionListener = foodDetectionListener.get()
        val shouldScanFood = isFoodEnabled.get()
        val shouldScanBarcode = isBarcodeEnabled.get()
        val shouldScanOCR = isOCREnabled.get()

        val currentNutritionFactsListener = nutritionFactsListener.get()
        val shouldScanNutriFacts = currentNutritionFactsListener != null

        resourcesAnalyzingImage.set(1)

        rgbBytes = IntArray(imageProxy.width * imageProxy.height)

        val planes = imageProxy.planes
        ImageUtils.imageProxyToYUV(planes, yuvBytes)

        val yRowStride = planes[0].rowStride
        val uvRowStride = planes[1].rowStride
        val uvPixelStride = planes[1].pixelStride

        ImageUtils.convertYUV420ToARGB8888CPP(
            yuvBytes[0]!!,
            yuvBytes[1]!!,
            yuvBytes[2]!!,
            imageProxy.width,
            imageProxy.height,
            yRowStride,
            uvRowStride,
            uvPixelStride,
            rgbBytes!!
        )

        val bitmap = Bitmap.createBitmap(
            rgbBytes!!,
            imageProxy.width,
            imageProxy.height,
            Bitmap.Config.ARGB_8888
        )

        var aggregateHelper: AggregateHelper? = null
        val numOfDetections =
            shouldScanFood.toInt() +
                    shouldScanBarcode.toInt() +
                    shouldScanOCR.toInt()

        if (currentFoodDetectionListener != null) {
            aggregateHelper = AggregateHelper(
                BitmapUtil.rotateBitmap(bitmap, cameraProxy.getSensorRotation()?.toFloat() ?: 0f),
                numOfDetections,
                mainExecutor,
                foodDetectionListener
            )
        }

        if (shouldScanFood) {
            if (modelHolder.objectDetector == null || modelHolder.hnnDetector == null) {
                aggregateHelper?.onFoodCandidates(listOf(), currentFoodDetectionListener)
            } else {
                resourcesAnalyzingImage.incrementAndGet()
                val startTime = System.currentTimeMillis()
                runVoting(bitmap, true) { result ->
                    tryCloseImage(imageProxy, bitmap)

                    val transformedResult = result.mapNotNull {
                        val relativeBoundingBox = RectF(
                            it.boundingBox.left / frameWidth,
                            it.boundingBox.top / frameHeight,
                            it.boundingBox.right / frameWidth,
                            it.boundingBox.bottom / frameHeight
                        )
                        val transformedObjectDetection = ObjectDetectionCandidate(
                            it.odCandidate.passioID,
                            it.odCandidate.confidence,
                            relativeBoundingBox
                        )
                        val label = labelManager.getMappedPassioID(it.votedCandidate.passioID)
                            ?: return@mapNotNull null
                        DebugCandidate(
                            label.first,
                            label.second,
                            it.votedCandidate,
                            transformedObjectDetection,
                            it.hnnCandidate,
                            it.knnCandidate,
                            it.croppedImage,
                            System.currentTimeMillis() - startTime,
                            it.alternatives
                        )
                    }
                    aggregateHelper?.onFoodCandidates(
                        transformedResult,
                        currentFoodDetectionListener
                    )
                }
            }
        }

        if (currentObjectDetectionListener != null && modelHolder.objectDetector != null) {
            resourcesAnalyzingImage.incrementAndGet()
            // Running on object detection thread
            runObjectDetection(bitmap) { result ->
                tryCloseImage(imageProxy, bitmap)

                result.map { candidate ->
                    val relativeBoundingBox = RectF(
                        candidate.boundingBox.left / frameWidth,
                        candidate.boundingBox.top / frameHeight,
                        candidate.boundingBox.right / frameWidth,
                        candidate.boundingBox.bottom / frameHeight
                    )
                    ObjectDetectionCandidate(
                        candidate.passioID,
                        candidate.confidence,
                        relativeBoundingBox
                    )
                }

                mainExecutor.execute {
                    objectDetectionListener.get()?.onObjectDetectionResult(result)
                }
            }
        }

        if (currentHNNDetectionListener != null && modelHolder.hnnDetector != null) {
            resourcesAnalyzingImage.incrementAndGet()

            runHNNDetection(bitmap) { result ->
                tryCloseImage(imageProxy, bitmap)

                mainExecutor.execute {
                    classificationListener.get()?.onClassificationResult(result)
                }
            }
        }

        if (shouldScanBarcode) {
            if (modelHolder.barcodeDetector == null) {
                aggregateHelper?.onBarcodeCandidates(listOf(), currentFoodDetectionListener)
            } else {
                resourcesAnalyzingImage.incrementAndGet()

                runBarcodeDetection(imageProxy) { barcodes ->
                    // Prune barcodes
                    val filtered = barcodes.filter { it.boundingBox != null }

                    val result = filtered.map { barcode ->
                        val relativeBoundingBox = RectF(
                            barcode.boundingBox!!.left.toFloat() / frameWidth,
                            barcode.boundingBox!!.top.toFloat() / frameHeight,
                            barcode.boundingBox!!.right.toFloat() / frameWidth,
                            barcode.boundingBox!!.bottom.toFloat() / frameHeight
                        )
                        BarcodeCandidate(barcode.value, relativeBoundingBox)
                    }
                    aggregateHelper?.onBarcodeCandidates(result, currentFoodDetectionListener)
                    tryCloseImage(imageProxy, bitmap)
                }
            }
        }

        var runPackaged = false
        var runNutriFacts = false

        if (shouldScanOCR) {
            if (modelHolder.ocrMatcher == null) {
                aggregateHelper?.onPackagedFoodCandidate(
                    emptyList(),
                    currentFoodDetectionListener
                )
            } else {
                resourcesAnalyzingImage.incrementAndGet()
                runPackaged = true
            }
        }

        if (shouldScanNutriFacts) {
            if (modelHolder.ocrDetector == null) {
                if (currentNutritionFactsListener == nutritionFactsListener.get()) {
                    mainExecutor.execute {
                        currentNutritionFactsListener?.onRecognitionResult(null, "")
                    }
                }
            } else {
                resourcesAnalyzingImage.incrementAndGet()
                runNutriFacts = true
            }
        }

        runOCR(
            imageProxy,
            runPackaged,
            { packagedFoodCandidate, _ ->
                if (shouldScanOCR) {
                    aggregateHelper?.onPackagedFoodCandidate(
                        packagedFoodCandidate,
                        currentFoodDetectionListener
                    )
                }
                tryCloseImage(imageProxy, bitmap)
            },
            runNutriFacts,
            { facts, text ->
                if (currentNutritionFactsListener == nutritionFactsListener.get()) {
                    mainExecutor.execute {
                        currentNutritionFactsListener?.onRecognitionResult(facts, text)
                    }
                }
                tryCloseImage(imageProxy, bitmap)
            })

        tryCloseImage(imageProxy, bitmap)
    }

    private fun tryCloseImage(imageProxy: ImageProxy, bitmap: Bitmap) {
        if (resourcesAnalyzingImage.decrementAndGet() == 0) {
            bitmap.recycle()
            imageProxy.close()
        }
    }

    fun startObjectDetection(objectDetectionListener: ObjectDetectionListener) {
        this.objectDetectionListener.set(objectDetectionListener)
    }

    fun stopObjectDetection() {
        this.objectDetectionListener.set(null)
    }

    fun startHierarchicalDetection(classificationListener: ClassificationListener) {
        this.classificationListener.set(classificationListener)
    }

    fun stopHierarchicalDetection() {
        this.classificationListener.set(null)
    }

    fun changeBoundingBoxFilter(mode: Int?) {
        objectDetectionStrategy = when (mode) {
            0 -> BiggerBoundingBoxStrategy()
            1 -> SmallerBoundingBoxStrategy()
            2 -> HigherConfidenceStrategy()
            null -> null
            else -> throw IllegalArgumentException("No known mode for Bounding Box Filer: $mode")
        }
    }

    fun startFoodDetection(
        foodRecognitionListener: FoodRecognitionListener,
        foodEnabled: Boolean,
        barcodeEnabled: Boolean,
        ocrEnabled: Boolean,
    ) {
        this.foodDetectionListener.set(foodRecognitionListener)
        this.isFoodEnabled.set(foodEnabled)
        this.isBarcodeEnabled.set(barcodeEnabled)
        this.isOCREnabled.set(ocrEnabled)
    }

    fun stopFoodDetection() {
        this.foodDetectionListener.set(null)
        this.isFoodEnabled.set(false)
        this.isBarcodeEnabled.set(false)
        this.isOCREnabled.set(false)
    }

    fun enableFrameAveraging(enabled: Boolean) {
        if (enabled) {
            frameAveragingRef.set(frameAveraging)
        } else {
            frameAveragingRef.set(null)
        }
    }

    protected fun Boolean.toInt() = if (this) 1 else 0

    fun searchNGrams(query: String): List<String> {
        if (modelHolder.searchMatcher == null) {
            return emptyList()
        }

        val result = modelHolder.searchMatcher!!.searchText(query)
        return result.map { it.first }
    }

    fun isAdvancedSearchEnabled(): Boolean {
        return modelHolder.searchMatcher != null
    }

    fun setNutritionFactsListener(listener: NutritionFactsRecognitionListener?) {
        nutritionFactsListener.set(listener)
    }

    fun runJustOCR(bitmap: Bitmap, callback: (result: String) -> Unit) {
        val inputImage = InputImage.fromBitmap(bitmap, 0)
        modelHolder.ocrDetector!!.process(inputImage).addOnSuccessListener {
            callback(it.text)
        }.addOnFailureListener {
            callback("")
        }
    }
}