package ai.passio.passiosdk.passiofood.recommend

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioFoodDataInfo
import ai.passio.passiosdk.passiofood.PassioMealTime
import ai.passio.passiosdk.passiofood.PassioSearchNutritionPreview
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlan
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlanItem
import ai.passio.passiosdk.passiofood.toDataModel
import org.json.JSONArray

internal object MealPlanService {

    fun fetchMealPlans(dev: Boolean, callback: (result: List<PassioMealPlan>) -> Unit) {
        fetchToken(false) { token ->
            if (token == null) {
                callback(emptyList())
                return@fetchToken
            }

            fetchMealPlansInternal(token, dev, callback)
        }
    }

    private fun fetchMealPlansInternal(
        token: String,
        dev: Boolean,
        callback: (result: List<PassioMealPlan>) -> Unit
    ) {
        val url = NativeUtils.instance.getMealPlanURL(dev)
        val task = NetworkStringTask(url, mapOf("Authorization" to token))
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    MealPlanService::class.java.simpleName,
                    "Could not fetch meal plans: $message"
                )
                callback(emptyList())
            }

            override fun onTokenExpired() {
                fetchMealPlans(dev, callback)
            }

            override fun onSuccess(result: String) {
                val jArray = JSONArray(result)
                val mealPlans = mutableListOf<PassioMealPlan>()
                for (i in 0 until jArray.length()) {
                    val jObject = jArray.getJSONObject(i)
                    val mealPlan = PassioMealPlan(
                        jObject.getString("mealPlanTitle"),
                        jObject.getString("mealPlanLabel"),
                        jObject.getJSONObject("macroTargets").getInt("carbs"),
                        jObject.getJSONObject("macroTargets").getInt("protein"),
                        jObject.getJSONObject("macroTargets").getInt("fat"),
                    )
                    mealPlans.add(mealPlan)
                }
                callback(mealPlans)
            }

        }, "fetchMealPlans")
    }

    fun fetchDayOf(
        label: String,
        day: Int,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<PassioMealPlanItem>) -> Unit
    ) {
        fetchToken(dev) { token ->
            if (token == null) {
                callback(emptyList())
                return@fetchToken
            }

            fetchDayOfInternal(token, label, day, dev, languageCode, callback)
        }
    }

    private fun fetchDayOfInternal(
        token: String,
        label: String,
        day: Int,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<PassioMealPlanItem>) -> Unit
    ) {
        val url = NativeUtils.instance.getMealPlanURL(dev)
        val headers = mutableMapOf("Authorization" to token)
        if (languageCode != null) {
            headers["Localization-ISO"] = languageCode
        }
        val task = NetworkStringTask("$url/$label/$day", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    MealPlanService::class.java.simpleName,
                    "Could not fetch meal plan $label for day $day: $message"
                )
                callback(emptyList())
            }

            override fun onTokenExpired() {
                fetchDayOf(label, day, dev, languageCode, callback)
            }

            override fun onSuccess(result: String) {
                val response = MealPlanResponse(result)
                val items = mutableListOf<PassioMealPlanItem>()
                response.dayMeals.forEach { dayResponse ->
                    dayResponse.meals.forEach { searchItem ->
                        var useShortName = true
                        val name = if (searchItem.shortName.isEmpty()) {
                            useShortName = false
                            searchItem.displayName
                        } else {
                            searchItem.shortName
                        }

                        val item = PassioMealPlanItem(
                            response.dayNumber,
                            response.dayTitle,
                            fromString(dayResponse.mealTimeLabel),
                            PassioFoodDataInfo(
                                searchItem.refCode,
                                name,
                                searchItem.brandName,
                                searchItem.iconId,
                                searchItem.score,
                                searchItem.scoredName,
                                searchItem.labelId,
                                searchItem.type,
                                searchItem.resultId,
                                useShortName,
                                searchItem.nutritionPreview.toDataModel(),
                                searchItem.tags
                            )
                        )

                        items.add(item)
                    }
                }
                callback(items)
            }

        }, "fetchMealPlanForDay")
    }

    private fun fetchToken(
        dev: Boolean = false,
        callback: (token: String?) -> Unit
    ) {
        TokenService.getInstance().getToken(
            dev,
            { token ->
                callback(token)
            }, { _ ->
                callback(null)
            }
        )
    }

    private fun fromString(mealName: String): PassioMealTime {
        return when (mealName) {
            "breakfast" -> PassioMealTime.BREAKFAST
            "lunch" -> PassioMealTime.LUNCH
            "dinner" -> PassioMealTime.DINNER
            "snack" -> PassioMealTime.SNACK
            else -> throw IllegalArgumentException("No known PassioMealTime: $mealName")
        }
    }
}