package ai.passio.passiosdk.passiofood.user

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.report.PostJsonNetworkTask
import ai.passio.passiosdk.passiofood.data.model.PassioFoodAmount
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioNutrients
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.user.UserService.toJsonPayload
import org.json.JSONArray
import org.json.JSONObject

private const val URL = "https://api.passiolife.com/v2/products/food/submit/product"
private const val REPORT_URL = "https://api.passiolife.com/v2/products/food/report/product"

internal object UserService {

    fun reportFoodItem(
        refCode: String,
        productCode: String,
        notes: List<String>?,
        dev: Boolean,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        if (refCode.isEmpty() && productCode.isEmpty()) {
            callback(PassioResult.Error("Either productCode or refCode must be supplied"))
            return
        }

        TokenService.getInstance().getToken(dev, { token ->
            reportFoodItemInternal(token, refCode, productCode, notes, dev, callback)
        }, { errorMessage ->
            callback(PassioResult.Error(errorMessage))
        })
    }

    private fun reportFoodItemInternal(
        token: String,
        refCode: String,
        productCode: String,
        notes: List<String>?,
        dev: Boolean,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        val jsonObject = JSONObject().apply {
            if (refCode.isNotEmpty()) put("refCode", refCode)
            if (productCode.isNotEmpty()) put("productCode", productCode)
            if (notes != null) put("notes", JSONArray(notes))
        }
        val json = jsonObject.toString()
        val headers = mapOf(
            "Content-Type" to "application/json",
            "Authorization" to token
        )
        val task = PostJsonNetworkTask(REPORT_URL, headers, json)
        NetworkService.instance.doRequest(task, object : NetworkCallback<Boolean> {
            override fun onFailure(code: Int, message: String) {
                callback(PassioResult.Error(message))
            }

            override fun onTokenExpired() {
                reportFoodItem(refCode, productCode, notes, dev, callback)
            }

            override fun onSuccess(result: Boolean) {
                callback(PassioResult.Success(result))
            }
        })
    }

    fun submitUserFood(
        foodItem: PassioFoodItem,
        dev: Boolean,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        TokenService.getInstance().getToken(dev, { token ->
            submitUserFoodInternal(token, foodItem, dev, callback)
        }, { errorMessage ->
            callback(PassioResult.Error(errorMessage))
        })
    }

    private fun submitUserFoodInternal(
        token: String,
        foodItem: PassioFoodItem,
        dev: Boolean,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        val json = foodItem.toJsonPayload()
        val headers = mapOf(
            "Content-Type" to "application/json",
            "Authorization" to token
        )
        val task = PostJsonNetworkTask(URL, headers, json)
        NetworkService.instance.doRequest(task, object : NetworkCallback<Boolean> {
            override fun onFailure(code: Int, message: String) {
                callback(PassioResult.Error(message))
            }

            override fun onTokenExpired() {
                submitUserFood(foodItem, dev, callback)
            }

            override fun onSuccess(result: Boolean) {
                callback(PassioResult.Success(result))
            }
        })
    }

    private fun PassioFoodItem.toJsonPayload(): String {
        val ingredient = this.ingredients.first()
        val jsonPayload = JSONObject()

        val jBranded = JSONObject().apply {
            put("ingredients", ingredient.metadata.ingredientsDescription)
            put("owner", this@toJsonPayload.details)
            put("productCode", ingredient.metadata.barcode)
        }
        jsonPayload.put("branded", jBranded)
        jsonPayload.put("id", this.id)
        jsonPayload.put("name", this.name)

        val nutrients = this.nutrientsReference()
        val jArray = nutrients.toJsonPayload()
        jsonPayload.put("nutrients", jArray)

        val jPortions = this.amount.toJsonPayload()
        jsonPayload.put("portions", jPortions)

        return jsonPayload.toString()
    }

    private fun PassioNutrients.toJsonPayload(): JSONArray {
        val jsonArray = JSONArray()
        this.nutrientDefaults.forEach { pn ->
            if (pn.field.get() == null) {
                return@forEach
            }

            val jNutrients = JSONObject()
            jNutrients.put("amount", pn.field.get()!!.value)
            jNutrients.put("id", pn.id)
            val jDefaultNutrient = JSONObject().apply {
                put("shortName", pn.shortName)
                put("unit", pn.field.get()!!.unit.symbol)
            }
            jNutrients.put("nutrient", jDefaultNutrient)
            jsonArray.put(jNutrients)
        }
        // Add missing calories
        if (this.calories() != null) {
            val jNutrients = JSONObject()
            jNutrients.put("amount", calories()!!.value)
            jNutrients.put("id", 1603211196548)
            val jDefaultNutrient = JSONObject().apply {
                put("shortName", "calories")
                put("unit", calories()!!.unit.symbol)
            }
            jNutrients.put("nutrient", jDefaultNutrient)
            jsonArray.put(jNutrients)
        }
        // Add missing vitaminA
        if (this.vitaminA() != null) {
            val jNutrients = JSONObject()
            jNutrients.put("amount", vitaminA()!!)
            jNutrients.put("id", 1603211196594)
            val jDefaultNutrient = JSONObject().apply {
                put("shortName", "vitaminA")
                put("unit", "IU")
            }
            jNutrients.put("nutrient", jDefaultNutrient)
            jsonArray.put(jNutrients)
        }

        return jsonArray
    }

    private fun PassioFoodAmount.toJsonPayload(): JSONArray {
        val jsonArray = JSONArray()
        servingSizes.forEach { servingSize ->
            val servingUnit =
                servingUnits.firstOrNull { it.unitName == servingSize.unitName } ?: return@forEach
            val jWeight = JSONObject().apply {
                put("unit", servingUnit.weight.unit.symbol)
                put("value", servingUnit.weight.value)
            }

            val jServing = JSONObject().apply {
                put("name", servingSize.unitName)
                put("quantity", servingSize.quantity)
                put("weight", jWeight)
            }

            jsonArray.put(jServing)
        }
        return jsonArray
    }
}