package ai.passio.passiosdk.passiofood.voice.remote

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.PostNetworkTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioFoodDataInfo
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorFoodInfo
import ai.passio.passiosdk.passiofood.data.model.PassioFoodResultType
import ai.passio.passiosdk.passiofood.toDataModel
import ai.passio.passiosdk.passiofood.data.model.PassioSpeechRecognitionModel
import ai.passio.passiosdk.passiofood.data.model.toLogAction
import ai.passio.passiosdk.passiofood.data.model.toMealTime
import org.json.JSONArray

private const val VOICE_URL = "extractMealLogAction"

internal object SpeechService {

    fun fetchSpeechRecognitionResult(
        text: String,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<PassioSpeechRecognitionModel>) -> Unit
    ) {
        fetchToken(dev) { token ->
            if (token == null) return@fetchToken

            fetchSpeechRecognitionResult(token, text, dev, languageCode, callback)
        }
    }

    private fun fetchSpeechRecognitionResult(
        token: String,
        text: String,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<PassioSpeechRecognitionModel>) -> Unit
    ) {
        val url = NativeUtils.instance.getAdvisorToolsUrl(dev)
        val headers = mutableMapOf(
            "Content-Type" to "text/plain",
            "Authorization" to token,
        )
        if (languageCode != null) {
            headers["Localization-ISO"] = languageCode
        }
        val task = PostNetworkTask(
            url + VOICE_URL,
            headers,
            text,
            readTimeout = 20 * 1000
        )
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    SpeechService::class.java.simpleName,
                    "Could not fetch voice result: $message"
                )
                callback(emptyList())
            }

            override fun onTokenExpired() {
                fetchSpeechRecognitionResult(text, dev, languageCode, callback)
            }

            override fun onSuccess(result: String) {
                val jArray = JSONArray(result)
                val list = mutableListOf<PassioSpeechRecognitionModel>()
                for (i in 0 until jArray.length()) {
                    val voiceItem = ResponseVoice(jArray.getJSONObject(i).toString())
                    voiceItem.items.forEach {
                        val foodDataInfo = PassioFoodDataInfo(
                            it.refCode,
                            it.shortName.ifEmpty { it.displayName },
                            it.brandName,
                            it.iconId,
                            it.score,
                            it.scoredName,
                            it.labelId,
                            it.type,
                            it.resultId,
                            true,
                            it.nutritionPreview.toDataModel(),
                            it.tags
                        )
                        val dataInfo = PassioAdvisorFoodInfo(
                            it.ingredientName,
                            it.portionSize,
                            it.weightGrams,
                            foodDataInfo,
                            null,
                            PassioFoodResultType.FOOD_ITEM,
                        )
                        val model = PassioSpeechRecognitionModel(
                            voiceItem.action.toLogAction(),
                            voiceItem.mealTime.toMealTime(),
                            voiceItem.date,
                            dataInfo
                        )
                        list.add(model)
                    }
                }
                callback(list)
            }

        }, "recognizeSpeechRemote")
    }

    private fun fetchToken(
        dev: Boolean,
        callback: (token: String?) -> Unit
    ) {

        TokenService.getInstance()
            .getToken(dev, { token -> callback(token) }, { _ -> callback(null) })
    }
}