package ai.passio.passiosdk.passiofood

import ai.passio.passiosdk.core.camera.PassioCameraConfigurator
import ai.passio.passiosdk.core.camera.PassioCameraViewProvider
import ai.passio.passiosdk.core.config.PassioConfiguration
import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.core.icons.IconSize
import ai.passio.passiosdk.passiofood.PassioSDK.Companion.instance
import ai.passio.passiosdk.passiofood.data.model.PassioAdvisorFoodInfo
import ai.passio.passiosdk.core.camera.PassioCameraData
import ai.passio.passiosdk.passiofood.data.model.PassioIDEntityType
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlan
import ai.passio.passiosdk.passiofood.data.model.PassioMealPlanItem
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.data.model.PassioSpeechRecognitionModel
import ai.passio.passiosdk.passiofood.data.model.PassioUPFRating
import android.content.Context
import android.graphics.Bitmap
import android.graphics.RectF
import android.graphics.drawable.Drawable
import androidx.annotation.Keep
import androidx.camera.core.CameraSelector

private const val MINIMUM_CONFIDENCE_TF_OD_API = 0.3f

/**
 * Main access point of the SDK. Defines all of the SDK's
 * main capabilities. The API of the SDK is designed as a
 * singleton with the [instance] containing the concrete
 * implementation of the interface. The SDK is design to
 * be called from the main thread. The execution of most
 * of its functions will be delegated to an internal
 * thread pool, but the resulting callback will always be
 * invoked on the main thread.
 */
@Keep
interface PassioSDK {

    /**
     * Defines the processing speed of the SDK's camera
     * recognition system.
     * ONE - indicates that one frame will be analyzed
     *       every second. If the analysis time of that
     *       frame exceeds time of a second, the next
     *       frame will be analyzed only after the
     *       previous one has been complete.
     * TWO - indicates that one frame will be analyzed
     *       every 500 milliseconds. This is the default
     *       behaviour of the SDK.
     * MAX - there is no possible waiting period between
     *       the analysis of two frames. As soon as the
     *       recognition system is done with the previous
     *       frame, the analysis of the next one is called.
     *       This mode might is computational heavy
     *       and might take up a lot of processing power
     *       of the system.
     */
    enum class FramesPerSecond {
        ONE,
        TWO,
        MAX
    }

    @Keep
    companion object {
        const val BKG_PASSIO_ID = "BKG0001"

        internal var internalInstance: PassioSDKImpl? = null

        /**
         * Use this singleton to access the Passio SDK.
         */
        @JvmStatic
        val instance: PassioSDK
            get() {
                if (internalInstance == null) {
                    internalInstance = PassioSDKImpl()
                }
                return internalInstance!!
            }

        /**
         * The value of confidence of recognition results below which results won't be returned
         * to the user. The default value is 0.5.
         */
        fun minimumConfidence() = MINIMUM_CONFIDENCE_TF_OD_API

        fun getVersion() = PassioSDKImpl.SDK_VERSION
    }

    /**
     * Initializes the SDK with the given [passioConfiguration]. See [PassioConfiguration] for more
     * information on the different types of SDK configuration. The initialization process includes
     * downloading or reading a cached version of the license and loading the models into the
     * TensorFlowLite runner. This process is being executed on a background thread, but the
     * callback with the result of the configuration process will be called on the main thread. You
     * may call [startCamera] or [startFoodDetection] before this method, there is no need to wait
     * the [onComplete] callback.
     *
     * @param passioConfiguration the input configuration for the SDK.
     * @param onComplete the callback that will be called at the end of the configuration process
     *        with the result in the form of the {@link ai.passio.passiosdk.core.config.PassioStatus}
     */
    fun configure(
        passioConfiguration: PassioConfiguration,
        onComplete: (status: PassioStatus) -> Unit
    )

    /**
     * Checks whether the SDK initialization ran to completion. The initialization process starts
     * with the [configure] function.
     *
     * @return true if the SDK has initialized without errors, false instead.
     */
    fun isSDKReady(): Boolean

    /**
     * Starts the camera preview with the given [PassioCameraViewProvider]. Using CameraX
     * (https://developer.android.com/training/camerax), the camera system will start rendering
     * the frames onto the PreviewView. Also this method binds the camera with the lifecycle
     * provider of the CameraViewProvider. When that lifecycle provider calls onStart() the
     * camera preview will start and when it calls onStop() the camera preview will stop and
     * start the shutdown process. [android.view.Surface.ROTATION_]
     *
     * @param passioCameraViewProvider provides the PreviewView to render the camera frames, the
     *        lifecycle holder which will start and stop the camera, and also the context provider
     *        needed to open the camera.
     * @param displayRotation if the orientation changes are enabled, use this parameter to notify
     *        the camera of the display rotation. Accepted values are: Surface.ROTATION_0,
     *        Surface.ROTATION_90, Surface.ROTATION_180, Surface.ROTATION_270.
     * @param tapToFocus set true to enable a tap listener on the passioCameraViewProvider's
     *        PreviewView that will start a manual focus operation.
     */
    fun startCamera(
        passioCameraViewProvider: PassioCameraViewProvider,
        displayRotation: Int = 0,
        @CameraSelector.LensFacing cameraFacing: Int = CameraSelector.LENS_FACING_BACK,
        tapToFocus: Boolean = false,
        onCameraReady: (data: PassioCameraData) -> Unit = {}
    )

    fun startCamera(
        viewProvider: PassioCameraViewProvider,
        configurator: PassioCameraConfigurator
    )

    fun stopCamera()

    /**
     * Turns the device's flashlight on or off.
     *
     * @param enabled if enabled is set to true, it will turn the flashlight on. If the enabled
     *        parameter is the to false, it will turn the flashlight off.
     */
    fun enableFlashlight(enabled: Boolean)

    fun setCameraZoomLevel(zoomLevel: Float)

    fun getMinMaxCameraZoomLevel(): Pair<Float?, Float?>

    fun takePicture(onResult: (bitmap: Bitmap?) -> Unit)

    /**
     * Sets the time between each analyzed frame. For example, it the [timeForAnalysis] is set to
     * 500L (0.5 seconds), the camera engine will analyze 2 frames per second. If set to 0L, the
     * camera will analyze frames as fast as possible. If the time is smaller than the time it takes
     * to analyze and process the frame, the SDK will work as if set to 0L. The default is set to
     * 1000L (1 second).
     *
     * @param timeForAnalysis the time passed between two frames that are being analyzed.
     */
    fun runFrameEvery(timeForAnalysis: Long)

    /**
     * Finalizes the PassioSDK instance. Deallocates the memory reserved for TFLite models,
     * transformation matrices and shuts down all the background queues that power the SDK. To run
     * the SDK again, [configure] must be called.
     */
    fun shutDownPassioSDK() {
        internalInstance = null
    }

    /**
     * Registers a listener for the food detection process. The results will be returned at a
     * frequency defined in the [FoodDetectionConfiguration.framesPerSecond] field.
     *
     * @param foodRecognitionListener the interface that serves as a callback for the recognition
     *        results.
     * @param detectionConfig an object that defines what to recognize, how often, and other
     *        recognition properties.
     *
     */
    fun startFoodDetection(
        foodRecognitionListener: FoodRecognitionListener,
        detectionConfig: FoodDetectionConfiguration = FoodDetectionConfiguration()
    ): Boolean

    /**
     * Stops the food detection process. After this method is called no results should be delivered
     * to a previously registered [FoodRecognitionListener].
     */
    fun stopFoodDetection(): Boolean

    fun lookupIconsFor(
        context: Context,
        passioID: PassioID,
        iconSize: IconSize = IconSize.PX90,
        type: PassioIDEntityType = PassioIDEntityType.item,
    ): Pair<Drawable, Drawable?>

    /**
     * For a given [PassioID] returns the corresponding image from the SDK's asset folder.
     *
     * @param context used to open assets
     * @param passioID key to find the image
     * @return if the [passioID] is valid returns the [Drawable] from the assets. If not returns null.
     */
    fun fetchIconFor(
        context: Context,
        passioID: PassioID,
        iconSize: IconSize = IconSize.PX90,
        callback: (drawable: Drawable?) -> Unit
    )

    /**
     * If not null, the [PassioStatusListener] will provide callbacks
     * when the internal state of the SDK's configuration process changes.
     * Passing null will unregister the listener.
     */
    fun setPassioStatusListener(statusListener: PassioStatusListener?)

    /**
     * Transforms the bounding box of the camera frame to the coordinates
     * of the preview view where it should be displayed.
     *
     * @param boundingBox the bounding box from the ObjectDetectionCandidate.
     * @param viewWidth the width of the camera preview view.
     * @param viewHeight the height of the camera preview view.
     * @param displayAngle the rotation of the camera preview view.
     * @param barcode does the bounding box belong to the barcode candidate.
     * @return the bounding box within the coordinate system of the camera view.
     */
    fun boundingBoxToViewTransform(
        boundingBox: RectF,
        viewWidth: Int,
        viewHeight: Int,
        displayAngle: Int = 0,
        barcode: Boolean = false
    ): RectF

    fun iconURLFor(passioID: PassioID, size: IconSize = IconSize.PX90): String

    fun fetchTagsFor(refCode: String, onTagsFetched: (tags: List<String>?) -> Unit)

    fun fetchInflammatoryEffectData(
        refCode: String,
        onResult: (data: List<InflammatoryEffectData>?) -> Unit
    )

    fun searchForFood(
        term: String,
        callback: (result: List<PassioFoodDataInfo>, searchOptions: List<String>) -> Unit
    )

    fun searchForFoodSemantic(
        term: String,
        callback: (result: List<PassioFoodDataInfo>, searchOptions: List<String>) -> Unit
    )

    fun fetchFoodItemForDataInfo(
        dataInfo: PassioFoodDataInfo,
        servingQuantity: Double? = null,
        servingUnit: String? = null,
        callback: (foodItem: PassioFoodItem?) -> Unit
    )

    fun fetchFoodItemForProductCode(
        productCode: String,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    )

    fun fetchFoodItemForPassioID(
        passioID: PassioID,
        onResult: (foodItem: PassioFoodItem?) -> Unit
    )

    fun fetchSuggestions(
        mealTime: PassioMealTime,
        callback: (results: List<PassioFoodDataInfo>) -> Unit
    )

    fun fetchMealPlans(callback: (result: List<PassioMealPlan>) -> Unit)

    fun fetchMealPlanForDay(
        mealPlanLabel: String,
        day: Int,
        callback: (result: List<PassioMealPlanItem>) -> Unit
    )

    fun recognizeSpeechRemote(
        text: String,
        callback: (result: List<PassioSpeechRecognitionModel>) -> Unit
    )

    fun recognizeImageRemote(
        bitmap: Bitmap,
        resolution: PassioImageResolution = PassioImageResolution.RES_512,
        message: String? = null,
        callback: (result: List<PassioAdvisorFoodInfo>) -> Unit
    )

    fun fetchFoodItemForRefCode(
        refCode: String,
        callback: (foodItem: PassioFoodItem?) -> Unit
    )

    fun startNutritionFactsDetection(listener: NutritionFactsRecognitionListener): Boolean

    fun stopNutritionFactsDetection()

    fun fetchFoodItemLegacy(
        passioID: PassioID,
        callback: (foodItem: PassioFoodItem?) -> Unit
    )

    fun recognizeNutritionFactsRemote(
        bitmap: Bitmap,
        resolution: PassioImageResolution = PassioImageResolution.RES_1080,
        callback: (foodItem: PassioFoodItem?) -> Unit
    )

    fun fetchHiddenIngredients(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    )

    fun fetchVisualAlternatives(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    )

    fun fetchPossibleIngredients(
        foodName: String,
        callback: (response: PassioResult<List<PassioAdvisorFoodInfo>>) -> Unit
    )

    fun setAccountListener(listener: PassioAccountListener?)

    fun getPassioStatus(): PassioStatus

    fun updateLanguage(languageCode: String): Boolean

    fun reportFoodItem(
        refCode: String = "",
        productCode: String = "",
        notes: List<String>? = null,
        callback: (result: PassioResult<Boolean>) -> Unit
    )

    fun submitUserCreatedFoodItem(
        foodItem: PassioFoodItem,
        callback: (result: PassioResult<Boolean>) -> Unit
    )

    fun predictNextIngredients(
        currentIngredients: List<String>,
        callback: (result: List<PassioFoodDataInfo>) -> Unit
    )

    fun fetchUltraProcessingFoodRating(
        foodItem: PassioFoodItem,
        callback: (result: PassioResult<PassioUPFRating>) -> Unit
    )
}