package ai.passio.passiosdk.passiofood.metadata

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.file.PassioFileManager
import ai.passio.passiosdk.core.file.PassioFileManager.Companion.getMetadataDir
import ai.passio.passiosdk.core.file.PassioFileManager.Companion.getMetadataDownloadDir
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkFileTask
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.FileUtil
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.config.passio_metadata
import android.content.Context
import android.util.JsonReader
import android.util.Log
import java.io.File
import java.io.FileInputStream
import java.io.InputStream
import java.io.InputStreamReader
import java.nio.charset.Charset
import java.util.concurrent.Executors
import kotlin.Exception

internal class MetadataManager(
    private val fileManager: PassioFileManager
) {

    private val executor = Executors.newSingleThreadExecutor()

    private var hash: String? = null
    private var generatedAt: String? = null
    private lateinit var passioIDs: List<PassioID>
    private val idMap: MutableMap<PassioID, Pair<PassioID, String>> = mutableMapOf()
    private val alternatives: MutableMap<PassioID, List<Pair<PassioID, String>>> = mutableMapOf()
    private val iconMap: MutableMap<PassioID, PassioID> = mutableMapOf()

    fun configure(
        context: Context,
        useAssets: Boolean,
        remoteOnly: Boolean,
        dev: Boolean,
    ): String? {
        val localFile = fileManager.getMetadataWithUpdate(context)
        val datasetId = if (!remoteOnly) {
            fileManager.getDatasetID(context, useAssets) ?: return null
        } else {
            NativeUtils.instance.getLocalDatasetId()
        }

        if (localFile != null && localFile.exists()) {
            val localHash = localFile.name.split(".")[1]
            fetchMetadataBackground(context, dev, localHash, datasetId)
            loadFromFile(localFile, false)
            return hash
        }

        val token = TokenService.getInstance().getTokenSync(dev) ?: return null

        val baseUrl = NativeUtils.instance.getMetadataURL(dev)
        val url = "$baseUrl$datasetId/0"
        val outPath = "${context.getMetadataDir()}/${passio_metadata.name}.0.passio2"
        val headers = mutableMapOf("Authorization" to token)

        val task = NetworkFileTask(url, headers, outPath)
        val file = NetworkService.instance.syncRequest(task) ?: return null
        loadFromFile(file, true)

        return hash
    }

    private fun fetchMetadataBackground(
        context: Context,
        dev: Boolean,
        localHash: String,
        datasetId: String
    ) {
        val baseUrl = NativeUtils.instance.getMetadataURL(dev)

        TokenService.getInstance().getToken(
            dev,
            { token ->
                val url = "$baseUrl$datasetId/${localHash}"
                val outPath =
                    "${context.getMetadataDownloadDir()}/${passio_metadata.name}.0.passio2"
                val headers = mutableMapOf("Authorization" to token)
                val task = NetworkFileTask(url, headers, outPath)
                NetworkService.instance.doRequest(task, object : NetworkCallback<File?> {
                    override fun onFailure(code: Int, message: String) {
                        PassioLog.e(
                            this::class.java.simpleName,
                            "Could not fetch metadata: $message"
                        )
                    }

                    override fun onTokenExpired() {
                        fetchMetadataBackground(context, dev, localHash, datasetId)
                    }

                    override fun onSuccess(result: File?) {
                        if (result == null) return

                        val fis = FileInputStream(result)
                        val hash = parseHash(fis)
                        fis.close()
                        FileUtil.renameTo(result, "${passio_metadata.name}.$hash.passio2")

                        PassioLog.i(
                            this::class.java.simpleName,
                            "Metadata $hash saved to download"
                        )
                    }
                })

            }, { errorMessage ->
                PassioLog.e(
                    this::class.java.simpleName,
                    "Could not fetch token for metadata: $errorMessage"
                )
            })

    }

    private fun loadFromFile(file: File?, rename: Boolean) {
        if (file == null) return
        val fis = FileInputStream(file)
        passioIDs = parseLabels(fis)
        fis.close()

        val fis2 = FileInputStream(file)
        hash = parseHash(fis2)
        fis2.close()

        val newFile = if (rename) {
            if (hash != null) {
                FileUtil.renameTo(file, "${passio_metadata.name}.$hash.passio2")
            } else {
                null
            }
        } else {
            file
        }
        if (newFile == null) return

        parseInBackground(newFile)
    }

    fun getVisualPassioIDs(): List<PassioID> = passioIDs

    fun getMappedPassioID(passioID: PassioID): Pair<PassioID, String>? {
        return idMap[passioID]
    }

    fun getAlternatives(passioID: PassioID): List<Pair<PassioID, String>>? {
        return alternatives[passioID]
    }

    fun getMappedIcon(passioID: PassioID): PassioID? {
        return iconMap[passioID]
    }

    private fun parseHash(inputStream: InputStream): String? {
        val inReader = InputStreamReader(inputStream, Charset.defaultCharset())
        val reader = JsonReader(inReader)
        reader.beginObject()
        while (reader.hasNext()) {
            when (reader.nextName()) {
                "hash" -> {
                    val hash = reader.nextString()
                    reader.close()
                    inReader.close()
                    return hash
                }

                else -> reader.skipValue()
            }
        }
        reader.endObject()
        reader.close()
        inReader.close()
        return null
    }

    private fun parseLabels(inputStream: InputStream): List<PassioID> {
        val inReader = InputStreamReader(inputStream, Charset.defaultCharset())
        val reader = JsonReader(inReader)
        val passioIDs = mutableListOf<PassioID>()

        reader.beginObject()
        while (reader.hasNext()) {
            when (reader.nextName()) {
                "labelNames" -> {
                    reader.beginArray()

                    while (reader.hasNext()) {
                        reader.beginObject()

                        while (reader.hasNext()) {
                            when (reader.nextName()) {
                                "labelname" -> {
                                    val passioID = reader.nextString()
                                    passioIDs.add(passioID)
                                }

                                else -> reader.skipValue()
                            }
                        }
                        reader.endObject()
                    }
                    reader.endArray()
                }

                else -> reader.skipValue()
            }
        }
        reader.endObject()
        reader.close()
        inReader.close()
        return passioIDs
    }

    private fun parseInBackground(file: File) {
        executor.submit(object : Runnable {
            override fun run() {
                val inputStream = FileInputStream(file)
                val inReader = InputStreamReader(inputStream, Charset.defaultCharset())
                val reader = JsonReader(inReader)
                try {
                    reader.beginObject()
                    while (reader.hasNext()) {
                        when (reader.nextName()) {
                            "hash" -> hash = reader.nextString()
                            "generatedAt" -> generatedAt = reader.nextString()
                            "labelIcons" -> {
                                reader.beginObject()

                                while (reader.hasNext()) {
                                    val labelId = reader.nextName()
                                    val labelValue = reader.nextString()
                                    iconMap[labelId] = labelValue
                                }

                                reader.endObject()
                            }

                            "labelNames" -> {
                                reader.beginArray()

                                while (reader.hasNext()) {
                                    reader.beginObject()

                                    var labelname = ""
                                    var displayName = ""
                                    val children = mutableListOf<Pair<PassioID, String>>()

                                    while (reader.hasNext()) {
                                        when (reader.nextName()) {
                                            "labelname" -> {
                                                labelname = reader.nextString()
                                            }

                                            "displayName" -> {
                                                displayName = reader.nextString()
                                            }

                                            "children" -> {
                                                reader.beginArray()

                                                while (reader.hasNext()) {
                                                    reader.beginObject()
                                                    var childLabelName = ""
                                                    var childDisplayName = ""
                                                    while (reader.hasNext()) {
                                                        when (reader.nextName()) {
                                                            "labelname" -> {
                                                                childLabelName = reader.nextString()
                                                            }

                                                            "displayName" -> {
                                                                childDisplayName =
                                                                    reader.nextString()
                                                            }

                                                            else -> reader.skipValue()
                                                        }
                                                    }
                                                    children.add(childLabelName to childDisplayName)
                                                    reader.endObject()
                                                }

                                                reader.endArray()
                                            }

                                            else -> reader.skipValue()
                                        }
                                    }

                                    if (children.isNotEmpty()) {
                                        val firstChild = children.first()
                                        idMap[labelname] = firstChild
                                        if (children.size > 1) {
                                            alternatives[labelname] =
                                                children.mapIndexedNotNull { index, l ->
                                                    if (index == 0) null else l
                                                }
                                        }
                                    } else {
                                        idMap[labelname] = labelname to displayName
                                    }

                                    reader.endObject()
                                }

                                reader.endArray()
                            }

                            else -> reader.skipValue()
                        }
                    }
                    reader.endObject()

                    reader.close()
                    inReader.close()
                    inputStream.close()
                    PassioLog.i(
                        this::class.java.simpleName,
                        "Metadata loaded from file in background"
                    )
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }

        })
    }
}