package ai.passio.passiosdk.passiofood.recommend

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.network.PostNetworkTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.report.PostJsonNetworkTask
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.PassioMealTime
import ai.passio.passiosdk.passiofood.PassioSearchNutritionPreview
import ai.passio.passiosdk.passiofood.PassioFoodDataInfo
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioNutrients
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.data.model.PassioUPFRating
import ai.passio.passiosdk.passiofood.search.SearchResult
import org.json.JSONArray
import org.json.JSONObject
import kotlin.math.round

private const val UPF_URL = "ultraProcessedFoodRating"

internal object SuggestionService {

    fun fetchSuggestions(
        mealTime: PassioMealTime,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<SearchResult>) -> Unit
    ) {
        fetchToken(dev) { token ->
            if (token == null) {
                callback(emptyList())
                return@fetchToken
            }

            fetchSuggestionsInternal(mealTime, token, dev, languageCode, callback)
        }
    }

    private fun fetchSuggestionsInternal(
        mealTime: PassioMealTime,
        token: String,
        dev: Boolean,
        languageCode: String?,
        callback: (result: List<SearchResult>) -> Unit
    ) {
        val url = NativeUtils.instance.getSuggestionURL(dev)
        val headers = mutableMapOf("Authorization" to token)
        if (languageCode != null) {
            headers["Localization-ISO"] = languageCode
        }
        val task = NetworkStringTask(url + mealTime.mealName, headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.w(
                    this@SuggestionService::class.java.simpleName,
                    "Could not fetch suggestions: $message"
                )
                callback(emptyList())
            }

            override fun onTokenExpired() {
                fetchSuggestions(mealTime, dev, languageCode, callback)

            }

            override fun onSuccess(result: String) {
                try {
                    val jArray = JSONArray(result)
                    val list = mutableListOf<SearchResult>()
                    for (i in 0 until jArray.length()) {
                        list.add(SearchResult(jArray.getJSONObject(i).toString()))
                    }
                    callback(list)
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SuggestionService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(emptyList())
                }
            }

        }, "fetchSuggestions")
    }

    fun fetchUltraProcessingFoodRating(
        foodItem: PassioFoodItem,
        dev: Boolean,
        languageCode: String?,
        callback: (result: PassioResult<PassioUPFRating>) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(PassioResult.Error("Could not fetch token"))
                return@fetchToken
            }

            fetchUltraProcessingFoodRatingInternal(
                foodItem, token, dev, languageCode, callback
            )
        }
    }

    private fun fetchUltraProcessingFoodRatingInternal(
        foodItem: PassioFoodItem,
        token: String,
        dev: Boolean,
        languageCode: String?,
        callback: (result: PassioResult<PassioUPFRating>) -> Unit
    ) {
        val url = NativeUtils.instance.getAdvisorToolsUrl(dev) + UPF_URL
        val headers = mutableMapOf("Authorization" to token)
        if (languageCode != null) {
            headers["Localization-ISO"] = languageCode
        }
        val payload = foodItem.toUPFPayload()
        val task = PostNetworkTask(
            url,
            headers,
            payload,
            readTimeout = 30_000,
            connectTimeout = 30_000
        )
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.w(
                    this@SuggestionService::class.java.simpleName,
                    "Could not fetch suggestions: $message"
                )
                callback(PassioResult.Error("Code: $code, message: $message"))
            }

            override fun onTokenExpired() {
                fetchUltraProcessingFoodRating(foodItem, dev, languageCode, callback)
            }

            override fun onSuccess(result: String) {
                try {
                    val json = ResponseUPF(result)
                    val score = PassioUPFRating(
                        json.rating,
                        json.highlightedIngredients ?: listOf()
                    )
                    callback(PassioResult.Success(score))
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SuggestionService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(PassioResult.Error("Could not parse result: ${e.message}"))
                }
            }

        }, "fetchUltraProcessingFoodRating")
    }

    private fun fetchToken(
        dev: Boolean = false,
        callback: (token: String?) -> Unit
    ) {

        TokenService.getInstance().getToken(
            dev,
            { token ->
                callback(token)
            }, { _ ->
                callback(null)
            }
        )
    }

    private fun PassioFoodItem.toUPFPayload(): String {
        val ingredient = this.ingredients.first()
        val root = JSONObject()
        root.put("name", this.name)
        root.put("tags", JSONArray(ingredient.metadata.tags))
        root.put("brand", this.details)
        root.put("ingredients", JSONArray(arrayOf(ingredient.metadata.ingredientsDescription)))
        root.put("nutrients", ingredient.referenceNutrients.toUPFPayload())

        return JSONObject().apply {
            put("content", root)
        }.toString()
    }

    private fun PassioNutrients.toUPFPayload(): JSONObject {
        val root = JSONObject()
        root.putOpt("fat", fat())
        root.putOpt("satFat", satFat())
        root.putOpt("monounsaturatedFat", monounsaturatedFat())
        root.putOpt("polyunsaturatedFat", polyunsaturatedFat())
        root.putOpt("protein", protein())
        root.putOpt("carbs", carbs())
        root.putOpt("calories", calories())
        root.putOpt("cholesterol", cholesterol())
        root.putOpt("sodium", sodium())
        root.putOpt("fibers", fibers())
        root.putOpt("transFat", transFat())
        root.putOpt("sugars", sugars())
        root.putOpt("sugarsAdded", sugarsAdded())
        root.putOpt("alcohol", alcohol())
        root.putOpt("iron", iron())
        root.putOpt("vitaminC", vitaminC())
        root.putOpt("vitaminD", vitaminD())
        root.putOpt("vitaminB6", vitaminB6())
        root.putOpt("vitaminB12", vitaminB12())
        root.putOpt("vitaminB12Added", vitaminB12Added())
        root.putOpt("vitaminE", vitaminE())
        root.putOpt("vitaminEAdded", vitaminEAdded())
        root.putOpt("iodine", iodine())
        root.putOpt("calcium", calcium())
        root.putOpt("potassium", potassium())
        root.putOpt("magnesium", magnesium())
        root.putOpt("phosphorus", phosphorus())
        root.putOpt("sugarAlcohol", sugarAlcohol())
        root.putOpt("vitaminA", vitaminA())
        root.putOpt("vitaminARAE", vitaminARAE())
        root.putOpt("vitaminKPhylloquinone", vitaminKPhylloquinone())
        root.putOpt("vitaminKMenaquinone4", vitaminKMenaquinone4())
        root.putOpt("vitaminKDihydrophylloquinone", vitaminKDihydrophylloquinone())
        root.putOpt("zinc", zinc())
        root.putOpt("chromium", chromium())
        root.putOpt("selenium", selenium())
        root.putOpt("folicAcid", folicAcid())
        return root
    }
}