package ai.passio.passiosdk.passiofood.version

import ai.passio.passiosdk.core.config.PassioSDKError
import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.core.config.SDKFileType
import ai.passio.passiosdk.core.file.PassioFileManager
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.core.version.ConfigurationListener
import ai.passio.passiosdk.passiofood.PassioSDKImpl
import ai.passio.passiosdk.passiofood.config.passio_dataset
import ai.passio.passiosdk.passiofood.file.PassioFoodFileManager
import android.content.Context
import android.net.Uri
import org.json.JSONException
import org.json.JSONObject
import java.io.File

private const val REMOTE_CONFIG_NAME = "android_remote_config.txt"

internal class PassioVersionManager(
    private val passioFileManager: PassioFoodFileManager,
    private val configurationListener: ConfigurationListener
) {

    fun configureSDKWithExternalFiles(
        context: Context,
        files: List<Uri>,
        remoteOnly: Boolean,
    ): PassioStatus {
        if (remoteOnly) {
            PassioLog.i(this::class.java.simpleName, "Remote only configuration")
            return configurationListener.initializeRemoteOnly(context)
        }
        // Clear temp and deprecated files
        passioFileManager.clearTempFiles(context)
        passioFileManager.deleteDeprecatedFiles(context)
        // Migrate from legacy structure
        passioFileManager.migrateFromLegacyFolderStructure(context)
        // Clear optional files
//        val searchFile = passioFileManager.getStoredFile(context, passio_search_vectorize)
//        if (!licenseConfiguration.projects.nutrition!!.advancedSearch && searchFile != null) {
//            searchFile.delete()
//            PassioLog.w(this::class.java.simpleName, "Removed search files")
//        }

        var installedVersion = passioFileManager.getInstalledVersion(context)
        if (installedVersion == PassioFileManager.FILES_NOT_COMPLETE) {
            passioFileManager.deleteInstalledFiles(context)
            installedVersion = 0
            PassioLog.w(this::class.java.simpleName, "Removed incomplete install folder")
        }

        var tempVersion = passioFileManager.getTempVersion(context)
        if (tempVersion == PassioFileManager.FILES_NOT_COMPLETE) {
            passioFileManager.deleteTempFiles(context)
            tempVersion = 0
            PassioLog.w(this::class.java.simpleName, "Removed incomplete temp folder")
        }
        val shippedVersion = passioFileManager.getShippedVersion(context)
        val deliveredVersion = passioFileManager.getDeliveredFilesVersion(files)

        PassioLog.i(
            this::class.java.simpleName,
            "Expected: ${PassioSDKImpl.FILE_VERSION}, Delivered: $deliveredVersion, Temp: $tempVersion, Shipped: $shippedVersion, Installed: $installedVersion"
        )

        // Continue working with the installed files
        if (installedVersion != 0 &&
            installedVersion >= shippedVersion &&
            installedVersion >= deliveredVersion &&
            installedVersion >= tempVersion
        ) {
            PassioLog.i(this::class.java.simpleName, "Init with installed files")
            return configurationListener.initializeSDKWithExternalModels(
                context,
                installedVersion
            )
        }

        // Install the shipped version
        if (shippedVersion > installedVersion &&
            shippedVersion >= deliveredVersion &&
            shippedVersion >= tempVersion
        ) {
            PassioLog.i(this::class.java.simpleName, "Init with shipped files")
            return configurationListener.initializeSDKWithAssetModels(
                context,
                shippedVersion
            )
        }

        // Delivered files should be installed
        if (deliveredVersion > installedVersion &&
            deliveredVersion > shippedVersion &&
            deliveredVersion > tempVersion
        ) {
            PassioLog.i(this::class.java.simpleName, "Init with delivered files")
            // Copy delivered files to the temp folder
            val copiedUris = mutableListOf<Uri>()
            files.forEach { uri ->
                val file = File(uri.path!!)
                val copiedUri = passioFileManager.copyToTempWithVersionUpgrade(context, file)
                if (copiedUri != null) {
                    copiedUris.add(copiedUri)
                }
            }
            // If all files weren't copied, abort the whole transaction
            if (copiedUris.size != files.size) {
                copiedUris.forEach {
                    File(it.path!!).delete()
                }
            }

            return configurationListener.initializeSDKWithExternalModels(
                context,
                deliveredVersion
            )
        }

        return PassioStatus.error(
            PassioSDKError.NO_MODELS_FILES_FOUND,
            "No models found while configuring the Passio SDK"
        )
    }

    fun configureSDKWithRemoteFiles(context: Context, remoteOnly: Boolean): PassioStatus {
        if (remoteOnly) {
            PassioLog.i(this::class.java.simpleName, "Remote only configuration")
            return configurationListener.initializeRemoteOnly(context)
        }

        // Clear temp and deprecated files
        passioFileManager.clearTempFiles(context)
        passioFileManager.deleteDeprecatedFiles(context)
        // Migrate from legacy structure
        passioFileManager.migrateFromLegacyFolderStructure(context)
        // Clear optional files
//        val searchFile = passioFileManager.getStoredFile(context, passio_search_vectorize)
//        if (!licenseConfiguration.projects.nutrition!!.advancedSearch && searchFile != null) {
//            searchFile.delete()
//            PassioLog.w(this::class.java.simpleName, "Removed search files")
//        }

        // Get files and delete incomplete folders
        val fileTypes = passioFileManager.getAvailableFileTypes()
        val shippedVersion = passioFileManager.getShippedVersion(context)

        var installedVersion = passioFileManager.getInstalledVersion(context)
        if (installedVersion == PassioFileManager.FILES_NOT_COMPLETE) {
            passioFileManager.deleteInstalledFiles(context)
            installedVersion = 0
            PassioLog.w(this::class.java.simpleName, "Removed incomplete install folder")
        }

        var downloadedVersion =
            passioFileManager.getDownloadedVersion(context)
        if (downloadedVersion == PassioFileManager.FILES_NOT_COMPLETE) {
            passioFileManager.deleteDownloadedFiles(context)
            downloadedVersion = 0
            PassioLog.w(this::class.java.simpleName, "Removed incomplete download folder")
        }

        var tempVersion = passioFileManager.getTempVersion(context)
        if (tempVersion == PassioFileManager.FILES_NOT_COMPLETE) {
            passioFileManager.deleteTempFiles(context)
            tempVersion = 0
            PassioLog.w(this::class.java.simpleName, "Removed incomplete temp folder")
        }

        PassioLog.i(
            this::class.java.simpleName,
            "Expected: ${PassioSDKImpl.FILE_VERSION}, Downloaded: $downloadedVersion, Temp: $tempVersion, Shipped: $shippedVersion, Installed: $installedVersion"
        )

        if (installedVersion > shippedVersion &&
            installedVersion > tempVersion &&
            installedVersion >= downloadedVersion
        ) {
            checkLatestVersion(context, installedVersion, fileTypes)

            return configurationListener.initializeSDKWithExternalModels(
                context,
                installedVersion
            )
        }

        if (shippedVersion >= installedVersion &&
            shippedVersion > tempVersion &&
            shippedVersion >= downloadedVersion &&
            shippedVersion != 0
        ) {
            checkLatestVersion(context, shippedVersion, fileTypes)

            return configurationListener.initializeSDKWithAssetModels(
                context,
                shippedVersion
            )
        }

        if (tempVersion > installedVersion &&
            tempVersion > shippedVersion &&
            tempVersion > downloadedVersion
        ) {
            checkLatestVersion(context, tempVersion, fileTypes)

            return configurationListener.initializeSDKWithExternalModels(
                context,
                tempVersion
            )
        }

        // We have all the files we need already downloaded
        if (downloadedVersion > installedVersion) {
            passioFileManager.transferFromDownloadToTemp(
                context, listOf(passio_dataset)
            )

            return configurationListener.initializeSDKWithExternalModels(
                context,
                downloadedVersion
            )
        }
        // Some of the files are missing
        else {
            checkLatestVersion(context, 0, fileTypes)
            return PassioStatus.downloading()
        }
    }

    private fun getLatestVersionAsync(callback: (version: Int?) -> Unit) {
        val url = NativeUtils.instance.getModelURL() + REMOTE_CONFIG_NAME
        val task = NetworkStringTask(url, emptyMap())
        NetworkService.instance.doRequest(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.w(
                    this@PassioVersionManager::class.java.simpleName,
                    "Remote config fetch failed: $message"
                )
                callback(null)
            }

            override fun onTokenExpired() { /*NO-OP*/
            }

            override fun onSuccess(result: String) {
                try {
                    val archJson = JSONObject(result).getJSONObject("architecture")
                    val rcVersion = archJson.getInt(PassioSDKImpl.ARCHITECTURE)
                    PassioLog.i(
                        this@PassioVersionManager::class.java.simpleName,
                        "Remote config version: $rcVersion"
                    )
                    callback(rcVersion)
                } catch (e: JSONException) {
                    PassioLog.w(
                        this@PassioVersionManager::class.java.simpleName,
                        "Remote config parse error: ${e.message}"
                    )
                    callback(null)
                }
            }

        })
    }

    private fun checkLatestVersion(
        context: Context,
        localVersion: Int,
        fileTypes: List<SDKFileType>
    ) {
        getLatestVersionAsync { rcVersion ->
            val version = rcVersion ?: PassioSDKImpl.FILE_VERSION
            if (version <= localVersion) {
                return@getLatestVersionAsync
            }
            val files = fileTypes.map {
                "${it.name}.$version.passiosecure2"
            }
            configurationListener.downloadFiles(context, version, files)
//            val files = fileTypes.map {
//                "${it.name}.20240108.passiosecure2"
//            }
//            configurationListener.downloadFiles(context, 20240108, files)
        }
    }
}