package ai.passio.passiosdk.core.download

import ai.passio.passiosdk.core.event.PassioEventBus
import ai.passio.passiosdk.core.file.PassioFileManager
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.PassioLog
import android.content.Context
import android.net.Uri
import android.util.Log
import java.io.File

internal class PassioDownloadManager(
    private val passioFileManager: PassioFileManager,
    private val passioDownloadListener: PassioDownloadListener
) {

    internal interface PassioDownloadListener {
        fun onFileDownloaded(fileUri: Uri, filesDownloaded: Int, filesToDownload: Int)
        fun onDownloadError(errorMessage: String)
        fun onDownloadComplete()
    }

    private var filesToDownload: Int = 0
    private var filesDownloaded: Int = 0

    fun onStart() {
        PassioEventBus.getInstance().registerListener(eventListener)
    }

    fun onStop() {
        PassioEventBus.getInstance().unRegisterListener(eventListener)
    }

    fun downloadFiles(
        context: Context,
        version: Int,
        filesToDownload: ArrayList<String>
    ) {

        PassioDownloadService.downloadMissingFiles(
            context,
            filesToDownload,
            "${NativeUtils.instance.getModelURL()}$version/",
            passioFileManager.getPathToDownloadFolder(context)
        )
    }

    private val eventListener = object : PassioEventBus.PassioEventListener {
        override fun onEvent(tag: String, message: String) {
            when (tag) {
                PassioDownloadService.PASSIO_BROADCAST_EXTRA_FILES_TO_DOWNLOAD -> {
                    filesToDownload = message.toInt()
                    Log.d(
                        PassioDownloadManager::class.java.simpleName,
                        "Started download, files to download: $filesToDownload"
                    )
                }
                PassioDownloadService.PASSIO_BROADCAST_EXTRA_SUCCESS -> {
                    val fileUri = Uri.fromFile(File(message))
                    filesDownloaded++
                    passioDownloadListener.onFileDownloaded(
                        fileUri,
                        filesDownloaded,
                        filesToDownload
                    )
                    if (filesDownloaded == filesToDownload) {
                        passioDownloadListener.onDownloadComplete()
                    }
                }
                PassioDownloadService.PASSIO_BROADCAST_EXTRA_ERROR -> {
                    filesDownloaded++
                    Log.d(
                        PassioDownloadManager::class.java.simpleName,
                        "Download error($message): $filesDownloaded/$filesToDownload"
                    )
                    passioDownloadListener.onDownloadError(message)
                    if (filesDownloaded == filesToDownload) {
                        Log.d(
                            PassioDownloadManager::class.java.simpleName,
                            "Download complete"
                        )
                        passioDownloadListener.onDownloadComplete()
                    }
                }
            }
        }
    }
}