package ai.passio.passiosdk.core.search.fuzzywuzzy

import java.util.concurrent.Callable
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import kotlin.math.min

private const val SCORE_CUTOFF = 60
private const val SCORE_EPSILON = 10

internal object SearchUtils {

    private val executor: ExecutorService by lazy {
        val nThreads = kotlin.run {
            var availableCores = Runtime.getRuntime().availableProcessors()
            if (availableCores > 1) {
                availableCores /= 2
            }
            availableCores
        }
        Executors.newFixedThreadPool(nThreads)
    }

    val ALPHA_NUM = Regex("[^A-Za-z0-9 ]")

    private fun String.removeSpecial(): String {
        val specialChars = listOf(',', '*', '\'', '-')
        return this.filterNot { it in specialChars }
    }

    private fun getPlural(term: String): String {
        if (term.endsWith("s")) {
            return term
        }
        if (term.endsWith("y")) {
            var pluralTerm = term
            pluralTerm = pluralTerm.substring(0, pluralTerm.length - 1)
            return "${pluralTerm}ies"
        }
        return "${term}s"
    }

    fun searchFor(
        term: String,
        list: List<Pair<String, String>>
    ): List<Pair<String, String>> {
        val searchTerm = term.toLowerCase()
        val searchPlural = getPlural(term)
        val searchNoSpecial = searchTerm.removeSpecial()
        val searchPluralNoSpecial = searchPlural.removeSpecial()
        val exactMatches = mutableListOf<Pair<String, String>>()
        val startingWith = mutableListOf<Pair<String, String>>()
        val newWordAtTheEnd = mutableListOf<Pair<String, String>>()
        val containingTerm = mutableListOf<Pair<String, String>>()

        list.forEach {
            val name = it.second.toLowerCase()
            val nameNoComma = name.removeSpecial()
            if (name == searchTerm ||
                name == searchPlural ||
                nameNoComma == searchNoSpecial ||
                nameNoComma == searchPluralNoSpecial
            ) {
                exactMatches.add(it.first to name)
            } else if (name.startsWith(searchTerm) ||
                name.startsWith(searchPlural) ||
                nameNoComma.startsWith(searchNoSpecial) ||
                nameNoComma.startsWith(searchPluralNoSpecial)
            ) {
                startingWith.add(it.first to name)
            } else if (name.endsWith(" $searchTerm") ||
                name.endsWith(" $searchPlural") ||
                nameNoComma.endsWith(" $searchNoSpecial") ||
                nameNoComma.endsWith(" $searchPluralNoSpecial")
            ) {
                newWordAtTheEnd.add(it.first to name)
            } else if ((name.contains(searchTerm) ||
                        name.contentEquals(searchPlural))
            ) {
                containingTerm.add(it.first to name)
            }
        }

        exactMatches.sortBy { it.second.length }
        var startingOrNewWordWith = startingWith + newWordAtTheEnd
        startingOrNewWordWith = startingOrNewWordWith.sortedBy { it.second.length }
        containingTerm.sortBy { it.second.length }
        return exactMatches + startingOrNewWordWith + containingTerm
    }

    private fun fuzzySearch(
        query: String,
        list: List<String>
    ): List<ExtractedResult> {
        val result = FuzzySearch.extractAll(query, list, SCORE_CUTOFF)
        return result
    }

    fun searchForFuzzy(
        query: String,
        list: List<Pair<String, String>>,
    ): List<Pair<String, String>> {

        val map = mutableMapOf<String, String>()
        val input = mutableListOf<String>()
        list.forEach { pair ->
            map[pair.second] = pair.first
            input.add(pair.second)
        }

        val startTime = System.currentTimeMillis()

        val delim = input.size / 4
        val list1 = input.subList(0, delim)
        val list2 = input.subList(delim, delim * 2)
        val list3 = input.subList(delim * 2, delim * 3)
        val list4 = input.subList(delim * 3, input.size)

        val sanitizedQuery = ALPHA_NUM.replace(query, "")

        val resultFutures = mutableListOf<Callable<List<ExtractedResult>>>()
        for (sublist in arrayOf(list1, list2, list3, list4)) {
            resultFutures.add(Callable {
                return@Callable fuzzySearch(sanitizedQuery, sublist)
            })
        }

        val executedTasks = executor.invokeAll(resultFutures)
        val result = executedTasks.map { it.get() }.reduce { acc, redList -> acc + redList }
        if (result.isEmpty()) {
            return emptyList()
        }

        result.find { it.string == query }?.setMaxScore()

        val maxScore = result.maxOf { it.score }

        val sortedList = result.sortedWith(ExtractedResultComparator(query, maxScore))
        val subMax = min(sortedList.size, 100)
        val topResults = sortedList.map { it.string }.subList(0, subMax)

        return topResults.map { Pair(map[it]!!, it) }
    }

    fun distance(s1: String, s2: String): Int {
        var distance = 0
        run loop@{
            s1.forEachIndexed { index, c ->
                if (index == s2.length) {
                    return@loop
                }
                if (c == s2[index]) {
                    distance++
                } else {
                    return@loop
                }
            }
        }

        return distance
    }

    fun similarity(s1: String, s2: String): Double {
        var longer = s1
        var shorter = s2
        if (s1.length < s2.length) {
            longer = s2
            shorter = s1
        }
        val longerLength = longer.length
        return if (longerLength == 0) {
            1.0 /* both strings are zero length */
        } else (longerLength - editDistance(longer, shorter)) / longerLength.toDouble()
    }

    private fun editDistance(s1: String, s2: String): Int {
        val costs = IntArray(s2.length + 1)
        for (i in 0..s1.length) {
            var lastValue = i
            for (j in 0..s2.length) {
                if (i == 0) costs[j] = j else {
                    if (j > 0) {
                        var newValue = costs[j - 1]
                        if (s1[i - 1] != s2[j - 1]) {
                            newValue = newValue.coerceAtMost(lastValue).coerceAtMost(costs[j]) + 1
                        }
                        costs[j - 1] = lastValue
                        lastValue = newValue
                    }
                }
            }
            if (i > 0) costs[s2.length] = lastValue
        }
        return costs[s2.length]
    }

    class ExtractedResultComparator(
        private val query: String,
        private val maxScore: Int,
    ) : Comparator<ExtractedResult> {

        override fun compare(r0: ExtractedResult, r1: ExtractedResult): Int {
            if (r0.score >= maxScore - SCORE_EPSILON && r1.score >= maxScore - SCORE_EPSILON) {
                val similarity0 = similarity(r0.string, query)
                val similarity1 = similarity(r1.string, query)

                if (similarity0 > similarity1) {
                    return -1
                } else if (similarity1 > similarity0) {
                    return 1
                }
            }

            if (r0.score > r1.score) {
                return -1
            } else if (r1.score > r0.score) {
                return 1
            }

            if (r0.string.length < r1.string.length) {
                return -1
            } else if (r1.string.length < r0.string.length) {
                return 1
            } else {
                return 0
            }
        }
    }
}