package ai.passio.passiosdk.passiofood.data.model

import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.Barcode
import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.data.measurement.*
import ai.passio.passiosdk.passiofood.data.model.internal.FoodItemDataResult
import ai.passio.passiosdk.passiofood.upc.ResponseIngredient
import ai.passio.passiosdk.passiofood.upc.ResponsePortions
import ai.passio.passiosdk.passiofood.upc.ResponseWeight
import org.json.JSONException
import java.text.DecimalFormat
import kotlin.reflect.KMutableProperty0

/**
 * Data class that contains the nutritional information for a single food item.
 */
internal data class PassioFoodItemData(
    /**
     * ID to query the nutritional database
     */
    var passioID: PassioID = "",
    /**
     * Name of the single food item.
     */
    var name: String = ""
) : Cloneable {

    /**
     * Currently selected weight unit.
     */
    var selectedUnit: String = ""

    /**
     * Currently selected weight quantity.
     */
    var selectedQuantity: Double = 0.0

    /**
     * Reference weight of the single food item.
     */
    var referenceWeight = UnitMass(Grams, 0.0)

    var fat: UnitMass? = null
    var satFat: UnitMass? = null
    var monounsaturatedFat: UnitMass? = null
    var polyunsaturatedFat: UnitMass? = null
    var proteins: UnitMass? = null
    var carbs: UnitMass? = null
    var calories: UnitEnergy? = null
    var cholesterol: UnitMass? = null
    var sodium: UnitMass? = null
    var fibers: UnitMass? = null
    var transFat: UnitMass? = null
    var sugars: UnitMass? = null
    var sugarsAdded: UnitMass? = null
    var alcohol: UnitMass? = null
    var iron: UnitMass? = null
    var vitaminC: UnitMass? = null
    var vitaminA: Double? = null
    var vitaminD: UnitMass? = null
    var vitaminB6: UnitMass? = null
    var vitaminB12: UnitMass? = null
    var vitaminB12Added: UnitMass? = null
    var vitaminE: UnitMass? = null
    var vitaminEAdded: UnitMass? = null
    var iodine: UnitMass? = null
    var calcium: UnitMass? = null
    var potassium: UnitMass? = null
    var magnesium: UnitMass? = null
    var phosphorus: UnitMass? = null
    var sugarAlcohol: UnitMass? = null

    private val gramName = Grams.unitName

    private val unitMap = mutableMapOf(
        ::fat to Grams,
        ::satFat to Grams,
        ::transFat to Grams,
        ::monounsaturatedFat to Grams,
        ::polyunsaturatedFat to Grams,
        ::cholesterol to Milligrams,
        ::sodium to Milligrams,
        ::carbs to Grams,
        ::fibers to Grams,
        ::sugars to Grams,
        ::sugarsAdded to Grams,
        ::proteins to Grams,
        ::vitaminD to Micrograms,
        ::calcium to Milligrams,
        ::iron to Milligrams,
        ::potassium to Milligrams,
        ::vitaminC to Milligrams,
        ::sugarAlcohol to Grams,
        ::vitaminB12Added to Micrograms,
        ::vitaminB12 to Micrograms,
        ::vitaminB6 to Milligrams,
        ::vitaminE to Milligrams,
        ::vitaminEAdded to Milligrams,
        ::magnesium to Milligrams,
        ::phosphorus to Milligrams,
        ::iodine to Micrograms,
        ::alcohol to Grams,
    )

    /**
     * List of predefined serving size from the nutritional database.
     */
    var servingSizes: List<PassioServingSize>

    /**
     * List of predefined serving units from the nutritional database.
     */
    var servingUnits: List<PassioServingUnit>

    /**
     * Type of the food item.
     */
    var entityType: PassioIDEntityType = PassioIDEntityType.item

    /**
     * List of [PassioAlternative]s that are one level below in the food hierarchy.
     */
    var children: List<PassioAlternative>? = null

    /**
     * List of [PassioAlternative]s that are on the same level the food hierarchy.
     */
    var siblings: List<PassioAlternative>? = null

    /**
     * List of [PassioAlternative]s that are one level above in the food hierarchy.
     */
    var parents: List<PassioAlternative>? = null

    /**
     * FDC code used to query to Food Data Central database.
     */
    var foodOrigins: List<PassioFoodOrigin>? = null

    /**
     * Corresponding barcode code of the food item.
     */
    var barcode: Barcode? = null

    /**
     * List of ingredients of the food item.
     */
    var ingredientsDescription: String? = null

    var tags: List<String>? = null

    private val format = DecimalFormat("#.##")

    init {
        servingUnits = listOf(PassioServingUnit(gramName, UnitMass(Grams, 1.0)))
        servingSizes = listOf(PassioServingSize(100.0, gramName))
        selectedUnit = servingSizes.first().unitName
        selectedQuantity = servingSizes.first().quantity
    }

    internal constructor(
        foodItemDataResult: FoodItemDataResult,
        children: List<PassioAlternative>?,
        siblings: List<PassioAlternative>?,
        parents: List<PassioAlternative>?,
        preselectedQuantity: Double? = null,
        preselectedUnit: String? = null
    ) : this(
        foodItemDataResult.id,
        foodItemDataResult.name,
    ) {
        this.entityType = PassioIDEntityType.item
        this.children = children
        this.siblings = siblings
        this.parents = parents

        this.servingUnits = foodItemDataResult.passioServingUnits ?: listOf(
            PassioServingUnit(
                gramName,
                UnitMass(Grams, 1.0)
            )
        )
        if (servingUnits.find { it.unitName == gramName } == null) {
            val temp = servingUnits.toMutableList()
            temp.add(PassioServingUnit(gramName, UnitMass(Grams, 1.0)))
            servingUnits = temp
        }

        val preselectedServingSize = if (preselectedQuantity != null && preselectedUnit != null) {
            PassioServingSize(preselectedQuantity, preselectedUnit)
        } else {
            null
        }
        val servingSizes = mutableSetOf<PassioServingSize>()
        if (preselectedServingSize != null) {
            servingSizes.add(preselectedServingSize)
        }
        if (foodItemDataResult.passioServingSizes != null) {
            servingSizes.addAll(foodItemDataResult.passioServingSizes)
        } else {
            servingSizes.add(PassioServingSize(1.0, servingUnits.first().unitName))
        }

        this.servingSizes = servingSizes.toList() + this.servingSizes

        selectedUnit = this.servingSizes.first().unitName
        selectedQuantity = this.servingSizes.first().quantity

        foodItemDataResult.nutritionValuesResult?.let {
            referenceWeight = UnitMass(Grams, it.referenceWeight.toDouble())
            it.fat.toUnitMassAndSet(::fat)
            it.protein.toUnitMassAndSet(::proteins)
            it.carbs.toUnitMassAndSet(::carbs)
            calories = it.cal.toKcal()
            it.cholesterol.toUnitMassAndSet(::cholesterol)
            it.sodium.toUnitMassAndSet(::sodium)
            it.fiber.toUnitMassAndSet(::fibers)
            it.iron.toUnitMassAndSet(::iron)
            it.vitaminC.toUnitMassAndSet(::vitaminC)
            vitaminA = it.vitaminA
            it.calcium.toUnitMassAndSet(::calcium)
            it.potassium.toUnitMassAndSet(::potassium)
            it.satFat.toUnitMassAndSet(::satFat)
            it.monoSatFat.toUnitMassAndSet(::monounsaturatedFat)
            it.polySatFat.toUnitMassAndSet(::polyunsaturatedFat)
            it.transFat.toUnitMassAndSet(::transFat)
            it.alcohol.toUnitMassAndSet(::alcohol)
            it.sugars.toUnitMassAndSet(::sugars)
            it.sugarAdded.toUnitMassAndSet(::sugarsAdded)
            it.vitaminD.toUnitMassAndSet(::vitaminD)
            it.sugarAlcohol.toUnitMassAndSet(::sugarAlcohol)
            it.vitaminB12Added.toUnitMassAndSet(::vitaminB12Added)
            it.vitaminB12.toUnitMassAndSet(::vitaminB12)
            it.vitaminB6.toUnitMassAndSet(::vitaminB6)
            it.vitaminE.toUnitMassAndSet(::vitaminE)
            it.vitaminEAdded.toUnitMassAndSet(::vitaminEAdded)
            it.magnesium.toUnitMassAndSet(::magnesium)
            it.phosphorus.toUnitMassAndSet(::phosphorus)
            it.iodine.toUnitMassAndSet(::iodine)
        }

        foodOrigins = foodItemDataResult.origins
        barcode = foodItemDataResult.productReference
    }

    @Throws(JSONException::class)
    internal constructor(
        responseIngredient: ResponseIngredient
    ) : this(
        responseIngredient.branded?.productCode ?: "",
        responseIngredient.name ?: "",
    ) {
        barcode = responseIngredient.branded?.productCode

        if (responseIngredient.origin != null) {
            val originsTemp = mutableListOf<PassioFoodOrigin>()
            responseIngredient.origin!!.forEach {
                originsTemp.add(PassioFoodOrigin(it.id, it.source, responseIngredient.licenseCopy))
            }
            foodOrigins = originsTemp
        }

        val servingSizesTemp = mutableListOf<PassioServingSize>()
        val servingUnitsTemp = mutableListOf<PassioServingUnit>()
        responseIngredient.portions?.forEach { portion ->
            val servingSizesAndUnit = getServingSizeAndUnit(portion) ?: return@forEach
            servingSizesTemp.addAll(servingSizesAndUnit.first)
            servingUnitsTemp.add(servingSizesAndUnit.second)
        }

        servingSizesTemp.add(PassioServingSize(100.0, gramName))
        servingUnitsTemp.add(PassioServingUnit(gramName, UnitMass(Grams, 1.0)))

        servingSizes = servingSizesTemp
        selectedQuantity = servingSizes.first().quantity
        selectedUnit = servingSizes.first().unitName
        servingUnits = servingUnitsTemp
        referenceWeight = UnitMass(Grams, 100.0)

        ingredientsDescription = responseIngredient.branded?.ingredients
        tags = responseIngredient.tags

        responseIngredient.nutrients?.forEach { upcNutrient ->
            when (upcNutrient.id) {
                1603211196544 -> upcNutrient.amount.toUnitMassAndSet(::proteins)
                1603211196545 -> upcNutrient.amount.toUnitMassAndSet(::fat)
                1603211196546 -> upcNutrient.amount.toUnitMassAndSet(::carbs)
                1603211196548 -> calories = upcNutrient.amount.toKcal()
                1603211196571 -> upcNutrient.amount.toUnitMassAndSet(::fibers)
                1603211196679 -> upcNutrient.amount.toUnitMassAndSet(::satFat)
                1603211196678 -> upcNutrient.amount.toUnitMassAndSet(::transFat)
                1603211196677 -> upcNutrient.amount.toUnitMassAndSet(::cholesterol)
                1603211196583 -> upcNutrient.amount.toUnitMassAndSet(::sodium)
                1603211196751 -> upcNutrient.amount.toUnitMassAndSet(::sugars)
                1603211196577 -> upcNutrient.amount.toUnitMassAndSet(::calcium)
                1603211196579 -> upcNutrient.amount.toUnitMassAndSet(::iron)
                1603211196582 -> upcNutrient.amount.toUnitMassAndSet(::potassium)
                1603211196594 -> vitaminA = upcNutrient.amount
                1603211196626 -> upcNutrient.amount.toUnitMassAndSet(::vitaminC)
                1603211196604 -> upcNutrient.amount.toUnitMassAndSet(::vitaminD)
                1603211196631 -> upcNutrient.amount.toUnitMassAndSet(::vitaminB6)
                1603211196634 -> upcNutrient.amount.toUnitMassAndSet(::vitaminB12)
                1603211196580 -> upcNutrient.amount.toUnitMassAndSet(::magnesium)
                1603211196581 -> upcNutrient.amount.toUnitMassAndSet(::phosphorus)
                else -> {
                    when (upcNutrient.nutrient?.shortName) {
                        "protein" -> upcNutrient.amount.toUnitMassAndSet(::proteins)
                        "fat" -> upcNutrient.amount.toUnitMassAndSet(::fat)
                        "carbs" -> upcNutrient.amount.toUnitMassAndSet(::carbs)
                        "calories" -> calories = upcNutrient.amount.toKcal()
                        "fibers" -> upcNutrient.amount.toUnitMassAndSet(::fibers)
                        "satFat" -> upcNutrient.amount.toUnitMassAndSet(::satFat)
                        "transFat" -> upcNutrient.amount.toUnitMassAndSet(::transFat)
                        "cholesterol" -> upcNutrient.amount.toUnitMassAndSet(::cholesterol)
                        "sodium" -> upcNutrient.amount.toUnitMassAndSet(::sodium)
                        "sugarTotal" -> upcNutrient.amount.toUnitMassAndSet(::sugars)
                        "alcohol" -> upcNutrient.amount.toUnitMassAndSet(::alcohol)
                        "monounsaturatedFat" -> upcNutrient.amount.toUnitMassAndSet(::monounsaturatedFat)
                        "polyunsaturatedFat" -> upcNutrient.amount.toUnitMassAndSet(::polyunsaturatedFat)
                        "calcium" -> upcNutrient.amount.toUnitMassAndSet(::calcium)
                        "iron" -> upcNutrient.amount.toUnitMassAndSet(::iron)
                        "potassium" -> upcNutrient.amount.toUnitMassAndSet(::potassium)
                        "vitaminA" -> vitaminA = upcNutrient.amount
                        "vitaminC" -> upcNutrient.amount.toUnitMassAndSet(::vitaminC)
                        "sugarAlcohol" -> upcNutrient.amount.toUnitMassAndSet(::sugarAlcohol)
                        "vitaminD" -> upcNutrient.amount.toUnitMassAndSet(::vitaminD)
                        "sugarsAdded" -> upcNutrient.amount.toUnitMassAndSet(::sugarsAdded)
                        "vitaminB6" -> upcNutrient.amount.toUnitMassAndSet(::vitaminB6)
                        "vitaminB12" -> upcNutrient.amount.toUnitMassAndSet(::vitaminB12)
                        "magnesium" -> upcNutrient.amount.toUnitMassAndSet(::magnesium)
                        "phosphorus" -> upcNutrient.amount.toUnitMassAndSet(::phosphorus)
                        "vitaminB12Added" -> upcNutrient.amount.toUnitMassAndSet(::vitaminB12Added)
                        "vitaminE" -> upcNutrient.amount.toUnitMassAndSet(::vitaminE)
                        "vitaminEAdded" -> upcNutrient.amount.toUnitMassAndSet(::vitaminEAdded)
                        "iodine" -> upcNutrient.amount.toUnitMassAndSet(::iodine)
                    }
                }
            }
        }
    }

    private fun ResponseWeight.toUnitMass(): UnitMass? {
        return when (this.unit) {
            "g", "G", "gr", "GR" -> UnitMass(Grams, this.value ?: 0.0)
            "ml", "ML" -> UnitMass(Milliliters, this.value ?: 0.0)
            else -> {
                PassioLog.e(
                    PassioFoodItemData::class.java.simpleName,
                    "Unknown unit of weight: ${this.unit}"
                )
                null
            }
        }
    }

    /**
     * Given the current [selectedUnit] and [selectedQuantity] calculates the [UnitMass] of the
     * food item.
     *
     * @return the weight of the food item in grams.
     */
    fun computedWeight(): UnitMass {
        val servingUnit = servingUnits.firstOrNull { it.unitName == selectedUnit }
        return if (servingUnit == null) {
            UnitMass(Grams, 0.0)
        } else {
            UnitMass(Grams, servingUnit.weight.gramsValue() * selectedQuantity)
        }
    }

    fun setServingSize(unit: String, quantity: Double): Boolean {
        val servingUnit = servingUnits.firstOrNull {
            it.unitName == unit
        } ?: return false

        selectedQuantity = quantity
        selectedUnit = unit
        return true
    }

    private fun scaleValueByAmount(
        field: KMutableProperty0<UnitMass?>
    ): UnitMass? {
        if (field.get() == null) {
            return null
        }

        return field.get()!! * (computedWeight().gramsValue() / referenceWeight.gramsValue())
    }

    private fun scaleValueByAmount(unitEnergy: UnitEnergy?): UnitEnergy? {
        if (unitEnergy == null) {
            return null
        }

        return unitEnergy * (computedWeight().gramsValue() / referenceWeight.gramsValue())
    }

    private fun scaleValueByAmount(value: Double?): Double? {
        if (value == null) return null

        val calculatedValue = value * (computedWeight().gramsValue() / referenceWeight.gramsValue())
        return format.format(calculatedValue).toDouble()
    }

    /**
     * The value of fat for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalFat(): UnitMass? = scaleValueByAmount(::fat)

    /**
     * The value of calories for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalCalories(): UnitEnergy? = scaleValueByAmount(calories)

    /**
     * The value of protein for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalProtein(): UnitMass? = scaleValueByAmount(::proteins)

    /**
     * The value of carbohydrates for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalCarbs(): UnitMass? = scaleValueByAmount(::carbs)

    /**
     * The value of saturated fat for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalSatFat(): UnitMass? = scaleValueByAmount(::satFat)

    /**
     * The value of monounsaturated fat for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalMonounsaturatedFat(): UnitMass? = scaleValueByAmount(::monounsaturatedFat)

    /**
     * The value of polyunsaturated fat for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalPolyunsaturatedFat(): UnitMass? = scaleValueByAmount(::polyunsaturatedFat)

    /**
     * The value of cholesterol for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalCholesterol(): UnitMass? = scaleValueByAmount(::cholesterol)

    /**
     * The value of sodium for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalSodium(): UnitMass? = scaleValueByAmount(::sodium)

    /**
     * The value of fibers for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalFibers(): UnitMass? = scaleValueByAmount(::fibers)

    /**
     * The value of trans fat for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalTransFat(): UnitMass? = scaleValueByAmount(::transFat)

    /**
     * The value of sugars for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalSugars(): UnitMass? = scaleValueByAmount(::sugars)

    /**
     * The value of sugars added for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalSugarsAdded(): UnitMass? = scaleValueByAmount(::sugarsAdded)

    /**
     * The value of cholesterol for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalAlcohol(): UnitMass? = scaleValueByAmount(::alcohol)

    /**
     * The value of iron for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalIron(): UnitMass? = scaleValueByAmount(::iron)

    /**
     * The value of vitamin C for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminC(): UnitMass? = scaleValueByAmount(::vitaminC)

    /**
     * The value of vitamin A for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminA(): Double? = scaleValueByAmount(vitaminA)

    /**
     * The value of vitamin D for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminD(): UnitMass? = scaleValueByAmount(::vitaminD)

    /**
     * The value of vitamin B6 for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminB6(): UnitMass? = scaleValueByAmount(::vitaminB6)

    /**
     * The value of vitamin B12 for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminB12(): UnitMass? = scaleValueByAmount(::vitaminB12)

    /**
     * The value of calcium for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalCalcium(): UnitMass? = scaleValueByAmount(::calcium)

    /**
     * The value of potassium for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalPotassium(): UnitMass? = scaleValueByAmount(::potassium)

    /**
     * The value of magnesium for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalMagnesium(): UnitMass? = scaleValueByAmount(::magnesium)

    /**
     * The value of phosphorus for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalPhosphorus(): UnitMass? = scaleValueByAmount(::phosphorus)

    /**
     * The value of sugar alcohol for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalSugarAlcohol(): UnitMass? = scaleValueByAmount(::sugarAlcohol)

    /**
     * The value of vitaminB12 added for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminB12Added(): UnitMass? = scaleValueByAmount(::vitaminB12Added)

    /**
     * The value of vitaminE for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminE(): UnitMass? = scaleValueByAmount(::vitaminE)

    /**
     * The value of vitaminE added for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalVitaminEAdded(): UnitMass? = scaleValueByAmount(::vitaminEAdded)

    /**
     * The value of iodine for the food item given the current [selectedUnit] and
     * [selectedQuantity].
     */
    fun totalIodine(): UnitMass? = scaleValueByAmount(::iodine)


    /**
     * Change the current [selectedUnit].
     *
     * @return false if the [amountLabel] can't be found in the list of [servingUnits].
     */
    fun setSelectedAmountLabel(amountLabel: String): Boolean {
        if (servingUnits.firstOrNull { it.unitName == amountLabel } == null) {
            return false
        }

        selectedUnit = amountLabel
        return true
    }

    /**
     * Change the current [selectedQuantity].
     */
    fun setSelectedAmountNumber(amountNumber: Double) {
        selectedQuantity = amountNumber
    }

    fun isOpenFood(): Boolean {
        return foodOrigins?.find { it.source == "openfood" } != null
    }

    fun getOpenFoodLicense(): String? {
        return foodOrigins?.firstOrNull { it.source == "openfood" }?.licenseCopy
    }

    private fun Double?.toUnitMassAndSet(field: KMutableProperty0<UnitMass?>) {
        if (this == null) {
            field.set(null)
            return
        }

        field.set(UnitMass(unitMap[field]!!, this))
    }

    private fun getServingSizeAndUnit(
        portion: ResponsePortions
    ): Pair<List<PassioServingSize>, PassioServingUnit>? {
        val servings = mutableListOf<PassioServingSize>()
        val unit: PassioServingUnit

        if (portion.name == null) {
            return null
        }

        if (portion.suggestedQuantity != null) {
            val suggestedServings = portion.suggestedQuantity!!.map { suggestedQuantity ->
                PassioServingSize(suggestedQuantity, portion.name!!)
            }
            servings.addAll(suggestedServings)
        } else if (portion.quantity != null) {
            servings.add(PassioServingSize(portion.quantity!!, portion.name!!))
        } else {
            return null
        }

        if (portion.weight == null) {
            return null
        }

        val unitMass = portion.weight!!.toUnitMass() ?: return null
        unit = PassioServingUnit(portion.name!!, unitMass)

        return Pair(servings, unit)
    }

    internal fun deepCopy(): PassioFoodItemData {
        val data = PassioFoodItemData(passioID, name)
        data.selectedUnit = selectedUnit
        data.selectedQuantity = selectedQuantity
        data.referenceWeight = referenceWeight
        data.fat = fat
        data.satFat = satFat
        data.monounsaturatedFat = monounsaturatedFat
        data.polyunsaturatedFat = monounsaturatedFat
        data.proteins = proteins
        data.carbs = carbs
        data.calories = calories
        data.cholesterol = cholesterol
        data.sodium = sodium
        data.fibers = fibers
        data.transFat = transFat
        data.sugars = sugars
        data.sugarsAdded = sugarsAdded
        data.alcohol = alcohol
        data.iron = iron
        data.vitaminC = vitaminC
        data.vitaminA = vitaminA
        data.vitaminD = vitaminD
        data.vitaminB6 = vitaminB6
        data.vitaminB12 = vitaminB12
        data.vitaminB12Added = vitaminB12Added
        data.vitaminE = vitaminE
        data.vitaminEAdded = vitaminEAdded
        data.iodine = iodine
        data.calcium = calcium
        data.potassium = potassium
        data.magnesium = magnesium
        data.phosphorus = phosphorus
        data.sugarAlcohol = sugarAlcohol
        data.servingSizes = servingSizes
        data.servingUnits = servingUnits
        data.entityType = entityType
        data.children = children
        data.siblings = siblings
        data.parents = parents
        data.foodOrigins = foodOrigins
        data.barcode = barcode
        data.ingredientsDescription = ingredientsDescription
        data.tags = tags
        return data
    }
}

private fun Double?.toKcal(): UnitEnergy? {
    if (this == null) return null

    return UnitEnergy(KiloCalories, this)
}